﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class StringFilter
    {
        private string _filter;
        private bool _ignoreCase = true;
        private List<string> _includes = new List<string>();
        private List<string> _excludes = new List<string>();

        /// <summary>
        /// フィルタリングの条件式です。
        /// "Apple !Green"のように記述します。
        /// この場合、Apple を含み Green を含まない文字列がマッチします。
        /// </summary>
        public string Filter
        {
            get
            {
                return _filter;
            }

            set
            {
                if (_filter == null || !_filter.Equals(value))
                {
                    _filter = value;
                    this.ParseFilter();
                }
            }
        }

        /// <summary>
        /// 大文字小文字を無視するかを設定・取得します。デフォルトは true です。
        /// </summary>
        public bool IgnoreCase
        {
            get
            {
                return _ignoreCase;
            }

            set
            {
                if (_ignoreCase != value)
                {
                    this.ParseFilter();
                }
            }
        }

        /// <summary>
        /// 文字列がフィルタリングの条件式にマッチするか判定します。
        /// </summary>
        /// <param name="value">判定対象の文字列です。</param>
        /// <returns>マッチしたら true を返します。</returns>
        public bool Execute(string value)
        {
            if (string.IsNullOrWhiteSpace(value))
            {
                return _includes.Count == 0 && _excludes.Count == 0;
            }

            var value2 = this.IgnoreCase ? value.ToLower() : value;

            foreach (var item in _includes)
            {
                if (!value2.Contains(item))
                {
                    return false;
                }
            }

            foreach (var item in _excludes)
            {
                if (value2.Contains(item))
                {
                    return false;
                }
            }

            return true;
        }

        private void ParseFilter()
        {
            _includes.Clear();
            _excludes.Clear();

            if (string.IsNullOrWhiteSpace(_filter))
            {
                return;
            }

            var filter2 = _ignoreCase ? _filter.ToLower() : _filter;

            var tokens = filter2.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            if (tokens.Length == 0)
            {
                return;
            }

            foreach (var token in tokens)
            {
                if (token[0] == '!')
                {
                    if (token.Length == 1)
                    {
                        continue;
                    }

                    _excludes.Add(token.Substring(1));
                }
                else
                {
                    _includes.Add(token);
                }
            }
        }
    }
}
