﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using NintendoWare.Spy;
using NintendoWare.Spy.Windows.Primitives;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Threading;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// Atk パフォーマンスステートを表示するコントロールです。
    /// </summary>
    [TemplatePart(Name = TemplatePartCanvas, Type = typeof(Canvas))]
    public sealed class AtkPerformanceStateView : Control
    {
        private class InternalItem
        {
            public IAtkPerformanceStateValue Data { get; set; }
            public AtkPerformanceStateViewItem View { get; set; }
        }

        private const string TemplatePartCanvas = "PART_PlotCanvas";

        public static readonly DependencyProperty OriginXProperty = DependencyProperty.Register(
            nameof(OriginX),
            typeof(double),
            typeof(AtkPerformanceStateView),
            new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnOriginXChanged(e)));

        public static readonly DependencyProperty ScaleXProperty = DependencyProperty.Register(
                nameof(ScaleX),
                typeof(double),
                typeof(AtkPerformanceStateView),
                new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnScaleXChanged(e)));

        public static readonly DependencyProperty ItemsSourceProperty = DependencyProperty.Register(
            nameof(ItemsSource),
            typeof(IList),
            typeof(AtkPerformanceStateView),
            new FrameworkPropertyMetadata(null, (d, e) => Self(d).OnItemsSourceChanged(e)));

        private Canvas _canvas;
        private readonly object _observerOwner = new object();
        private readonly List<InternalItem> _internalItems = new List<InternalItem>();
        private readonly Stack<AtkPerformanceStateViewItem> _unusedViewsCache1 = new Stack<AtkPerformanceStateViewItem>();
        private readonly Stack<AtkPerformanceStateViewItem> _unusedViewsCache2 = new Stack<AtkPerformanceStateViewItem>();
        private double _lastScaleX;
        private int _lastBeginIndex;
        private int _lastEndIndex;
        private bool _needFullUpdateItemsVisibility;
        private readonly MergedRequestDispatcher _updateVisibilityDispatcher = new MergedRequestDispatcher();

        static AtkPerformanceStateView()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(AtkPerformanceStateView),
                new FrameworkPropertyMetadata(typeof(AtkPerformanceStateView)));
        }

        public AtkPerformanceStateView()
        {
        }

        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        private double ValidatedOriginX
        {
            get { return PlotCanvas.GetValidatedOriginX(this.OriginX); }
        }

        private double ValidatedScaleX
        {
            get { return PlotCanvas.GetValidatedScaleX(this.ScaleX); }
        }

        public IList ItemsSource
        {
            get { return (IList)this.GetValue(ItemsSourceProperty); }
            set { this.SetValue(ItemsSourceProperty, value); }
        }

        private void OnOriginXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateItemsVisibility();
        }

        private void OnScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateItemsVisibility();
        }

        private void OnItemsSourceChanged(DependencyPropertyChangedEventArgs e)
        {
            this.AttachItemsSource();
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _canvas = this.Template.FindName(TemplatePartCanvas, this) as Canvas;

            if (_canvas != null && !_internalItems.IsEmpty())
            {
                foreach (var item in _internalItems)
                {
                    if (item.View != null)
                    {
                        _canvas.Children.Add(item.View);
                    }
                }
            }
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.RequestUpdateItemsVisibility();
        }

        private void OnItemsSourceCollectionChanged(object sender, NotifyCollectionChangedEventArgs args)
        {
            if (args != null)
            {
                switch (args.Action)
                {
                    case NotifyCollectionChangedAction.Add:
                        AddInternalItems(args.NewItems);
                        UpdateItemsVisibility();
                        break;

                    default:
                        this.SyncInternalItems();
                        break;
                }
            }
        }

        private void AttachItemsSource()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            if (this.ItemsSource != null)
            {
                if (this.ItemsSource is INotifyCollectionChanged)
                {
                    CollectionChangedObservation.GetObserver(_observerOwner, this.ItemsSource)
                        .AddHandler(OnItemsSourceCollectionChanged);
                }
            }

            this.SyncInternalItems();
        }

        private void SyncInternalItems()
        {
            this.ClearUpdateItemsVisibilityInfos();
            this.ClearInternalItems();

            if (this.ItemsSource != null)
            {
                AddInternalItems(this.ItemsSource);
            }

            this.RequestUpdateItemsVisibility();
        }

        private void AddInternalItems(IEnumerable items)
        {
            items.OfType<IAtkPerformanceStateValue>().ForEach(it => AddInternalItem(it));
        }

        private void AddInternalItem(IAtkPerformanceStateValue item)
        {
            var internalItem = new InternalItem()
            {
                Data = item,
            };

            _internalItems.Add(internalItem);
        }

        private void ClearInternalItems()
        {
            foreach (var item in _internalItems)
            {
                if (item.View != null)
                {
                    StoreAtkPerformanceStateViewItemToCache1(item.View);
                }
            }

            _internalItems.Clear();
        }

        private void RequestUpdateItemsVisibility(bool fullUpdate = false)
        {
            _needFullUpdateItemsVisibility = _needFullUpdateItemsVisibility || fullUpdate;
            _updateVisibilityDispatcher.Request(UpdateItemsVisibility, DispatcherPriority.Render);
        }

        private void ClearUpdateItemsVisibilityInfos()
        {
            // UpdateItemsVisibility() 関連のフィールドを初期化します。
            // (すべてのアイテムが非表示の状態)
            // _internalItems の状態と一致している必要があります。

            _lastScaleX = 0;
            _lastBeginIndex = 0;
            _lastEndIndex = 0;
            _needFullUpdateItemsVisibility = false;
        }

        private void UpdateItemsVisibility()
        {
            if (_internalItems.IsEmpty())
            {
                FlushUnusedItemsCache1(flushAll: true);
                return;
            }

            var maxRightX = _internalItems.Last().Data.GetValidatedRightX();

            var scaleX = this.ValidatedScaleX;
            var minX = this.ValidatedOriginX;
            var maxX = this.ValidatedOriginX + this.ActualWidth / scaleX;

            // 表示範囲の最初の要素のインデックス
            int beginIndex;
            {
                beginIndex = BinarySearchUtility.BinarySearch(_internalItems, minX, it => it.Data.LeftX);
                if (beginIndex < 0)
                {
                    beginIndex = Math.Max(0, ~beginIndex - 1);
                }
            }

            // 表示範囲の最後の要素の次のインデックス
            int endIndex;
            {
                endIndex = BinarySearchUtility.BinarySearch(_internalItems, maxX, it => it.Data.GetValidatedRightX());
                if (endIndex < 0)
                {
                    endIndex = Math.Min(~endIndex, _internalItems.Count - 1);
                }
                endIndex += 1;
            }

            if (_needFullUpdateItemsVisibility)
            {
                _needFullUpdateItemsVisibility = false;

                if (beginIndex < endIndex)
                {
                    // 全アイテムのビジビリティを再設定します。

                    Enumerable.Range(0, beginIndex)
                        .ForEach(i => this.SetItemHidden(_internalItems[i]));

                    var lastRight = 0.0d;
                    Enumerable.Range(beginIndex, endIndex - beginIndex)
                        .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));

                    Enumerable.Range(endIndex, _internalItems.Count - endIndex)
                        .ForEach(i => this.SetItemHidden(_internalItems[i]));
                }
                else
                {
                    for (int i = 0; i < _internalItems.Count; ++i)
                    {
                        SetItemHidden(_internalItems[i]);
                    }
                }
            }
            else
            {
                if (beginIndex < endIndex)
                {
                    // 新しい表示区間にアイテムがある場合。

                    if (_lastBeginIndex < _lastEndIndex)
                    {
                        // 以前の表示区間にアイテムがあった場合。
                        // 以前の表示区間と新しい表示区間との差分のみを設定します。

                        var commonBeginIndex = Math.Max(beginIndex, _lastBeginIndex);
                        var commonEndIndex = Math.Min(endIndex, _lastEndIndex);

                        if (commonBeginIndex < commonEndIndex)
                        {
                            // 以前の表示区間と新しい表示区間に重なる領域がある場合。

                            if (_lastBeginIndex < commonBeginIndex)
                            {
                                Enumerable.Range(_lastBeginIndex, commonBeginIndex - _lastBeginIndex)
                                    .ForEach(i => this.SetItemHidden(_internalItems[i]));
                            }

                            if (commonEndIndex < _lastEndIndex)
                            {
                                Enumerable.Range(commonEndIndex, _lastEndIndex - commonEndIndex)
                                    .ForEach(i => this.SetItemHidden(_internalItems[i]));
                            }

                            if (scaleX != _lastScaleX)
                            {
                                // スケールが変わったときは新しい表示領域の全体を設定します。
                                var lastRight = 0.0d;
                                Enumerable.Range(beginIndex, endIndex - beginIndex)
                                    .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));
                            }
                            else
                            {
                                // スケールが変わっていないときは差分のみ設定します。
                                var lastRight = 0.0d;

                                if (beginIndex < commonBeginIndex)
                                {
                                    Enumerable.Range(beginIndex, endIndex - commonBeginIndex)
                                        .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));
                                }

                                if (commonEndIndex < endIndex)
                                {
                                    Enumerable.Range(beginIndex, endIndex - commonBeginIndex)
                                        .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));
                                }
                            }
                        }
                        else
                        {
                            // 以前の表示区間と新しい表示区間に重なる領域がない場合。

                            for (int i = _lastBeginIndex; i < _lastEndIndex; ++i)
                            {
                                SetItemHidden(_internalItems[i]);
                            }

                            var lastRight = 0.0d;
                            Enumerable.Range(beginIndex, endIndex - beginIndex)
                                .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));
                        }
                    }
                    else
                    {
                        // 以前の表示区間にアイテムがなかった場合。
                        var lastRight = 0.0d;
                        Enumerable.Range(beginIndex, endIndex - beginIndex)
                            .ForEach(i => this.SetItemVisibility(_internalItems[i], scaleX, ref lastRight));
                    }
                }
                else
                {
                    // 新しい表示区間にアイテムがない場合。
                    if (_lastBeginIndex < _lastEndIndex)
                    {
                        // 以前の表示区間にアイテムがあった場合。
                        // 以前の表示区間のアイテムを非表示にします。
                        for (int i = _lastBeginIndex; i < _lastEndIndex; ++i)
                        {
                            SetItemHidden(_internalItems[i]);
                        }
                    }
                }
            }

            FlushUnusedItemsCache1();

            _lastScaleX = scaleX;
            _lastBeginIndex = beginIndex;
            _lastEndIndex = endIndex;
        }

        private void SetItemVisibility(InternalItem item, double scaleX, ref double lastRight)
        {
            var right = Math.Ceiling(item.Data.GetValidatedRightX() * scaleX);

            // RightX が重なる場合は間引きます
            if (lastRight < right)
            {
                lastRight = right;

                if (item.View == null)
                {
                    item.View = GetNewAtkPerformanceStateViewItem(item.Data);
                }

                item.View.Visibility = Visibility.Visible;
            }
            else
            {
                if (item.View != null)
                {
                    item.View.Visibility = Visibility.Collapsed;
                }
            }
        }

        private void SetItemHidden(InternalItem item)
        {
            if (item.View != null)
            {
                StoreAtkPerformanceStateViewItemToCache1(item.View);
                item.View = null;
            }
        }

        private AtkPerformanceStateViewItem GetNewAtkPerformanceStateViewItem(IAtkPerformanceStateValue data)
        {
            // NOTE:
            // * _unusedViewsCache1 は UpdateItemsVisibility() のあいだだけアイテムを保持するキャッシュです。
            //   アイテムは _canvas.Children に登録されたままになっています。
            //   アイテムが再利用されるときは DataContext だけが差し替えられることになります。
            // * _unusedViewsCache2 は、より寿命の長いキャッシュです。
            //   アイテムは _canvas.Children から取り除かれます。
            //   再利用される際には View が再構築されます。

            AtkPerformanceStateViewItem item;

            if (!_unusedViewsCache1.IsEmpty())
            {
                item = _unusedViewsCache1.Pop();

                item.DataContext = data;

                return item;
            }

            if (_unusedViewsCache2.IsEmpty())
            {
                item = new AtkPerformanceStateViewItem();
            }
            else
            {
                item = _unusedViewsCache2.Pop();
            }

            // item.DataContext == null のときは親の DataContext が参照されてしまうため、
            // 親に登録する前に item.DataContext を設定する必要があります。
            item.DataContext = data;

            if (_canvas != null)
            {
                _canvas.Children.Add(item);
            }

            if (!_unusedViewsCache2.IsEmpty())
            {
                ShrinkUnusedItemsCache2();
            }

            return item;
        }

        private void StoreAtkPerformanceStateViewItemToCache1(AtkPerformanceStateViewItem item)
        {
            _unusedViewsCache1.Push(item);
        }

        private void StoreAtkPerformanceStateViewItemToCache2(AtkPerformanceStateViewItem item)
        {
            item.DataContext = null;
            _unusedViewsCache2.Push(item);
        }

        private void FlushUnusedItemsCache1(bool flushAll = false)
        {
            if (flushAll)
            {
                _canvas?.Children.Clear();

                while (!_unusedViewsCache1.IsEmpty())
                {
                    var item = _unusedViewsCache1.Pop();

                    item.DataContext = null;

                    _unusedViewsCache2.Push(item);
                }
            }
            else
            {
                while (!_unusedViewsCache1.IsEmpty())
                {
                    var item = _unusedViewsCache1.Pop();

                    _canvas?.Children.Remove(item);

                    item.DataContext = null;

                    _unusedViewsCache2.Push(item);
                }
            }
        }

        private void ShrinkUnusedItemsCache2()
        {
            if (_canvas != null)
            {
                var limit = Math.Max(100, _canvas.Children.Count);
                if (limit < _unusedViewsCache2.Count)
                {
                    _unusedViewsCache2.Pop();
                }
            }
        }

        private static AtkPerformanceStateView Self(DependencyObject d)
        {
            return (AtkPerformanceStateView)d;
        }
    }
}
