﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Communications;
using System;
using System.IO;
using System.Net.Sockets;

namespace NintendoWare.CafeSpyPlugin.Foundation.Communications
{
    /// <summary>
    /// Cafe HostIO のチャンネルを制御するクラスです。
    /// </summary>
    public class HostIOCafeChannel : IComChannel
    {
        private readonly int _port;
        private readonly string _portName;

        private TcpClient _tcpClient;
        private int _sendTimeout = 1000;
        private int _receiveTimeout = 1000;
        private int _keepAliveTimeout = 1000;

        private HostIOCafeStream _cachedStream;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        protected HostIOCafeChannel(int port)
        {
            Assertion.Argument.True(port > 0);
            _port = port;
            _portName = string.Empty;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        private HostIOCafeChannel(int port, string portName)
        {
            Assertion.Argument.True(port > 0);
            _port = port;
            _portName = portName;
        }

        /// <summary>
        /// デストラクタです。
        /// </summary>
        ~HostIOCafeChannel()
        {
            this.Close();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ポート番号を取得します。
        /// </summary>
        public int Port
        {
            get { return _port; }
        }

        /// <summary>
        /// ポート名を取得します。
        /// </summary>
        public string PortName
        {
            get { return _portName; }
        }

        /// <summary>
        /// 開閉状態の有無を取得します。
        /// </summary>
        public bool IsOpened
        {
            get
            {
                return _tcpClient == null ? false : _tcpClient.Connected;
            }
        }

        /// <summary>
        /// 送信タイムアウトを取得または設定します。
        /// </summary>
        public int SendTimeout
        {
            get { return _sendTimeout; }
            set
            {
                if (_sendTimeout == value)
                {
                    return;
                }

                _sendTimeout = value;

                if (_tcpClient != null)
                {
                    _tcpClient.SendTimeout = value;
                }
            }
        }

        /// <summary>
        /// 受信タイムアウトを取得または設定します。
        /// </summary>
        public int ReceiveTimeout
        {
            get { return _receiveTimeout; }
            set
            {
                if (_receiveTimeout == value)
                {
                    return;
                }

                _receiveTimeout = value;

                if (_tcpClient != null)
                {
                    _tcpClient.ReceiveTimeout = value;
                }
            }
        }

        /// <summary>
        /// キープアライブタイムアウトを取得または設定します。
        /// </summary>
        public int KeepAliveTimeout
        {
            get { return _keepAliveTimeout; }
            set
            {
                if (_keepAliveTimeout == value)
                {
                    return;
                }

                _keepAliveTimeout = value;
                this.ApplyKeepAliveTimeout();
            }
        }

        /// <summary>
        /// TCP クライアントを取得します。
        /// </summary>
        protected TcpClient TcpClient
        {
            get { return _tcpClient; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 新しいチャンネルを開きます。
        /// </summary>
        /// <param name="port">ポート番号を指定します。</param>
        /// <param name="portName">ポート名を指定します。</param>
        public static IComChannel Open(int port, string portName)
        {
            var newChannel = new HostIOCafeChannel(port, portName);

            try
            {
                newChannel.Open();
            }
            catch
            {
                newChannel.Dispose();
                throw;
            }

            return newChannel;
        }

        /// <summary>
        /// チャンネルを閉じます。
        /// </summary>
        public void Close()
        {
            this.OnClosing();

            if (_cachedStream != null)
            {
                _cachedStream.Close();
            }

            if (_tcpClient != null)
            {
                _tcpClient.Close();
            }

            this.OnClosed();
        }

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        public void Dispose()
        {
            this.Close();
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ストリームを取得します。
        /// </summary>
        public Stream GetStream()
        {
            return _tcpClient != null ? _cachedStream : null;
        }

        protected virtual void OnOpened() { }
        protected virtual void OnClosing() { }
        protected virtual void OnClosed() { }

        /// <summary>
        /// チャンネルを開きます。
        /// </summary>
        protected void Open()
        {
            // 接続先は "localhost" 固定。
            _tcpClient = new TcpClient("localhost", _port);
            Ensure.Operation.NotNull(_tcpClient);

            this.ApplyKeepAliveTimeout();

            // 送受信タイムアウトは TcpClient のデフォルト値を利用。
            this.SendTimeout = _tcpClient.SendTimeout;
            this.ReceiveTimeout = _tcpClient.ReceiveTimeout;

            _cachedStream = new HostIOCafeStream(_tcpClient.GetStream());
        }

        /// <summary>
        /// キープアライブタイムアウトを反映します。
        /// </summary>
        private void ApplyKeepAliveTimeout()
        {
            if (_tcpClient == null)
            {
                return;
            }

            byte[] buffer = new byte[12];
            BitConverter.GetBytes(_keepAliveTimeout == 0 ? 0 : 1).CopyTo(buffer, 0);    // スイッチ
            BitConverter.GetBytes(_keepAliveTimeout).CopyTo(buffer, 4);                 // msec

            _tcpClient.Client.IOControl(IOControlCode.KeepAliveValues, buffer, null);
        }
    }
}
