﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.AudioToolKit.Xml
{
    using Nintendo.Foundation.Contracts;
    using System;
    using System.IO;
    using System.Xml;
    using System.Xml.Serialization;

    /// <summary>
    /// ウェーブサウンドリソースデータ XML モデルのシリアライズをサポートします。
    /// </summary>
    public class WaveSoundResourceDataXmlSerializer
    {
        private static Lazy<WaveSoundResourceDataXmlSerializer> defaultInstance =
            new Lazy<WaveSoundResourceDataXmlSerializer>();

        private static Lazy<XmlSerializer> serializer = new Lazy<XmlSerializer>(
            () => new XmlSerializer(typeof(WaveSoundResourceDataXml))
            );

        private XmlSerializerNamespaces namespaces;
        private XmlWriterSettings writerSettings;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WaveSoundResourceDataXmlSerializer()
        {
            this.namespaces = CreateDefaultXmlNamespaces();
            this.writerSettings = CreateDefaultXmlWritterSettings();
        }

        /// <summary>
        /// WaveSoundDataXmlSerializer のデフォルトインスタンスを取得します。
        /// </summary>
        public static WaveSoundResourceDataXmlSerializer Default
        {
            get { return defaultInstance.Value; }
        }

        /// <summary>
        /// WaveSoundDataXml をシリアライズし、指定ストリームに出力します。
        /// </summary>
        /// <param name="output">出力先のストリーム。</param>
        /// <param name="model">シリアライズ対象の XML モデル。</param>
        public void Serialize(Stream output, WaveSoundResourceDataXml model)
        {
            Ensure.Operation.NotNull(this.writerSettings);

            using (var writer = XmlWriter.Create(output, this.writerSettings))
            {
                this.SerializeImpl(writer, model);
            }
        }

        /// <summary>
        /// WaveSoundDataXml をシリアライズし、指定 TextWriter に出力します。
        /// </summary>
        /// <param name="textWriter">出力先の TextWriter。</param>
        /// <param name="model">シリアライズ対象の XML モデル。</param>
        public void Serialize(TextWriter textWriter, WaveSoundResourceDataXml model)
        {
            Ensure.Operation.NotNull(this.writerSettings);

            using (var writer = XmlWriter.Create(textWriter, this.writerSettings))
            {
                this.SerializeImpl(writer, model);
            }
        }

        /// <summary>
        /// 指定ストリームを読み込み、WaveSoundDataXml をデシリアライズします。
        /// </summary>
        /// <param name="input">入力元のストリーム。</param>
        /// <returns>デシリアライズした WaveSoundDataXml。</returns>
        public WaveSoundResourceDataXml Deserialize(Stream input)
        {
            using (var reader = XmlReader.Create(input))
            {
                return this.DeserializeImpl(reader);
            }
        }

        /// <summary>
        /// 指定 TextReader を読み込み、WaveSoundDataXml をデシリアライズします。
        /// </summary>
        /// <param name="input">入力元の TextReader。</param>
        /// <returns>デシリアライズした WaveSoundDataXml。</returns>
        public WaveSoundResourceDataXml Deserialize(TextReader textReader)
        {
            using (var reader = XmlReader.Create(textReader))
            {
                return this.DeserializeImpl(reader);
            }
        }

        private static XmlSerializerNamespaces CreateDefaultXmlNamespaces()
        {
            var result = new XmlSerializerNamespaces();
            result.Add(String.Empty, String.Empty);

            return result;
        }

        private static XmlWriterSettings CreateDefaultXmlWritterSettings()
        {
            return new XmlWriterSettings()
            {
                Indent = true,
                IndentChars = "  ",
                NewLineChars = Environment.NewLine,
                NamespaceHandling = NamespaceHandling.OmitDuplicates,
                OmitXmlDeclaration = true,
            };
        }

        private void SerializeImpl(XmlWriter writer, WaveSoundResourceDataXml model)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(model);
            Ensure.Operation.NotNull(this.namespaces);

            serializer.Value.Serialize(writer, model, this.namespaces);
        }

        private WaveSoundResourceDataXml DeserializeImpl(XmlReader reader)
        {
            Assertion.Argument.NotNull(reader);
            return (WaveSoundResourceDataXml)serializer.Value.Deserialize(reader);
        }
    }
}
