﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using Nintendo.Foundation.Audio;
using Nintendo.Foundation.IO;
using Nintendo.Foundation.Utilities;
using Nintendo.Foundation.Windows.Primitives.Shapes;
using Nintendo.Foundation.Windows.Primitives.Geometries;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    using Nintendo.Foundation.Audio;

    public static class WaveImage
    {
        /// <summary>
        /// 画像イメージの作成
        /// </summary>
        /// <param name="path"></param>
        /// <param name="parentSize"></param>
        /// <returns></returns>
        public static Image CreateImage(string filePath, Size parentSize, double begin, double end)
        {
            if (filePath == null || File.Exists(filePath) == false)
            {
                return null;
            }

            //if (!aal.ToolFoundation.Util.Utility.IsWaveExtension(path))
            //    return null;

            var info = WaveFileReader.ReadWaveFileInfo(filePath);

            var dataSet = WaveImage.CreateImageSource(filePath, info, parentSize, begin, end);
            if (dataSet == null)
            {
                return null;
            }

            var imageWidth = end - begin + 1.0;
            var imageHeight = parentSize.Height;
            return new Image() { Source = CreateWritableBitmap(dataSet, info.WaveFormat.ChannelCount, imageWidth, imageHeight, begin) };
        }

        private static WriteableBitmap CreateWritableBitmap(PointDataSet dataSet, int channelCount, double imageWidth, double imageHeight, double begin)
        {
            WriteableBitmap wb = new WriteableBitmap((int)imageWidth, (int)imageHeight, 96, 96, PixelFormats.Pbgra32, null);

            try
            {
                Color waveColor = Color.FromRgb(0x60, 0x60, 0x60);
                Color zeroColor = Color.FromRgb(0xb0, 0xb0, 0xb0);

                wb.Lock();
                var b = new ShapeDrawingWriteableBitmapEx(wb);

                // 値0(中心)に線を描画します。
                var height = imageHeight / channelCount;
                var center = (height - 1) / 2.0;

                for (int channel = 0; channel < channelCount; channel++)
                {
                    var y = center + (channel * height);
                    b.DrawLine(new GeometryLineSegment(0, y, imageWidth, y), zeroColor);
                }

                // サンプルの線を描画します。
                if (dataSet.PointKind == PointDataSet.PointKinds.Peek)
                {
                    for (int channel = 0; channel < dataSet.PointDataArrays.Count(); channel++)
                    {
                        for (int i = 0; i < dataSet.PointDataArrays[channel].PointDatas.Length; i++)
                        {
                            var x = dataSet.PointDataArrays[channel].PointDatas[i].X - (int)begin;
                            var y1 = (int)(dataSet.PointDataArrays[channel].PointDatas[i].Y1 + 0.5);
                            var y2 = (int)(dataSet.PointDataArrays[channel].PointDatas[i].Y2 + 0.5);

                            if (y1 == y2)
                            {
                                b.DrawLine(new GeometryLineSegment(x - 1, y1, x, y2), waveColor);
                            }
                            else
                            {
                                b.DrawLine(new GeometryLineSegment(x, y1, x, y2), waveColor);
                            }
                        }
                    }
                }
                else if (dataSet.PointKind == PointDataSet.PointKinds.Point)
                {
                    foreach (var pointDataArray in dataSet.PointDataArrays)
                    {
                        if (pointDataArray.PointDatas.Count() > 0)
                        {
                            double lx = pointDataArray.PointDatas[0].X - (int)begin;
                            double ly = pointDataArray.PointDatas[0].Y1;

                            for (int i = 0; i < pointDataArray.PointDatas.Length; i++)
                            {
                                var x = pointDataArray.PointDatas[i].X - (int)begin;
                                var y = pointDataArray.PointDatas[i].Y1;
                                b.DrawLine(new GeometryLineSegment(x, y, lx, ly), waveColor);

                                if (dataSet.HorizontalRatio < 0.15)
                                {
                                    b.FillCircle(new GeometryRectangle(x - 2, y - 2, 5, 5), waveColor);
                                }

                                lx = x;
                                ly = y;
                            }
                        }
                    }
                }

                return wb;
            }
            finally
            {
                wb.Unlock();
            }
        }

        public static WaveFileInfo ReadWaveInfo(string filePath)
        {
            var info = new WaveFileInfo();

            if (filePath == null || System.IO.File.Exists(filePath) == false)
            {
                return info;
            }

            //if (!aal.ToolFoundation.Util.Utility.IsWaveExtension(path))
            //    return info;

            try
            {
                info = WaveFileReader.ReadWaveFileInfo(filePath);
            }
            catch
            {
            }

            return info;
        }

        /// <summary>
        /// 波形サンプルの構築
        /// </summary>
        /// <param name="path"></param>
        /// <param name="parentSize"></param>
        /// <returns></returns>
        private static PointDataSet CreateImageSource(string filePath, WaveFileInfo info, Size parentSize, double begin, double end)
        {
            // サイズ
            var channelCount = info.WaveFormat.ChannelCount;
            var width = (int)parentSize.Width;
            var height = parentSize.Height / channelCount;
            var center = (height - 1) / 2.0;

            if (begin > end || end <= 0 || width <= 0)
            {
                return null;
            }

            double heightRatio;
            if (info.WaveFormat.BitPerSample == 16)
            {
                heightRatio = center / (double)(short.MaxValue);
            }
            else if (info.WaveFormat.BitPerSample == 24)
            {
                heightRatio = center / (double)(128 * 256 * 256);
            }
            else if (info.WaveFormat.BitPerSample == 8)
            {
                heightRatio = center / (double)(sbyte.MaxValue);
            }
            else
            {
                heightRatio = center / (double)(short.MaxValue);
            }

            // サンプルを保存領域を確保します。
            PointDataSet dataSet = null;
            var perSample = (int)((double)info.SampleCount / (double)width);

            if (width <= info.SampleCount && perSample > 1)
            {
                var requiredSampleCount = (int)end - (int)begin;
                dataSet = new PointDataSet(channelCount, requiredSampleCount, PointDataSet.PointKinds.Peek, 0.0);

                // サンプルを集計します。
                var data = WaveFileReader.ReadWaveData(filePath, info);
                if (data == null)
                {
                    return null;
                }

                Parallel.For((int)begin, (int)end, x =>
                {
                    int index = x - (int)begin;
                    int sampleBegin = (int)(((double)data.SampleCount / (double)width) * (double)x) - 1;
                    int sampleEnd = (int)(((double)data.SampleCount / (double)width) * (double)(x + 1));

                    if (sampleBegin < 0)
                    {
                        sampleBegin = 0;
                    }

                    for (int channel = 0; channel < channelCount; channel++)
                    {
                        double maxPeek = data.Samples[channel][sampleBegin];
                        double minPeek = data.Samples[channel][sampleBegin];
                        for (int p = sampleBegin; p < sampleEnd; p++)
                        {
                            var peek = data.Samples[channel][p];
                            if (peek > maxPeek)
                            {
                                maxPeek = peek;
                            }
                            else if (peek < minPeek)
                            {
                                minPeek = peek;
                            }
                        }

                        double max = center + (channel * height) - (maxPeek * heightRatio);
                        double min = center + (channel * height) - (minPeek * heightRatio);

                        dataSet.PointDataArrays[channel].PointDatas[index] = new PointData() { X = x, Y1 = max, Y2 = min };
                    }
                });
            }
            else
            {
                var ratio = (double)width / (double)info.SampleCount;
                var b = (int)(begin / ratio);
                var e = (int)(end / ratio) + 2;

                if (e > info.SampleCount)
                {
                    e = info.SampleCount;
                }

                var requiredSampleCount = e - b;
                dataSet = new PointDataSet(channelCount, requiredSampleCount, PointDataSet.PointKinds.Point, (double)info.SampleCount / (double)width);

                var data = WaveFileReaderKai.ReadWaveData(filePath, info, (long)b, (long)e);

                for (int channel = 0; channel < channelCount; channel++)
                {
                    for (int x = b; x < e; x++)
                    {
                        int index = x - b;
                        var y = center + (channel * height) - (data.Samples[channel][index] * heightRatio);
                        dataSet.PointDataArrays[channel].PointDatas[index] = new PointData() { X = x * ratio, Y1 = y, Y2 = y };
                    }
                }
            }

            return dataSet;
        }

        /// <summary>
        ///
        /// </summary>
        private class PointData
        {
            public double X { get; set; }
            public double Y1 { get; set; }
            public double Y2 { get; set; }
        }

        private class PointDataArray
        {
            private PointData[] pointDatas = null;

            public PointDataArray(int count)
            {
                this.pointDatas = new PointData[count];
            }

            public PointData[] PointDatas
            {
                get
                {
                    return this.pointDatas;
                }
            }
        }

        private class PointDataSet
        {
            private PointDataArray[] pointDataArrays = null;
            private PointKinds pointKind;
            private double horizontalRatio = 0.0;

            public enum PointKinds
            {
                Point,
                Peek,
            }

            public PointDataSet(int maxChannelCount, int maxSampleCount, PointKinds pointKind, double horizontalRatio)
            {
                this.pointKind = pointKind;
                this.horizontalRatio = horizontalRatio;

                this.pointDataArrays = new PointDataArray[maxChannelCount];
                for (int channel = 0; channel < maxChannelCount; channel++)
                {
                    this.pointDataArrays[channel] = new PointDataArray(maxSampleCount);
                }
            }

            public PointDataArray[] PointDataArrays
            {
                get { return this.pointDataArrays; }
            }

            public PointKinds PointKind
            {
                get { return this.pointKind; }
            }

            public double HorizontalRatio
            {
                get { return this.horizontalRatio; }
            }
        }


        public static class WaveFileReaderKai
        {
            public static WaveSampleData ReadWaveData(string filepath, WaveFileInfo fileInfo, long begin, long end)
            {
                WaveSampleData result;
                using (FileStream fileStream = new FileStream(filepath, FileMode.Open, FileAccess.Read))
                {
                    BinaryReader reader = null;
                    if (fileInfo.FileFormat == WaveFileFormat.Wave)
                    {
                        reader = LittleEndianBinaryReader.Create(fileStream);
                    }
                    else
                    {
                        reader = BigEndianBinaryReader.Create(fileStream);
                    }
                    result = WaveFileReaderKai.ReadWaveData(reader, fileInfo, begin, end);
                }
                return result;
            }

            private static WaveSampleData ReadWaveData(BinaryReader reader, WaveFileInfo fileInfo, long begin, long end)
            {
                long oneSampleSize = 0;
                Func<int> func;

                switch (fileInfo.WaveFormat.BitPerSample)
                {
                    case 8:
                        func = (() => (int)reader.ReadSByte());
                        oneSampleSize = 1;
                        break;

                    case 16:
                        func = (() => (int)reader.ReadInt16());
                        oneSampleSize = 2;
                        break;

                    case 24:
                        func = (() => WaveFileReaderKai.ReadInt24(reader));
                        oneSampleSize = 3;
                        break;

                    default:
                        throw new NotSupportedException();
                }

                long skip = begin * oneSampleSize * fileInfo.WaveFormat.ChannelCount;
                reader.BaseStream.Position = fileInfo.SampleDataOffset + skip;

                long requiredSampleCount = end - begin;
                WaveSampleData waveSampleData = new WaveSampleData(fileInfo.WaveFormat.ChannelCount, (int)requiredSampleCount);

                for (int i = 0; i < requiredSampleCount; i++)
                {
                    for (int j = 0; j < fileInfo.WaveFormat.ChannelCount; j++)
                    {
                        waveSampleData.Samples[j][i] = func();
                    }
                }
                return waveSampleData;
            }

            internal static int ReadInt24(BinaryReader reader)
            {
                byte a = reader.ReadByte();
                byte b = reader.ReadByte();
                byte c = reader.ReadByte();
                byte d = (Byte)(((c & 128) != 0) ? 255 : 0);
                return BitConverter.ToInt32(new byte[] { a, b, c, d }, 0);
            }
        }
    }
}
