﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Xml;
using System.Globalization;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    using Nintendo.AudioToolkit.Extensions;

    /// <summary>
    /// タイムラインコントロールが使用するパネルクラスです。
    /// </summary>
    public class TimelinePanel : Canvas
    {
        /// <summary>
        ///
        /// </summary>
        public double ViewportX
        {
            get { return (double)GetValue(ViewportXProperty); }
            set { SetValue(ViewportXProperty, value); }
        }

        public static readonly DependencyProperty ViewportXProperty =
            DependencyProperty.Register("ViewportX", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(OnViewportXChanged)));

        private static void OnViewportXChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var panel = d as TimelinePanel;
            panel.InvalidateMeasure();
            panel.InvalidateVisual();
        }

        public double ViewportWidth
        {
            get { return (double)GetValue(ViewportWidthProperty); }
            set { SetValue(ViewportWidthProperty, value); }
        }

        public static readonly DependencyProperty ViewportWidthProperty =
            DependencyProperty.Register("ViewportWidth", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(OnViewportWidthChanged)));

        private static void OnViewportWidthChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var panel = d as TimelinePanel;
            panel.InvalidateMeasure();
            panel.InvalidateVisual();
        }

        public double MinimumViewportWidth
        {
            get { return (double)GetValue(MinimumViewportWidthProperty); }
            set { SetValue(MinimumViewportWidthProperty, value); }
        }

        public static readonly DependencyProperty MinimumViewportWidthProperty =
            DependencyProperty.Register("MinimumViewportWidth", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0));

        public double MinorTick
        {
            get { return (double)GetValue(MinorTickProperty); }
            set { SetValue(MinorTickProperty, value); }
        }

        public static readonly DependencyProperty MinorTickProperty =
            DependencyProperty.Register("MinorTick", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(1000.0));

        public double MinimumSpanMajorTick
        {
            get { return (double)GetValue(MinimumSpanMajorTickProperty); }
            set { SetValue(MinimumSpanMajorTickProperty, value); }
        }

        public static readonly DependencyProperty MinimumSpanMajorTickProperty =
            DependencyProperty.Register("MinimumSpanMajorTick", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(1.0, null));

        public double MajorTick
        {
            get { return (double)GetValue(MajorTickProperty); }
            set { SetValue(MajorTickProperty, value); }
        }

        public static readonly DependencyProperty MajorTickProperty =
            DependencyProperty.Register("MajorTick", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(1000.0));

        public Point MousePosition
        {
            get { return (Point)GetValue(MousePositionProperty); }
            set { SetValue(MousePositionProperty, value); }
        }

        public static readonly DependencyProperty MousePositionProperty =
            DependencyProperty.Register("MousePosition", typeof(Point), typeof(TimelinePanel),
            new FrameworkPropertyMetadata());

        /// <summary>
        /// 水平方向の拡大率です。
        /// </summary>
        private double HorizontalScale
        {
            get
            {
                return this.FindParent<TimelineControl>().HorizontalScale;
            }
        }

        protected override Size MeasureOverride(Size availableSize)
        {
            var size = base.MeasureOverride(availableSize);

            if (size.Width < this.MinimumViewportWidth)
            {
                size.Width = this.MinimumViewportWidth;
            }

            return size;
        }

        protected override void OnMouseLeave(MouseEventArgs e)
        {
            base.OnMouseLeave(e);

            this.MousePosition = new Point(double.NaN, double.NaN);
        }

        protected override void OnPreviewMouseMove(MouseEventArgs e)
        {
            base.OnPreviewMouseMove(e);

            this.MousePosition = e.GetPosition(this);
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            this.UpdateTick();
            //this.DrawTick(drawingContext, this.MinorTick, 1.0);
            this.DrawTick(drawingContext, this.MajorTick, 3.0);
        }

        private void DrawTick(DrawingContext drawingContext, double tick, double dashStyle)
        {
            var span = tick * this.HorizontalScale;
            var x = (double)((int)(this.ViewportX / span) * span);
            var xcount = (int)(this.ViewportWidth / span) + 2;

            var brush = new SolidColorBrush(Color.FromRgb(0xb0, 0xb0, 0xb0));
            brush.Freeze();
            var pen = new Pen(brush, 1);
            pen.DashStyle = new DashStyle();
            pen.DashStyle.Dashes.Add(dashStyle);
            pen.Freeze();

            for (int index = 0; index < xcount; index++)
            {
                var guideline = new GuidelineSet();
                guideline.GuidelinesX.Add(x - 0.5);
                guideline.GuidelinesX.Add(x + 0.5);

                drawingContext.PushGuidelineSet(guideline);
                drawingContext.DrawGeometry(null, pen, new LineGeometry(new Point(x, 0), new Point(x, this.ActualHeight)));
                drawingContext.Pop();

                x += span;
            }

            base.OnRender(drawingContext);
        }

        private void UpdateTick()
        {
            var value = TickAndDivideCalculator.GetTick(this.MinimumSpanMajorTick, this.HorizontalScale);
            var majorTick = value.Item1;
            var minorTick = value.Item2;

            if (majorTick != this.MajorTick)
            {
                this.MajorTick = majorTick;
            }

            if (minorTick != this.MinorTick)
            {
                this.MinorTick = minorTick;
            }
        }
    }
}
