﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Xml;
using System.Globalization;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    using Nintendo.AudioToolkit.Extensions;

    /// <summary>
    /// タイムラインコントロールを管理するリストボックスです。
    /// </summary>
    public class TimelineListBox : ListBox
    {
        private RubberBandAdorner adorner = null;
        private bool selecting = false;

        public TimelineListBox()
        {
            this.adorner = new RubberBandAdorner(this);
        }

        public override void OnApplyTemplate()
        {
            // HACK : ControlTemplate を書き換えるのも大変なので、無理やりパディング調整
            var border = this.Template.FindName("Bd", this) as Border;

            if (border != null)
            {
                border.Padding = new Thickness();
            }

            base.OnApplyTemplate();
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            base.PrepareContainerForItemOverride(element, item);


            var control = element as ListBoxItem;
            if (control != null)
            {
                control.AddHandler(Control.LoadedEvent, new RoutedEventHandler(OnLoadedTimelineListBoxItem));
                control.AddHandler(Control.UnloadedEvent, new RoutedEventHandler(OnUnloadedTimelineListBoxItem));

            }
        }

        protected override void ClearContainerForItemOverride(DependencyObject element, object item)
        {
            var control = element as TimelineControlItem;
            if (control != null)
            {
                control.RemoveHandler(Control.LoadedEvent, new RoutedEventHandler(OnLoadedTimelineListBoxItem));
                control.RemoveHandler(Control.UnloadedEvent, new RoutedEventHandler(OnUnloadedTimelineListBoxItem));
            }
        }

        private void OnLoadedTimelineListBoxItem(object sender, RoutedEventArgs e)
        {
            var item = sender as ListBoxItem;
            var control = item.FindChild<TimelineControl>();

            control.DropFile += OnDropFile;
            control.BeginDrag += OnBeginDrag;
        }

        private void OnUnloadedTimelineListBoxItem(object sender, RoutedEventArgs e)
        {
            var item = sender as ListBoxItem;
            var control = item.FindChild<TimelineControl>();

            control.DropFile -= OnDropFile;
            control.BeginDrag -= OnBeginDrag;
        }

        private void OnDropFile(object sender, TimelineControl.DropFileEventArgs e)
        {
            if (this.DropFileCommand != null &&
                this.DropFileCommand.CanExecute(e) == true)
            {
                this.DropFileCommand.Execute(e);
            }
        }

        private void OnBeginDrag(object sender, TimelineControl.BeginDragEventArgs e)
        {
            foreach (var obj in this.Items)
            {
                ListBoxItem item = this.ItemContainerGenerator.ContainerFromItem(obj) as ListBoxItem;
                item.FindChildren<TimelineControlItem>()
                    .Where(i => i.IsSelected == true)
                    .ToList()
                    .ForEach(i => e.DragItems.Add(i));
            }
        }

        public ICommand DropFileCommand
        {
            get { return (ICommand)base.GetValue(DropFileCommandProperty); }
            set { base.SetValue(DropFileCommandProperty, value); }
        }

        public static readonly DependencyProperty DropFileCommandProperty = DependencyProperty.Register
            ("DropFileCommand", typeof(ICommand), typeof(TimelineListBox),
            new FrameworkPropertyMetadata(null, null));

        protected override void OnPreviewMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            this.Focus();

            var result = VisualTreeHelper.HitTest(this, e.GetPosition(this));
            if (result.VisualHit is TimelinePanel)
            {
                this.Begin(e.GetPosition(this));
            }
            else
            {
                // 既に選択されているアイテムをクリックしたのか？
                var item = result.VisualHit.FindParent<TimelineControlItem>();
                if (item != null && item.IsSelected == true)
                {

                    if ((Keyboard.Modifiers & ModifierKeys.Control) == ModifierKeys.Control)
                    {
                        item.IsSelected = !item.IsSelected;
                    }
                }
                else
                {
                    if ((Keyboard.Modifiers & ModifierKeys.Control) != ModifierKeys.Control)
                    {
                        this.UnselectAllTimelineControlItem();
                    }

                    if (item != null)
                    {
                        item.IsSelected = true;
                    }
                }
            }

            var index = this.GetItemIndex(e.GetPosition(this));
            if (index >= 0)
            {
                var item = this.ItemContainerGenerator.ContainerFromIndex(index) as ListBoxItem;
                item.IsSelected = true;
            }
        }

        private int GetItemIndex(Point position)
        {
            var result = VisualTreeHelper.HitTest(this, position);
            if (result != null)
            {
                var visual = result.VisualHit;
                while (visual != null)
                {
                    if (visual is ListBoxItem)
                    {
                        return this.Items.IndexOf(((ListBoxItem)visual).Content);
                    }
                    visual = VisualTreeHelper.GetParent(visual);
                }
            }
            return -1;
        }

        protected override void OnPreviewMouseLeftButtonUp(MouseButtonEventArgs e)
        {
            // 範囲選択が行われていたのか？
            if (this.selecting == true)
            {
                this.adorner.End();
                this.selecting = false;
            }
            else
            {
                var result = VisualTreeHelper.HitTest(this, e.GetPosition(this));
                if (result != null)
                {
                    var item = result.VisualHit.FindParent<TimelineControlItem>();
                    if (item != null && item.IsSelected == true)
                    {
                        if ((Keyboard.Modifiers & ModifierKeys.Control) != ModifierKeys.Control)
                        {
                            this.UnselectAllTimelineControlItem();
                            item.IsSelected = true;
                        }
                    }
                }
            }
        }

        protected override void OnMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            if (this.selecting == false)
            {
                this.Begin(e.GetPosition(this));
            }
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (e.LeftButton == MouseButtonState.Pressed && this.selecting == true)
            {
                this.adorner.EndPoint = e.GetPosition(this);
                this.adorner.InvalidateVisual();

                Rect rubberBandBounds = new Rect(this.adorner.BeginPoint, this.adorner.EndPoint);

                foreach (var obj in this.Items)
                {
                    ListBoxItem item = this.ItemContainerGenerator.ContainerFromItem(obj) as ListBoxItem;
                    if (item != null)
                    {
                        Point point = item.TransformToAncestor(this).Transform(new Point(0, 0));
                        Rect bounds = new Rect(point.X, point.Y, item.ActualWidth, item.ActualHeight);
                        if (rubberBandBounds.IntersectsWith(bounds) == true)
                        {
                            SelectProcessForTimelineControlItem(rubberBandBounds, item.FindChildren<TimelineControlItem>());
                        }
                        else
                        {
                            item.FindChildren<TimelineControlItem>()
                                .ToList()
                                .ForEach(i => i.IsSelected = false);
                        }
                    }
                }
            }
        }

        private void Begin(Point beginPoint)
        {
            this.UnselectAllTimelineControlItem();

            this.adorner.Begin(beginPoint);
            this.selecting = true;
        }

        private void UnselectAllTimelineControlItem()
        {
            foreach (var obj in this.Items)
            {
                ListBoxItem item = this.ItemContainerGenerator.ContainerFromItem(obj) as ListBoxItem;
                item.FindChildren<TimelineControlItem>()
                    .ToList()
                    .ForEach(i => i.IsSelected = false);
            }
        }

        private void SelectProcessForTimelineControlItem(Rect rubberBandBounds, IEnumerable<TimelineControlItem> items)
        {
            foreach (var item in items)
            {
                var point = item.TransformToAncestor(this).Transform(new Point(0, 0));
                var bounds = new Rect(point.X, point.Y, item.ActualWidth, item.ActualHeight);
                item.IsSelected = rubberBandBounds.IntersectsWith(bounds) == true ? true : false;
            }
        }

        /// <summary>
        /// ラバーバンドのAdornerです。
        /// </summary>
        class RubberBandAdorner : Adorner
        {
            private Point beginPoint;
            private Point endPoint;
            private bool working = false;

            public RubberBandAdorner(UIElement element)
                : base(element)
            {
                this.AdornedElement.AddHandler(Mouse.LostMouseCaptureEvent, new RoutedEventHandler(OnLostMouseCapture));
            }

            public Point BeginPoint
            {
                get { return this.beginPoint; }
            }

            public Point EndPoint
            {
                get { return this.endPoint; }
                set { this.endPoint = value; }
            }

            public void Begin(Point beginPoint)
            {
                this.working = true;
                this.beginPoint = beginPoint;
                this.endPoint = beginPoint;

                var adornerLayer = AdornerLayer.GetAdornerLayer(this.AdornedElement);
                adornerLayer.Add(this);

                Mouse.Capture(this.AdornedElement);
                this.InvalidateVisual();
            }

            public void End()
            {
                if (this.working == false)
                {
                    return;
                }

                var adornerLayer = AdornerLayer.GetAdornerLayer(this.AdornedElement);
                adornerLayer.Remove(this);

                this.AdornedElement.ReleaseMouseCapture();
                this.working = false;
            }

            protected override void OnRender(DrawingContext drawingContext)
            {
                Rect bounds = new Rect(this.beginPoint, this.endPoint);
                var brush = new SolidColorBrush(SystemColors.HighlightColor);
                brush.Opacity = 0.2;

                drawingContext.DrawGeometry(brush, new Pen(SystemColors.HighlightBrush, 1), new RectangleGeometry(bounds));
                base.OnRender(drawingContext);
            }

            private void OnLostMouseCapture(object sender, RoutedEventArgs e)
            {
                this.End();
            }
        }
    }
}
