﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Xml;
using System.Globalization;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    using Nintendo.AudioToolkit.Extensions;

    /// <summary>
    ///
    /// </summary>
    public class TimelineControl : Selector
    {
        private Control dragItem = null;
        private Point dragPosition;
        private Stopwatch dragScrollCounter = new Stopwatch();
        private long lastDragScrollCount = 0;
        private bool canDrop = true;

        private static IList<DragAdorner> sDragAdorners = new List<DragAdorner>();
        private static TimelineControl sDragAdornerCreateControl = null;

        private ResizeAdorner resizeAdorner = null;

        public event DropFileEventHandler DropFile;
        internal event BeginDragEventHandler BeginDrag;

        public delegate void DropFileEventHandler(object sender, DropFileEventArgs e);
        internal delegate void BeginDragEventHandler(object sender, BeginDragEventArgs e);

        public ICommand ResizeStartedCommand
        {
            get { return (ICommand)base.GetValue(ResizeStartedCommandProperty); }
            set { base.SetValue(ResizeStartedCommandProperty, value); }
        }

        public static readonly DependencyProperty ResizeStartedCommandProperty = DependencyProperty.Register
            ("ResizeStartedCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand ResizeCompletedCommand
        {
            get { return (ICommand)base.GetValue(ResizeCompletedCommandProperty); }
            set { base.SetValue(ResizeCompletedCommandProperty, value); }
        }

        public static readonly DependencyProperty ResizeCompletedCommandProperty = DependencyProperty.Register
            ("ResizeCompletedCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand ResizeCanceledCommand
        {
            get { return (ICommand)base.GetValue(ResizeCanceledCommandProperty); }
            set { base.SetValue(ResizeCanceledCommandProperty, value); }
        }

        public static readonly DependencyProperty ResizeCanceledCommandProperty = DependencyProperty.Register
            ("ResizeCanceledCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand DropStartedCommand
        {
            get { return (ICommand)base.GetValue(DropStartedCommandProperty); }
            set { base.SetValue(DropStartedCommandProperty, value); }
        }

        public static readonly DependencyProperty DropStartedCommandProperty = DependencyProperty.Register
            ("DropStartedCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand DropCompletedCommand
        {
            get { return (ICommand)base.GetValue(DropCompletedCommandProperty); }
            set { base.SetValue(DropCompletedCommandProperty, value); }
        }

        public static readonly DependencyProperty DropCompletedCommandProperty = DependencyProperty.Register
            ("DropCompletedCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand DropCanceledCommand
        {
            get { return (ICommand)base.GetValue(DropCanceledCommandProperty); }
            set { base.SetValue(DropCanceledCommandProperty, value); }
        }

        public static readonly DependencyProperty DropCanceledCommandProperty = DependencyProperty.Register
            ("DropCanceledCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public ICommand CheckDragClipCommand
        {
            get { return (ICommand)base.GetValue(CheckDragClipCommandProperty); }
            set { base.SetValue(CheckDragClipCommandProperty, value); }
        }

        public static readonly DependencyProperty CheckDragClipCommandProperty = DependencyProperty.Register
            ("CheckDragClipCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());

        public Point MousePosition
        {
            get { return (Point)GetValue(MousePositionProperty); }
            set { SetValue(MousePositionProperty, value); }
        }

        public static readonly DependencyProperty MousePositionProperty =
            DependencyProperty.Register("MousePosition", typeof(Point), typeof(TimelineControl),
            new FrameworkPropertyMetadata());



        public ICommand RequestDuplicateCommand
        {
            get { return (ICommand)base.GetValue(RequestDuplicateCommandProperty); }
            set { base.SetValue(RequestDuplicateCommandProperty, value); }
        }

        public static readonly DependencyProperty RequestDuplicateCommandProperty = DependencyProperty.Register
            ("RequestDuplicateCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());


        public ICommand RequestMoveCommand
        {
            get { return (ICommand)base.GetValue(RequestMoveCommandProperty); }
            set { base.SetValue(RequestMoveCommandProperty, value); }
        }

        public static readonly DependencyProperty RequestMoveCommandProperty = DependencyProperty.Register
            ("RequestMoveCommand", typeof(ICommand), typeof(TimelineControl),
            new FrameworkPropertyMetadata());


        /// <summary>
        ///
        /// </summary>
        public TimelineControl()
        {
            //DefaultStyleKeyProperty.OverrideMetadata
            //    (typeof(TimelineControl), new FrameworkPropertyMetadata(typeof(TimelineControl)));

            AllowDrop = true;

            this.resizeAdorner = new ResizeAdorner(this);
        }

        /// <summary>
        /// 水平方向の拡大率です。
        /// </summary>
        public double HorizontalScale
        {
            get { return (double)GetValue(HorizontalScaleProperty); }
            set { SetValue(HorizontalScaleProperty, value); }
        }

        public static readonly DependencyProperty HorizontalScaleProperty =
            DependencyProperty.Register("HorizontalScale", typeof(double), typeof(TimelineControl),
            new FrameworkPropertyMetadata(1.0, new PropertyChangedCallback(OnHorizontalScaleChanged)));

        private static void OnHorizontalScaleChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            TimelineControl control = d as TimelineControl;
            control.InvalidateVisual();

            var panel = control.FindChild<TimelinePanel>();
            if (panel != null)
            {
                panel.InvalidateVisual();

                panel.FindChildren<TimelineControlItem>().ToList().ForEach(i => i.InvalidateVisual());
            }
        }

        /// <summary>
        /// 水平方向のスナップ値です。
        /// </summary>
        public double HorizontalSnap
        {
            get { return (double)GetValue(HorizontalSnapProperty); }
            set { SetValue(HorizontalSnapProperty, value); }
        }

        public static readonly DependencyProperty HorizontalSnapProperty =
            DependencyProperty.Register("HorizontalSnap", typeof(double), typeof(TimelineControl),
            new FrameworkPropertyMetadata(1.0));

        public double VerticalSnap
        {
            get { return (double)GetValue(VerticalSnapProperty); }
            set { SetValue(VerticalSnapProperty, value); }
        }

        public static readonly DependencyProperty VerticalSnapProperty =
            DependencyProperty.Register("VerticalSnap", typeof(double), typeof(TimelineControl),
            new FrameworkPropertyMetadata(20.0));

        protected override bool IsItemItsOwnContainerOverride(object item)
        {
            return item is TimelineControlItem;
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            base.PrepareContainerForItemOverride(element, item);

            var control = element as TimelineControlItem;
            if (control != null)
            {
                control.Content = item;
                control.DataContext = item;

                control.AddHandler(Thumb.DragDeltaEvent, new DragDeltaEventHandler(OnDragDeltaResizeThumb));
                control.AddHandler(Thumb.DragStartedEvent, new DragStartedEventHandler(OnDragStartedResizeThumb));
                control.AddHandler(Thumb.DragCompletedEvent, new DragCompletedEventHandler(OnDragCompletedResizeThumb));
                control.AddHandler(Mouse.MouseDownEvent, new MouseButtonEventHandler(OnMouseLeftButtonDown));
                control.AddHandler(Mouse.MouseMoveEvent, new MouseEventHandler(OnMouseMove));
            }
        }

        protected override void ClearContainerForItemOverride(DependencyObject element, object item)
        {
            var control = element as TimelineControlItem;
            if (control != null)
            {
                control.RemoveHandler(Thumb.DragDeltaEvent, new DragDeltaEventHandler(OnDragDeltaResizeThumb));
                control.RemoveHandler(Thumb.DragStartedEvent, new DragStartedEventHandler(OnDragStartedResizeThumb));
                control.RemoveHandler(Thumb.DragCompletedEvent, new DragCompletedEventHandler(OnDragCompletedResizeThumb));
                control.RemoveHandler(Mouse.MouseDownEvent, new MouseButtonEventHandler(OnMouseLeftButtonDown));
                control.RemoveHandler(Mouse.MouseMoveEvent, new MouseEventHandler(OnMouseMove));
            }

            base.ClearContainerForItemOverride(element, item);
        }

        protected override DependencyObject GetContainerForItemOverride()
        {
            return new TimelineControlItem();
        }

        private void OnDragDeltaResizeThumb(object sender, DragDeltaEventArgs e)
        {
            var thumb = e.OriginalSource as Thumb;
            var item = sender as TimelineControlItem;

            if (thumb == item.RightThumb)
            {
                var nx = (e.HorizontalChange + this.resizeAdorner.BeginRightPoint.X) / this.HorizontalScale;
                nx = this.SnapHorizontal(nx) * this.HorizontalScale - this.resizeAdorner.BeginRightPoint.X;

                this.resizeAdorner.RelativeRight = new Vector(nx, 0);
                e.Handled = true;
            }
            else if (thumb == item.LeftThumb)
            {
                var nx = (e.HorizontalChange + this.resizeAdorner.BeginLeftPoint.X) / this.HorizontalScale;
                nx = this.SnapHorizontal(nx) * this.HorizontalScale - this.resizeAdorner.BeginLeftPoint.X;

                this.resizeAdorner.RelativeLeft = new Vector(nx, 0);
                e.Handled = true;
            }
        }

        private void OnDragStartedResizeThumb(object sender, DragStartedEventArgs e)
        {
            var thumb = e.OriginalSource as Thumb;
            var item = sender as TimelineControlItem;
            var x = Canvas.GetLeft(item);
            var y = Canvas.GetTop(item);

            if (thumb == item.RightThumb)
            {
                var minWidth = 1.0;
                var maxWidth = item.MaximumWidth ?? double.MaxValue;

                var minLimitX = x + minWidth;
                var maxLimitX = x + maxWidth - item.OffsetX;

                this.resizeAdorner.Begin
                    (new Point(x, y), new Point(x + item.Width, y + item.Height),
                     default(Point), default(Point),
                     new Point(minLimitX, 0), new Point(maxLimitX, 0));

                this.OnResizeStarted();
            }
            else if (thumb == item.LeftThumb)
            {
                var minLimitX = x - item.OffsetX;
                var maxLimitX = x + item.ActualWidth - thumb.Width;

                this.resizeAdorner.Begin
                    (new Point(x, y), new Point(x + item.Width, y + item.Height),
                     new Point(minLimitX, 0), new Point(maxLimitX, 0),
                     default(Point), default(Point));

                this.OnResizeStarted();
            }
        }

        private void OnDragCompletedResizeThumb(object sender, DragCompletedEventArgs e)
        {
            var thumb = e.OriginalSource as Thumb;
            var item = sender as TimelineControlItem;

            try
            {
                if (thumb == item.RightThumb)
                {
                    item.Width += this.resizeAdorner.RelativeRight.X;
                }
                else if (thumb == item.LeftThumb)
                {
                    var oldX = Canvas.GetLeft(item);
                    var newX = this.resizeAdorner.LeftPoint.X;

                    Canvas.SetLeft(item, newX);
                    item.Width += -this.resizeAdorner.RelativeLeft.X;
                    item.OffsetX += newX - oldX;
                }
            }
            catch
            {
                this.OnResizeCanceled();
            }
            finally
            {
                this.OnResizeCompleted();
            }
        }

        private void OnMouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            var item = sender as TimelineControlItem;

            this.dragItem = item;
            this.dragPosition = e.GetPosition(this);
        }

        private void OnMouseMove(object sender, MouseEventArgs e)
        {
            var item = sender as TimelineControlItem;

            if (e.OriginalSource is Thumb)
            {
                return;
            }

            if (e.LeftButton == MouseButtonState.Pressed && this.dragItem == item)
            {
                var position = e.GetPosition(this);
                if (Math.Abs(position.X - this.dragPosition.X) > SystemParameters.MinimumHorizontalDragDistance ||
                    Math.Abs(position.Y - this.dragPosition.Y) > SystemParameters.MinimumVerticalDragDistance)
                {
                    var args = new BeginDragEventArgs();
                    this.OnBeginDrag(args);

                    var dragItems = args.DragItems;
                    if (dragItems.Count > 0)
                    {
                        //this.dragItem.AllowDrop = true;
                        this.AllowDrop = true;

                        // とりあえずGridにしている
                        var grid = this.FindParent<Grid>();
                        var layer = AdornerLayer.GetAdornerLayer(grid);
                        this.CreateDragAdorner(dragItems, this.dragPosition).ToList().ForEach(a => sDragAdorners.Add(a));

                        sDragAdorners.ToList().ForEach(a => layer.Add(a));
                        sDragAdornerCreateControl = this;

                        // ドラッグを開始します。
                        var data = new DataObject();
                        data.SetData("ItemsSource", this.ItemsSource);
                        data.SetData("DragPosition", this.dragPosition);
                        data.SetData("DragItem", this.dragItem);
                        data.SetData("DragItems", dragItems);

                        this.dragScrollCounter.Reset();
                        this.dragScrollCounter.Start();
                        this.lastDragScrollCount = 0;

                        DragDrop.DoDragDrop(item, data, DragDropEffects.Move | DragDropEffects.Copy);

                        this.dragScrollCounter.Stop();

                        // ドラッグ完了後の後始末をします。
                        this.RemoveDragAdorner();

                        this.MousePosition = new Point(double.NaN, double.NaN);
                        sDragAdornerCreateControl = null;

                        e.Handled = true;
                    }
                }
            }
        }

        protected override void OnQueryContinueDrag(QueryContinueDragEventArgs e)
        {
            base.OnQueryContinueDrag(e);

            this.MousePosition = MouseCursor.GetPosition(this);

            // ドラッグ時にListBoxの範囲外にマウスが出たらスクロールを行います。
            var nowDragScrollCount = this.dragScrollCounter.ElapsedMilliseconds;
            if (nowDragScrollCount - lastDragScrollCount > 50)
            {
                var control = this.FindParent<TimelineListBox>();
                var border = VisualTreeHelper.GetChild(control, 0) as Border;
                var scrollViewer = VisualTreeHelper.GetChild(border, 0) as ScrollViewer;
                var width = scrollViewer.ViewportWidth;
                var height = scrollViewer.ViewportHeight;

                var mousePosition = MouseCursor.GetPosition(control);
                var hideAdorners = false;

                if (mousePosition.X < 0)
                {
                    ExecuteCommand(this.RequestMoveCommand, new RequestMoveEventArgs(RequestMoveEventArgs.MoveDirections.Left, -mousePosition.X));
                    hideAdorners = true;
                }
                else if (mousePosition.X > width)
                {
                    ExecuteCommand(this.RequestMoveCommand, new RequestMoveEventArgs(RequestMoveEventArgs.MoveDirections.Right, mousePosition.X - width));
                    hideAdorners = true;
                }

                if (mousePosition.Y < 0)
                {
                    ExecuteCommand(this.RequestMoveCommand, new RequestMoveEventArgs(RequestMoveEventArgs.MoveDirections.Up, -mousePosition.Y));
                    hideAdorners = true;
                }
                else if (mousePosition.Y > height)
                {
                    ExecuteCommand(this.RequestMoveCommand, new RequestMoveEventArgs(RequestMoveEventArgs.MoveDirections.Down, mousePosition.Y - height));
                    hideAdorners = true;
                }

                if (hideAdorners == true)
                {
                    sDragAdorners.ForEach(a => a.Visibility = Visibility.Collapsed);
                }

                this.lastDragScrollCount = nowDragScrollCount;
            }
        }

        protected override void OnGiveFeedback(System.Windows.GiveFeedbackEventArgs e)
        {
            if (this.canDrop == false)
            {
                Mouse.SetCursor(Cursors.No);
                e.Handled = true;
            }
        }

        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            if (sDragAdornerCreateControl != null && sDragAdornerCreateControl != this)
            {
                this.RemoveDragAdorner();

                var dragItems = e.Data.GetData("DragItems") as IList<TimelineControlItem>;
                var grid = this.FindParent<Grid>();
                var layer = AdornerLayer.GetAdornerLayer(grid);
                this.CreateDragAdorner(dragItems, dragPosition).ToList().ForEach((a) => { layer.Add(a); sDragAdorners.Add(a); });
                sDragAdornerCreateControl = this;

            }
        }

        protected override void OnDragOver(DragEventArgs e)
        {
            base.OnDragOver(e);

            if (sDragAdornerCreateControl != null)
            {
                var cardinalDragItem = e.Data.GetData("DragItem") as TimelineControlItem;
                var dragItems = e.Data.GetData("DragItems") as IList<TimelineControlItem>;

                var dropPosition = e.GetPosition(this);
                var beginDragPosition = (Point)e.Data.GetData("DragPosition");
                var x = dropPosition.X - beginDragPosition.X;
                var y = dropPosition.Y;

                var by = Canvas.GetTop(cardinalDragItem);

                object[] contents = null;
                if ((e.KeyStates & DragDropKeyStates.ControlKey) == DragDropKeyStates.ControlKey)
                {
                    contents = new object[0];
                }
                else
                {
                    contents = dragItems
                        .Select(i => i.Content)
                        .ToArray();
                }

                this.canDrop = true;
                for (int index = 0; index < sDragAdorners.Count(); index++)
                {
                    var dragAdorner = sDragAdorners[index];
                    var dragItem = dragItems[index];
                    var dy = Canvas.GetTop(dragItem) - by;

                    var nx = (Canvas.GetLeft(dragItem) + x) / this.HorizontalScale;
                    var ny = y + dy;
                    this.Snap(ref nx, ref ny);

                    dragAdorner.LeftOffset = nx * this.HorizontalScale;
                    dragAdorner.TopOffset = ny;
                    dragAdorner.Visibility = Visibility.Visible;

                    if (this.canDrop == true)
                    {
                        var args = new CheckDragClipEventArgs(this, dragItem.Content, contents, nx, ny);
                        ExecuteCommand(this.CheckDragClipCommand, args);
                        if (args.Result == false)
                        {
                            this.canDrop = false;
                        }
                    }
                }
            }

            if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                string[] files = e.Data.GetData(DataFormats.FileDrop) as string[];
            }
        }

        /// <summary>
        /// ドロップのハンドラです。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnDrop(DragEventArgs e)
        {
            if (sDragAdornerCreateControl != null && this.canDrop == true)
            {
                var dropPosition = e.GetPosition(this);
                var beginDragPosition = (Point)e.Data.GetData("DragPosition");
                var x = dropPosition.X - beginDragPosition.X;
                var y = dropPosition.Y;

                var cardinalDragItem = e.Data.GetData("DragItem") as TimelineControlItem;
                var dragItems = e.Data.GetData("DragItems") as IList<TimelineControlItem>;

                var sourceItemsSource = e.Data.GetData("ItemsSource") as System.Collections.IList;
                var targetItemsSource = this.ItemsSource as System.Collections.IList;
                var by = Canvas.GetTop(cardinalDragItem);

                try
                {
                    this.OnDropStarted();

                    if (e.KeyStates == DragDropKeyStates.ControlKey)
                    {
                        foreach (var dragItem in dragItems)
                        {
                            var content = dragItem.Content;
                            var nx = (Canvas.GetLeft(dragItem) + x) / this.HorizontalScale;
                            var ny = Canvas.GetTop(dragItem) + y - by;
                            this.Snap(ref nx, ref ny);

                            var args = new RequestDuplicateEventArgs(this, nx, ny, content);
                            ExecuteCommand(this.RequestDuplicateCommand, args);
                        }
                    }
                    else
                    {
                        if (sourceItemsSource != targetItemsSource)
                        {
                            foreach (var dragItem in dragItems)
                            {
                                TimelineControlItem sourceItem = dragItem;
                                var content = sourceItem.Content;
                                var sourceItemIsSelected = sourceItem.IsSelected;
                                var sourceItemOffsetX = sourceItem.OffsetX;

                                sourceItemsSource.Remove(content);
                                targetItemsSource.Add(content);

                                TimelineControlItem targetItem = this.ItemContainerGenerator.ContainerFromItem(content) as TimelineControlItem;
                                targetItem.IsSelected = sourceItemIsSelected;
                                targetItem.OffsetX = sourceItemOffsetX;

                                var nx = Canvas.GetLeft(targetItem) + x;
                                var ny = Canvas.GetTop(targetItem) + y - by;

                                this.Snap(ref nx, ref ny);
                                Canvas.SetLeft(targetItem, nx);
                                Canvas.SetTop(targetItem, ny);
                            }
                        }
                        else
                        {
                            foreach (var dragItem in dragItems)
                            {
                                var nx = (Canvas.GetLeft(dragItem) + x) / this.HorizontalScale;
                                var ny = Canvas.GetTop(dragItem) + y - by;
                                this.Snap(ref nx, ref ny);

                                Canvas.SetLeft(dragItem, nx * this.HorizontalScale);
                                Canvas.SetTop(dragItem, ny);
                            }
                        }
                    }
                }
                catch
                {
                    this.OnDropCanceled();
                }
                finally
                {
                    this.OnDropCompleted();
                }
            }
            else if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                string[] filePaths = e.Data.GetData(DataFormats.FileDrop) as string[];
                var position = e.GetPosition(this);

                this.OnDropFile(filePaths, new Point(position.X / this.HorizontalScale, position.Y / this.VerticalSnap));
            }

            base.OnDrop(e);
        }

        /// <summary>
        /// ファイルがドロップされた時のハンドラです。
        /// </summary>
        private void OnDropFile(string[] filePaths, Point position)
        {
            if (this.DropFile != null)
            {
                this.DropFile(this, new DropFileEventArgs(this, position, filePaths));
            }
        }

        private void OnBeginDrag(BeginDragEventArgs e)
        {
            if (this.BeginDrag != null)
            {
                this.BeginDrag(this, e);
            }
        }

        public class DropFileEventArgs : EventArgs
        {
            private object sender = null;
            private string[] filePaths;
            private Point position;

            public DropFileEventArgs(object sender, Point position, string[] filePaths)
            {
                this.sender = sender;
                this.position = position;
                this.filePaths = filePaths;
            }

            public object Sender
            {
                get
                {
                    return this.sender;
                }
            }

            public Point Position
            {
                get
                {
                    return this.position;
                }
            }

            public string[] FilePaths
            {
                get
                {
                    return this.filePaths;
                }
            }
        }

        public class CheckDragClipEventArgs : EventArgs
        {
            private object sender = null;
            private object target = null;
            private object[] excludeTargets = null;
            private double x = 0.0;
            private double y = 0.0;

            public CheckDragClipEventArgs(object sender, object target, object[] excludeTargets, double x, double y)
            {
                this.sender = sender;
                this.target = target;
                this.excludeTargets = excludeTargets;
                this.x = x;
                this.y = y;
                this.Result = true;
            }

            public bool Result
            {
                get;
                set;
            }

            public object Sender
            {
                get
                {
                    return this.sender;
                }
            }

            public object Target
            {
                get
                {
                    return this.target;
                }
            }

            public object[] ExcludeTargets
            {
                get
                {
                    return this.excludeTargets;
                }
            }

            public double X
            {
                get
                {
                    return this.x;
                }
            }

            public double Y
            {
                get
                {
                    return this.y;
                }
            }
        }

        public class RequestDuplicateEventArgs : EventArgs
        {
            private object sender = null;
            private object target;
            private double x;
            private double y;

            public RequestDuplicateEventArgs(object sender, double x, double y, object target)
            {
                this.sender = sender;
                this.x = x;
                this.y = y;
                this.target = target;
            }

            public object Sender
            {
                get
                {
                    return this.sender;
                }
            }

            public object Target
            {
                get
                {
                    return this.target;
                }
            }

            public double X
            {
                get
                {
                    return this.x;
                }
            }

            public double Y
            {
                get
                {
                    return this.y;
                }
            }
        }

        public class RequestMoveEventArgs : EventArgs
        {
            public enum MoveDirections
            {
                Left,
                Right,
                Up,
                Down,
            };

            private MoveDirections moveDirection;
            private double protrusionValue = 0.0;

            public RequestMoveEventArgs(MoveDirections moveDirection, double protrusionValue)
            {
                this.moveDirection = moveDirection;
                this.protrusionValue = protrusionValue;
            }

            public MoveDirections MoveDirection
            {
                get
                {
                    return this.moveDirection;
                }
            }

            public double ProtrusionValue
            {
                get
                {
                    return this.protrusionValue;
                }
            }
        }

        internal class BeginDragEventArgs : EventArgs
        {
            private IList<TimelineControlItem> dragItems = new List<TimelineControlItem>();

            public BeginDragEventArgs()
            {
            }

            public IList<TimelineControlItem> DragItems
            {
                get
                {
                    return this.dragItems;
                }
            }
        }

        private void ExecuteCommand(ICommand command, object args = null)
        {
            if (command != null && command.CanExecute(null) == true)
            {
                command.Execute(args);
            }
        }

        private void OnResizeStarted()
        {
            ExecuteCommand(this.ResizeStartedCommand);
        }

        private void OnResizeCompleted()
        {
            ExecuteCommand(this.ResizeCompletedCommand);
        }

        private void OnResizeCanceled()
        {
            ExecuteCommand(this.ResizeCanceledCommand);
        }

        private void OnDropStarted()
        {
            ExecuteCommand(this.DropStartedCommand);
        }

        private void OnDropCompleted()
        {
            ExecuteCommand(this.DropCompletedCommand);
        }

        private void OnDropCanceled()
        {
            ExecuteCommand(this.DropCanceledCommand);
        }

        private IEnumerable<DragAdorner> CreateDragAdorner(IEnumerable<TimelineControlItem> items, Point position)
        {
            var adorned = this;
            return items.Select(i => new DragAdorner(adorned, i, 0.5, position));
        }

        private void RemoveDragAdorner()
        {
            var grid = this.FindParent<Grid>();
            var layer = AdornerLayer.GetAdornerLayer(grid);
            sDragAdorners.ToList().ForEach(a => layer.Remove(a));
            sDragAdorners.Clear();
        }

        /// <summary>
        /// スナップ処理です。
        /// </summary>
        /// <param name="x"></param>
        private void Snap(ref double x, ref double y)
        {
            x = (int)(x / this.HorizontalSnap) * this.HorizontalSnap;
            y = (int)(y / this.VerticalSnap) * this.VerticalSnap;
        }

        private double SnapHorizontal(double value)
        {
            return (double)((int)(value / this.HorizontalSnap) * this.HorizontalSnap);
        }

        /// <summary>
        /// リサイズ用のAdornerです。
        /// </summary>
        class ResizeAdorner : Adorner
        {
            private Point beginLeftPoint;
            private Point beginRightPoint;
            private Vector relativeLeft;
            private Vector relativeRight;
            private Point leftMinLimitPoint;
            private Point leftMaxLimitPoint;
            private Point rightMinLimitPoint;
            private Point rightMaxLimitPoint;
            private bool working = false;

            public ResizeAdorner(UIElement owner) : base(owner)
            {
                this.AdornedElement.AddHandler(Mouse.LostMouseCaptureEvent, new RoutedEventHandler(OnLostMouseCapture));
            }

            public Vector RelativeLeft
            {
                get { return this.relativeLeft; }
                set
                {
                    var rx = this.ValidateRelativeValue(value.X, this.beginLeftPoint.X, this.leftMinLimitPoint.X, this.leftMaxLimitPoint.X);
                    var ry = value.Y;
                    this.relativeLeft = new Vector(rx, ry);
                    this.InvalidateVisual();
                }
            }

            public Vector RelativeRight
            {
                get { return this.relativeRight; }
                set
                {
                    var rx = this.ValidateRelativeValue(value.X, this.beginRightPoint.X, this.rightMinLimitPoint.X, this.rightMaxLimitPoint.X);
                    var ry = value.Y;
                    this.relativeRight = new Vector(rx, ry);
                    this.InvalidateVisual();
                }
            }

            public Point BeginLeftPoint
            {
                get
                {
                    return this.beginLeftPoint;
                }
            }

            public Point BeginRightPoint
            {
                get
                {
                    return this.beginRightPoint;
                }
            }

            public Point LeftPoint
            {
                get
                {
                    return Point.Add(this.beginLeftPoint, this.relativeLeft);
                }
            }

            public Point RightPoint
            {
                get
                {
                    return Point.Add(this.beginRightPoint, this.relativeRight);
                }
            }

            public void Begin(Point leftPoint, Point rightPoint, Point leftMinLimitPoint, Point leftMaxLimitPoint, Point rightMinLimitPoint, Point rightMaxLimitPoint)
            {
                this.working = true;

                // 移動の開始です。
                this.beginLeftPoint = leftPoint;
                this.beginRightPoint = rightPoint;

                // 移動可能な範囲です。
                this.leftMinLimitPoint = leftMinLimitPoint;
                this.leftMaxLimitPoint = leftMaxLimitPoint;
                this.rightMinLimitPoint = rightMinLimitPoint;
                this.rightMaxLimitPoint = rightMaxLimitPoint;

                // どれだけ移動させるかの相対値です。
                this.relativeLeft = new Vector(0.0, 0.0);
                this.relativeRight = new Vector(0.0, 0.0);

                var adornerLayer = AdornerLayer.GetAdornerLayer(this.AdornedElement);
                adornerLayer.Add(this);

                this.InvalidateVisual();
            }

            public void End()
            {
                if (this.working == false)
                {
                    return;
                }

                var adornerLayer = AdornerLayer.GetAdornerLayer(this.AdornedElement);
                adornerLayer.Remove(this);

                this.working = false;
            }

            protected override void OnRender(DrawingContext drawingContext)
            {
                Rect bounds = new Rect(Point.Add(this.LeftPoint, new Vector(1, 1)), Point.Add(this.RightPoint, new Vector(-1, -1)));
                var brush = new SolidColorBrush(SystemColors.HighlightColor);
                brush.Opacity = 0.2;

                drawingContext.DrawGeometry(brush, new Pen(SystemColors.HighlightBrush, 0.5), new RectangleGeometry(bounds));
                base.OnRender(drawingContext);
            }

            private void OnLostMouseCapture(object sender, RoutedEventArgs e)
            {
                this.End();
            }


            private double ValidateRelativeValue(double relativeValue, double baseValue, double minLimit, double maxLimit)
            {
                var value = baseValue + relativeValue;

                if (value > maxLimit)
                {
                    relativeValue = maxLimit - baseValue;
                }
                if (value < minLimit)
                {
                    relativeValue = minLimit - baseValue;
                }
                return relativeValue;
            }
        }

        /// <summary>
        /// ドラック時のAdornerです。
        /// </summary>
        class DragAdorner : Adorner
        {
            private double leftOffset;
            private double topOffset;
            private UIElement element;

            public DragAdorner(UIElement owner) : base(owner)
            {
            }

            public DragAdorner(UIElement owner, UIElement adornerElement, double opacity, Point dragPosition)
                : base(owner)
            {
                var adornerLocation = adornerElement.PointToScreen(new Point(0, 0));
                var ownerLocation = owner.PointToScreen(new Point(0, 0));

                var bounds = VisualTreeHelper.GetDescendantBounds(adornerElement);
                this.element = new Rectangle()
                {
                    Width = bounds.Width,
                    Height = bounds.Height,
                    Fill = new VisualBrush(adornerElement) { Opacity = opacity }
                };
            }

            public double LeftOffset
            {
                get { return this.leftOffset; }
                set
                {
                    this.leftOffset = value;
                    UpdatePosition();
                }
            }

            public double TopOffset
            {
                get { return this.topOffset; }
                set
                {
                    this.topOffset = value;
                    UpdatePosition();
                }
            }

            private void UpdatePosition()
            {
                var adorner = this.Parent as AdornerLayer;
                if (adorner != null)
                {
                    adorner.Update(this.AdornedElement);
                }
            }

            protected override Visual GetVisualChild(int index)
            {
                return this.element;
            }

            protected override int VisualChildrenCount
            {
                get { return 1; }
            }

            protected override Size MeasureOverride(Size finalSize)
            {
                this.element.Measure(finalSize);
                return this.element.DesiredSize;
            }

            protected override Size ArrangeOverride(Size finalSize)
            {

                this.element.Arrange(new Rect(this.element.DesiredSize));
                return finalSize;
            }

            public override GeneralTransform GetDesiredTransform(GeneralTransform transform)
            {
                var x = (int)this.leftOffset;
                var y = (int)this.topOffset;

                var result = new GeneralTransformGroup();
                result.Children.Add(base.GetDesiredTransform(transform));
                result.Children.Add(new TranslateTransform(x, y));
                return result;
            }
        }
    }

    /// <summary>
    /// カーソル位置を取得するクラスです。
    /// </summary>
    public static class MouseCursor
    {
        [StructLayout(LayoutKind.Sequential)]
        public struct POINT
        {
            public int X;
            public int Y;
        }

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool GetCursorPos(out POINT lpPoint);

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool ScreenToClient(IntPtr hWnd, ref POINT lpPoint);

        public static Point GetPosition(Visual visual)
        {
            POINT point;
            GetCursorPos(out point);
            return visual.PointFromScreen(new Point(point.X, point.Y));
        }
    }
}
