﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace Nintendo.AudioToolkit.Extensions
{
    /// <summary>
    ///
    /// </summary>
    public static class PropertyChangedEventHandlerExtensions
    {
        public static void Raise<T>(this PropertyChangedEventHandler handler, Expression<Func<T>> propertyName)
        {
            if (handler == null)
            {
                return;
            }

            var member = propertyName.Body as MemberExpression;
            if (member == null)
            {
                throw new ArgumentException();
            }

            var constant = member.Expression as ConstantExpression;
            if (constant == null)
            {
                throw new ArgumentException();
            }

            var sender = System.Linq.Expressions.Expression.Lambda(constant).Compile().DynamicInvoke();
            handler(sender, new PropertyChangedEventArgs(member.Member.Name));
        }
    }


    /// <summary>
    ///
    /// </summary>
    public static class INotifyPropertyChangedExtensions
    {
        public static void AddHandler<T>(this INotifyPropertyChanged target, EventHandler<PropertyChangedEventArgs> handler, Expression<Func<T>> propertyName)
        {
            var member = propertyName.Body as MemberExpression;
            if (member == null)
            {
                throw new ArgumentException();
            }

            PropertyChangedEventManager.AddHandler(target, handler, member.Member.Name);
        }


        public static bool SetProperty<T>(this INotifyPropertyChanged target, PropertyChangedEventHandler handler, ref T storage, T value, [CallerMemberName] String propertyName = null)
        {
            if (object.Equals(storage, value) == true)
            {
                return false;
            }

            storage = value;

            if (handler != null)
            {
                handler(target, new PropertyChangedEventArgs(propertyName));
            }
            return true;
        }

        public static bool SetProperty<T>(this INotifyPropertyChanged target, PropertyChangedEventHandler handler, ref T storage, T value, Expression<Func<T>> propertyName)
        {
            var member = propertyName.Body as MemberExpression;
            if (member == null)
            {
                throw new ArgumentException();
            }

            return SetProperty(target, handler, ref storage, value, member.Member.Name);
        }

        public static void AddPropertyChanged<TObj, TProp>(this TObj target, Expression<Func<TObj, TProp>> propertyName, Action<TObj> handler) where TObj : INotifyPropertyChanged
        {
            var name = ((MemberExpression)propertyName.Body).Member.Name;
            target.PropertyChanged += (sender, e) =>
            {
                if (e.PropertyName == name)
                {
                    handler(target);
                }
            };
        }

    }




    public static class DependencyObjectExtensions
    {
        public static T FindParent<T>(this DependencyObject target) where T : DependencyObject
        {
            DependencyObject parent = VisualTreeHelper.GetParent(target);

            if (parent == null)
            {
                return null;
            }

            if (parent is T)
            {
                return parent as T;
            }

            return FindParent<T>(parent);
        }

        public static T FindChild<T>(this DependencyObject target) where T : DependencyObject
        {
            for (int index = 0; index < VisualTreeHelper.GetChildrenCount(target); index++)
            {
                var child = VisualTreeHelper.GetChild(target, index);
                if (child is T)
                {
                    return child as T;
                }

                T result = FindChild<T>(child);
                if (result != null)
                {
                    return result;
                }
            }
            return null;
        }

        public static IEnumerable<T> FindChildren<T>(this DependencyObject target) where T : DependencyObject
        {
            for (int index = 0; index < VisualTreeHelper.GetChildrenCount(target); index++)
            {
                var child = VisualTreeHelper.GetChild(target, index);
                if (child is T)
                {
                    yield return child as T;
                }

                foreach (var result in child.FindChildren<T>())
                {
                    yield return result;
                }
            }
        }
    }
}
