﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"
#include <panels/cafe/WavePreviewCafePanel.h>

#include <nw/ut/ut_Memory.h>
#include <nw/snd/fnd/io/sndfnd_File.h>
#include <resources/GlobalResources.h>

namespace nw {
namespace snd {

WavePreviewCafePanel::WavePreviewCafePanel()
#if defined(NW_PLATFORM_CAFE)
: m_FsClient(NULL)
, m_FsCmdBlock(NULL)
#endif
{
}

WavePreviewCafePanel::~WavePreviewCafePanel()
{
}

bool WavePreviewCafePanel::Initialize(IResourceProvider& resourceProvider)
{
    ut::IAllocator* allocator = reinterpret_cast<ut::IAllocator*>(
        resourceProvider.GetResource(GlobalResources::AllocatorPath));

    Base::Initialize(allocator, m_FileBrowser);

#if defined(NW_PLATFORM_CAFE)
    m_FsClient = reinterpret_cast<FSClient*>(
        resourceProvider.GetResource(GlobalResources::FsClientPath));

    m_FsCmdBlock = reinterpret_cast<FSCmdBlock*>(
        resourceProvider.GetResource(GlobalResources::FsCmdBlockPath));

    const char* hfioVolume = reinterpret_cast<const char*>(
        resourceProvider.GetConstResource(GlobalResources::HfioVolumePath));

    m_FileBrowser.Initialize(m_FsClient, m_FsCmdBlock);

    m_FileBrowser.SetCurrentDirectoryPath(hfioVolume);
    m_FileBrowser.UpdateFileEntries();

    m_FsCmdBlockForPreview = reinterpret_cast<FSCmdBlock*>(allocator->Alloc(sizeof(FSCmdBlock)));
    FSInitCmdBlock(m_FsCmdBlockForPreview);
#endif

    return true;
}

void WavePreviewCafePanel::Finalize()
{
#if defined(NW_PLATFORM_CAFE)
    if(m_FsCmdBlockForPreview != NULL)
    {
        NW_ASSERT_NOT_NULL(GetAllocator() != NULL);
        GetAllocator()->Free(m_FsCmdBlockForPreview);
        m_FsCmdBlockForPreview = NULL;
    }
#endif

    m_FileBrowser.Finalize();
}

internal::fnd::FileStream*
WavePreviewCafePanel::OpenFileImpl(const char* filePath)
{
#if defined(NW_PLATFORM_CAFE)
    if(m_FileStream.IsOpened())
    {
        m_FileStream.Close();
    }

    if(internal::fnd::File::Open(
        &m_FileStream,
        m_FsClient,
        m_FsCmdBlockForPreview,
        filePath,
        internal::fnd::File::ACCESS_MODE_READ
        ).IsFailed())
    {
        return NULL;
    }

    return &m_FileStream;
#else
    return NULL;
#endif
}

WaveRenderer& WavePreviewCafePanel::GetWaveRenderer()
{
    return m_WaveRenderer;
}

void WavePreviewCafePanel::InitializeLampBar(nw::internal::dw::StackPanel& lampBar)
{
    m_LampBarContents.AddItem(&m_MainLamp);
    m_LampBarContents.AddItem(&m_DrcLamp);
    lampBar.SetContents(m_LampBarContents);

    m_MainLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_MainLamp.SetLabelText("Main");
    m_MainLamp.SetIsLamped(true);

    m_DrcLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_DrcLamp.SetLabelText("DRC");
}

void WavePreviewCafePanel::SetOutputLine(WaveRenderer& renderer)
{
    if(IsMainOutput())
    {
        m_WaveRenderer.SetOutputLine(WaveRendererCafe::OUTPUT_MAIN);
    }
    else
    {
        m_WaveRenderer.SetOutputLine(WaveRendererCafe::OUTPUT_DRC0);
    }
}

bool WavePreviewCafePanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(Base::OnUpdateFocusedInput(inputs))
    {
        return true;
    }

    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
        pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
    {
        if(IsMainOutput())
        {
            m_MainLamp.SetIsLamped(false);
            m_DrcLamp.SetIsLamped(true);
        }
        else
        {
            m_MainLamp.SetIsLamped(true);
            m_DrcLamp.SetIsLamped(false);
        }

        return true;
    }

    return false;
}

bool WavePreviewCafePanel::IsMainOutput()
{
    return m_MainLamp.GetIsLamped();
}

} // snd
} // nw
