﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"
#include <panels/WavePreviewPanel.h>

#include <nw/snd/fnd/io/sndfnd_FileStream.h>
#include <resources/GlobalResources.h>
#include <io/WaveRenderer.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace snd {

WavePreviewPanel::WavePreviewPanel() :
m_pFileBrowser(NULL),
m_pCurrentReader(NULL),
m_ReadFileBuffer(NULL),
m_FileFilterEventHandler(this, &WavePreviewPanel::OnFilterFile),
m_FileExecuteEventHandler(this, &WavePreviewPanel::OnExecuteFile)
{
}

WavePreviewPanel::~WavePreviewPanel()
{
}

void WavePreviewPanel::Initialize(ut::IAllocator* allocator, nw::internal::dw::FileBrowser& fileBrowser)
{
    NW_ASSERT_NOT_NULL(allocator);

    m_ReadFileBuffer = allocator->Alloc(READ_BUFFER_LENGTH);
    m_Allocator = allocator;

    NW_ASSERT_NOT_NULL(m_ReadFileBuffer);

    SetMargin(nw::internal::dw::Thickness(0.f));

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    nw::internal::dw::UIElementList& containerItems = m_Container.GetContents();
    containerItems.AddItem(&m_CurrentDirectoryLabel);
    containerItems.AddItem(&m_LampBar);
    containerItems.AddItem(&fileBrowser);

    m_Container.SetMargin(nw::internal::dw::Thickness(0.f));

    m_CurrentDirectoryLabel.SetMargin(nw::internal::dw::Thickness(0.f));
    m_CurrentDirectoryLabel.SetDock(nw::internal::dw::DOCK_TOP);
    m_CurrentDirectoryLabel.SetBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f)));

    m_LampBar.SetOrientation(nw::internal::dw::HORIZONTAL);
    m_LampBar.SetDock(nw::internal::dw::DOCK_BOTTOM);
    m_LampBar.SetMargin(nw::internal::dw::Thickness(0.f));
    m_LampBar.SetBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f)));

    fileBrowser.SetMargin(nw::internal::dw::Thickness(2.f, 1.f));
    fileBrowser.SetFilterEventHandler(&m_FileFilterEventHandler);
    fileBrowser.SetExecuteEventHandler(&m_FileExecuteEventHandler);

    InitializeLampBar(m_LampBar);

    m_pFileBrowser = &fileBrowser;
}

void WavePreviewPanel::Finalize()
{
    CloseWaveFile();

    if(m_Allocator != NULL)
    {
        m_Allocator->Free(m_ReadFileBuffer);
        m_Allocator = NULL;
        m_ReadFileBuffer = NULL;
    }
}

bool WavePreviewPanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_B))
    {
        CloseWaveFile();
        return true;
    }

    return false;
}

void WavePreviewPanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    if(m_ASyncWaveReader.IsOpened() &&
        m_ASyncWaveReader.IsFinished() &&
        GetWaveRenderer().IsFinished())
    {
        CloseWaveFile();
    }

    m_CurrentDirectoryLabel.SetText(m_pFileBrowser->GetCurrentDirectoryPath());
}

bool WavePreviewPanel::OpenWaveFile(const char* pPath)
{
    NW_NULL_ASSERT(pPath);
    NW_ASSERT(m_pCurrentReader == NULL);
    NW_ASSERT(!m_WavReader.IsOpened());
    NW_ASSERT(!m_AiffReader.IsOpened());

    // Wav / Aiff ファイルを開きます。
    internal::fnd::FileStream* fileStream = OpenFileImpl(pPath);

    if(fileStream == NULL)
    {
        return false;
    }

    fileStream->EnableCache(m_ReadFileBuffer, READ_BUFFER_LENGTH);

    if(m_WavReader.Open(fileStream).IsFailed() ||
        m_WavReader.ReadHeader().IsFailed())
    {
        fileStream->Seek(0, internal::fnd::Stream::SEEK_ORIGIN_BEGIN);

        if(m_AiffReader.Open(fileStream).IsFailed() ||
            m_AiffReader.ReadHeader().IsFailed())
        {
            CloseWaveFile();
            return false;
        }

        m_pCurrentReader = &m_AiffReader;
    }
    else
    {
        m_pCurrentReader = &m_WavReader;
    }

    NW_NULL_ASSERT(m_pCurrentReader);

    // 出力ラインを設定します。
    SetOutputLine(GetWaveRenderer());

    // WaveRenderer を開きます。
    if(!GetWaveRenderer().Open(m_pCurrentReader->GetWaveFormat(), m_pCurrentReader->GetDataSize()))
    {
        CloseWaveFile();
        return false;
    }

    // データの読み込みは非同期で処理します。
    if(!m_ASyncWaveReader.Open(m_pCurrentReader, &GetWaveRenderer()))
    {
        CloseWaveFile();
        return false;
    }

    return true;
}

void WavePreviewPanel::CloseWaveFile()
{
    if(m_ASyncWaveReader.IsOpened())
    {
        m_ASyncWaveReader.Close();
    }

    if(GetWaveRenderer().IsOpened())
    {
        GetWaveRenderer().Close();
    }

    if(m_WavReader.IsOpened())
    {
        m_WavReader.Close();
    }

    if(m_AiffReader.IsOpened())
    {
        m_AiffReader.Close();
    }

    m_pCurrentReader = NULL;

    internal::fnd::FileStream* fileStream = GetOpenedFileStream();

    if(fileStream != NULL)
    {
        fileStream->Close();
    }
}

void WavePreviewPanel::OnFilterFile(nw::internal::dw::FileBrowserFilterEventArgs& args)
{
    const char* pName = args.GetName();
    NW_NULL_ASSERT(pName);

    u32 length = strlen(pName);

    // ASCII 範囲外を含む場合は除外します。
    for(u32 i = 0; i < length; ++i)
    {
        if(static_cast<u8>(pName[i]) > 0x7f)
        {
            args.SetIsCanceled(true);
            return;
        }
    }

    // ディレクトリは表示します。
    if(args.GetIsDirectory())
    {
        return;
    }

    // .wav / .aif / .aiff 以外のファイルを除外します。
    static const char* pWavExtention = ".wav";
    static const char* pAifExtention = ".aif";
    static const char* pAiffExtention = ".aiff";
    static const u32 wavAifExtentionLength = strlen(pWavExtention);
    static const u32 aiffExtentionLength = strlen(pAiffExtention);

    if(length < wavAifExtentionLength)
    {
        args.SetIsCanceled(true);
        return;
    }

    if(nw::ut::stricmp(pWavExtention, pName + length - wavAifExtentionLength) == 0)
    {
        return;
    }

    if(nw::ut::stricmp(pAifExtention, pName + length - wavAifExtentionLength) == 0)
    {
        return;
    }

    if(length < aiffExtentionLength)
    {
        args.SetIsCanceled(true);
        return;
    }

    if(nw::ut::stricmp(pAiffExtention, pName + length - aiffExtentionLength) == 0)
    {
        return;
    }

    args.SetIsCanceled(true);
}

void WavePreviewPanel::OnExecuteFile(nw::internal::dw::FileBrowserExecuteEventArgs& args)
{
    CloseWaveFile();
    OpenWaveFile(args.GetFilePath());
}

} // snd
} // nw
