﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewEffectSet.h>

namespace nw {
namespace snd {

PreviewEffectSet::PreviewEffectSet() :
m_IsDirty(false),
m_pCurrent(&m_None),
m_pAssignedEffect(NULL)
{
}

bool PreviewEffectSet::GetIsDirty() const
{
    if(m_IsDirty)
    {
        return true;
    }

    return m_pCurrent->GetIsDirty();
}

void PreviewEffectSet::ResetIsDirty()
{
    m_IsDirty = false;

    m_ReverbHi.ResetIsDirty();
    m_ReverbStd.ResetIsDirty();
    m_Chorus.ResetIsDirty();
    m_Delay.ResetIsDirty();
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    m_MultiChReverb.ResetIsDirty();
    m_MultiChChorus.ResetIsDirty();
    m_MultiChDelay.ResetIsDirty();
#endif
}

IPreviewEffect& PreviewEffectSet::GetCurrent() const
{
    NW_NULL_ASSERT(m_pCurrent);
    return *m_pCurrent;
}

void PreviewEffectSet::SetCurrent(PreviewEffectType previewEffectType)
{
    IPreviewEffect* pNewCurrent = NULL;

    switch(previewEffectType)
    {
    case EFFECT_NONE:
        pNewCurrent = &m_None;
        break;

    case EFFECT_REVERB_HI:
        pNewCurrent = &m_ReverbHi;
        break;

    case EFFECT_REVERB_STD:
        pNewCurrent = &m_ReverbStd;
        break;

    case EFFECT_CHORUS:
        pNewCurrent = &m_Chorus;
        break;

    case EFFECT_DELAY:
        pNewCurrent = &m_Delay;
        break;

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    case EFFECT_MULTICH_REVERB:
        pNewCurrent = &m_MultiChReverb;
        break;

    case EFFECT_MULTICH_CHORUS:
        pNewCurrent = &m_MultiChChorus;
        break;

    case EFFECT_MULTICH_DELAY:
        pNewCurrent = &m_MultiChDelay;
        break;
#endif

    default:
        NW_FATAL_ERROR("[PreviewEffectSet] invalid PreviewEffectType.\n");
        break;
    }

    if(pNewCurrent == m_pCurrent)
    {
        return;
    }

    m_pCurrent = pNewCurrent;
    m_IsDirty = true;
}

u32 PreviewEffectSet::GetCurrentRequiredMemorySize() const
{
    return m_pCurrent->GetRequiredMemorySize();
}

IPreviewEffect* PreviewEffectSet::GetAssignedEffect() const
{
    return m_pAssignedEffect;
}

bool PreviewEffectSet::AssignBuffer(void* pBuffer, u32 bufferSize)
{
    ReleaseBuffer();

    if(GetCurrentRequiredMemorySize() == 0)
    {
        return true;
    }

    NW_NULL_ASSERT(pBuffer);
    NW_ASSERT(bufferSize >= GetCurrentRequiredMemorySize());

    if(!m_pCurrent->AssignBuffer(pBuffer, bufferSize))
    {
        return false;
    }

    m_pAssignedEffect = m_pCurrent;

    return true;
}

void PreviewEffectSet::ReleaseBuffer()
{
    if(m_pAssignedEffect == NULL)
    {
        return;
    }

    m_pAssignedEffect->ReleaseBuffer();
    m_pAssignedEffect = NULL;
}

void PreviewEffectSet::SetChannelMode(FxBase::ChannelMode mode)
{
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    m_MultiChReverb.SetChannelMode(mode);
    m_MultiChChorus.SetChannelMode(mode);
    m_MultiChDelay.SetChannelMode(mode);
#endif
}

} // nw::snd
} // nw
