﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <mcs/HioChannel.h>
#include <mcs/HioProtocol.h>

#if !defined(NW_SND_EDIT_USE_MCS)
namespace {
    const u32 DEFAULT_WORK_BUFFER_LENGTH =  1 * 1024;
}
#endif

namespace nw {
namespace snd {

//----------------------------------------------------------
HioChannel::HioChannel() :
m_Buffer(NULL),
m_BufferSize(0),
m_Reference(*this)
{
}

//----------------------------------------------------------
HioChannel::~HioChannel()
{
    Finalize();
}

//----------------------------------------------------------
bool
HioChannel::Initialize(nw::ut::IAllocator& allocator, u32 bufferSize)
{
    if(IsInitialized())
    {
        if(m_Allocator == &allocator && m_BufferSize == bufferSize)
        {
            return true;
        }

        Finalize();
    }

    if(bufferSize == 0)
    {
        NW_WARNING(false, "[nw::snd] buffer size must not be 0.");
        return false;
    }

    m_Buffer = allocator.Alloc(bufferSize);

    if(m_Buffer == NULL)
    {
        NW_WARNING(false, "[nw::snd] out of memory.");
        return false;
    }

    m_Allocator = &allocator;
    m_BufferSize = bufferSize;

#if !defined(NW_SND_EDIT_USE_MCS)
    m_WorkBufferSize = DEFAULT_WORK_BUFFER_LENGTH;
    m_WorkBuffer = allocator.Alloc(m_WorkBufferSize);

    if(m_WorkBuffer == NULL)
    {
        NW_WARNING(false, "[nw::snd] out of memory.");
        return false;
    }

    m_Stream.Initialize(m_WorkBuffer, m_WorkBufferSize);
#endif

    return true;
}

//----------------------------------------------------------
void
HioChannel::Finalize()
{
    Close();

#if !defined(NW_SND_EDIT_USE_MCS)
    m_Stream.Finalize();

    if(m_WorkBuffer != NULL)
    {
        NW_NULL_ASSERT(m_Allocator);
        nw::ut::SafeFree(m_WorkBuffer, m_Allocator);
    }

    m_WorkBufferSize = 0;
#endif

    if(m_Buffer != NULL)
    {
        NW_NULL_ASSERT(m_Allocator);
        nw::ut::SafeFree(m_Buffer, m_Allocator);
        m_Buffer = NULL;
    }

    m_BufferSize = 0;
    m_Allocator = NULL;
}

//----------------------------------------------------------
bool
HioChannel::Open(ChannelType channel)
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_Stream.Open(channel, m_Buffer, m_BufferSize);
}

//----------------------------------------------------------
void
HioChannel::Close()
{
    if(!IsInitialized())
    {
        return;
    }

    if(m_Stream.IsAvailable())
    {
        m_Stream.Close();
    }
}

//----------------------------------------------------------
void
HioChannel::RegisterMessageHandler(HioMessageHandler& handler)
{
    m_MessageHandlerRefs.push_back(&handler.GetReference());
}

//----------------------------------------------------------
void
HioChannel::UnregisterMessageHandler(HioMessageHandler& handler)
{
    // HACK: 多重 erase しないようにするための保護処理
    if(handler.GetReference().node.GetNext() == NULL &&
        handler.GetReference().node.GetPrev() == NULL)
    {
        return;
    }

    m_MessageHandlerRefs.erase(&handler.GetReference());
}

//----------------------------------------------------------
void
HioChannel::Update()
{
    if(!IsInitialized())
    {
        return;
    }

    if(!m_Stream.IsAvailable())
    {
        return;
    }

#if defined(NW_DEBUG)
#if defined(NW_PLATFORM_CAFE)
    if(std::strcmp(m_Stream.GetChannel(), "NW_SND_TOOL") == 0)
#else
    if(m_Stream.GetChannel() == HIO_SOUNDPLAYER_PASSIVE_CHANNEL)
#endif
    {
        static u32 lastReadableBytes = 0;
        u32 readableBytes = m_Stream.GetReadableBytes();

        if(readableBytes != lastReadableBytes)
        {
            NW_LOG("[HioPassiveChannel] ReadableBytes = %d\n", readableBytes);
            lastReadableBytes = readableBytes;
        }
    }
#endif

    // メッセージヘッダサイズ分のデータが溜まるまで処理しません。
    if(m_Stream.GetReadableBytes() < sizeof(HioMessageHeader))
    {
        return;
    }

    // メッセージヘッダを読み込みます。
    HioMessageHeader header;
    if(m_Stream.Peek(&header, sizeof(HioMessageHeader)) < sizeof(HioMessageHeader))
    {
        NW_WARNING(false, "failed to Peek().");
        return;
    }

    // メッセージ全体サイズ分のデータが溜まるまで処理しません。
    if(m_Stream.GetReadableBytes() < sizeof(HioMessageHeader) + header.size)
    {
        return;
    }

    HioMessageHandler* handler = GetMessageHandler(header);

#if defined(NW_DEBUG)
#if defined(NW_PLATFORM_CAFE)
    if(std::strcmp(m_Stream.GetChannel(), "NW_SND_TOOL") == 0)
#else
    if(m_Stream.GetChannel() == HIO_SOUNDPLAYER_PASSIVE_CHANNEL)
#endif
    {
        NW_LOG(
            "receive message : %s(0x%08x), Size=%d, BodySize=%d.\n",
            header.message.ToString(),
            u32(header.message),
            sizeof(HioMessageHeader) + header.size,
            static_cast<u32>(header.size));
    }
#endif

    if(handler == NULL)
    {
        NW_WARNING(false, "unknown HIO message. : ID=0x%08x, Size=%d", u32(header.message), u32(header.size));

        // 処理できないメッセージをスキップします。
        if(m_Stream.Skip(sizeof(HioMessageHeader) + header.size) < 0)
        {
            NW_WARNING(false, "failed to Skip().");
        }

        return;
    }

    // メッセージ本体を読み込みます。
    void* dataBuffer = NULL;
    {
        // ヘッダサイズ分をスキップします。
        if(m_Stream.Skip(sizeof(HioMessageHeader)) < 0)
        {
            NW_WARNING(false, "failed to Skip().");
            return;
        }

        if(header.size > 0)
        {
            NW_NULL_ASSERT(m_Allocator);

            dataBuffer = m_Allocator->Alloc(header.size);
            if(dataBuffer == NULL)
            {
                // メモリ不足の場合はスキップします。
                NW_WARNING(false, "out of memory.");
                m_Stream.Skip(header.size);
                return;
            }

            if(m_Stream.Read(dataBuffer, header.size) != header.size)
            {
                NW_WARNING(false, "failed to read data.");
                nw::ut::SafeFree(dataBuffer, m_Allocator);
                return;
            }
        }
    }

    handler->Invoke(header, dataBuffer, m_Stream);

    nw::ut::SafeFree(dataBuffer, m_Allocator);
}

#if !defined(NW_SND_EDIT_USE_MCS)
//----------------------------------------------------------
void
HioChannel::Polling()
{
    if (!m_Stream.Connect())
    {
        m_Stream.Disconnect();
    }
}
#endif

//----------------------------------------------------------
HioMessageHandler*
HioChannel::GetMessageHandler(const HioMessageHeader& header) const
{
    for(HioMessageHandlerRefList::const_iterator it = m_MessageHandlerRefs.begin();
        it != m_MessageHandlerRefs.end();
        ++it)
    {
        if(it->value.CanRespond(header))
        {
            return &it->value;
        }
    }

    return NULL;
}

} // namespace nw::snd
} // namespace nw
