﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include "precompiled.h"

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
  #include <cafe/fs.h>
  #include <cafe/ax.h>
  #include <cafe/pads/wpad/wpad.h>
  #include <cafe/im.h>
  #include <cafe/hio.h>
#else
  #include <winext/cafe/ax.h>
#endif

#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/demo/demo_System.h>
#include <nw/dw/system/dw_FileLoader.h>

#if defined(NW_PLATFORM_CAFE)
  #include <nw/dev/cafe/dev_PadCafe.h>
  #include <nw/dev/cafe/dev_MouseCafe.h>
#else
  #include <nw/dev/win/dev_PadWin.h>
  #include <nw/dev/win/dev_MouseWin.h>
#endif

#if defined(NW_PLATFORM_CAFE)
  #include <cafe/pad.h>
  #include <cafe/pads/kpad/kpad.h>
#else
  #include <winext/cafe/pad.h>
#endif

#include "Application.h"
#include <anim/AnimSoundTargetCafe.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
using namespace nw::internal::winext;
#endif

#if defined( NW_PLATFORM_WIN32 )
  #include <windows.h>
#endif

#if defined( NW_USE_NINTENDO_SDK )
  #include <nn/nn_Windows.h>
#endif

//---------------------------------------------------------------------------
// マクロ

#define HEAP_SIZE (128 * 1024 * 1024)

//---------------------------------------------------------------------------
// プロトタイプ宣言

bool Initialize();
bool InitializeDemo();
bool InitializeAudio();
void Finalize();
void FinalizeDemo();
void FinalizeAudio();
void DoFrame();
void ConnectCallback( s32 channel, s32 reaseon );
void SpeakerSetupCallback( s32 channel, s32 result );
void SpeakerShutdownCallback( s32 channel, s32 result );

//---------------------------------------------------------------------------
// グローバル変数

namespace
{

nw::ut::MemoryAllocator s_Allocator;
#if defined(NW_PLATFORM_CAFE)
nw::demo::DemoSystemDRC* s_pDemo;
#else
nw::demo::DemoSystem* s_pDemo;
#endif
void* s_pMemoryForSoundSystem;

#if defined(NW_PLATFORM_CAFE)
OSRendezvous s_Rendezvous;
FSClient s_FsClient;
FSClient s_FsClientSpy;

const s32 PAD_CHANNEL = 0;
nw::dev::PadCafe *          s_Pad( PAD_CHANNEL ); //!< デバッグ用パッドを管理します。
nw::dev::MouseCafe* s_Mouse;                      //!< マウス入力キャプチャです。

#if defined(NW_SND_EDIT_USE_MCS)
const u32 MCS_INPUT_DEVICE_BUF_SIZE = 64 * 1024;  //!< mcsHID のバッファサイズです。
u8* s_InputDeviceBuf = NULL;                      //!< mcsHID のバッファです。
#endif
#else
const s32 JOYPAD_CHANNEL = 0;
nw::dev::PadWin *           s_Pad( JOYPAD_CHANNEL ); //!< デバッグ用パッドを管理します。
nw::dev::MouseWin *         s_Mouse;                //!<  Mouse を PC で再現します。
#endif

}

#if defined(NW_PLATFORM_CAFE)
static void
StateChangeCallback(
    FSClient* pClient,
    FSVolumeState state,
    void* pContext
)
{
    FSError lastError = FSGetLastError(pClient);
    NW_LOG("Volume state of client 0x%08x changed to %d\n", pClient, state);
    NW_LOG("Last error: %d\n", lastError);
}
#endif

#if defined(NW_USE_NINTENDO_SDK)
static void* FsAllocate(size_t size)
{
    return s_Allocator.Alloc(size);
}

static void FsDeallocate(void* address, size_t size)
{
    NW_UNUSED_VARIABLE(size);
    s_Allocator.Free(address);
}
#endif

//---------------------------------------------------------------------------
//! @brief  メイン関数です。
//---------------------------------------------------------------------------
int
NwDemoMain(int argc, char **argv)
{
    if (argc >= 2)
    {
        s32 num = argc - 1;
        for (s32 i = 0; i < num; ++i)
        {
            u32 port = static_cast<u32>(atoi(argv[1 + i]));
            if (port == 0)
            {
                NW_LOG("%s is invalid argument\n");
            }
            else
            {
                nw::snd::Application::GetInstance().SetCustomPort(i, port);
            }
        }
    }

#if defined( NW_USE_NINTENDO_SDK )
    // nn::fs::SetAllocator() は一度しか呼び出すことはできません。
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
#endif

    if(!Initialize())
    {
        return 1;
    }

    while ( nw::snd::Application::GetInstance().IsAvailable() &&
        !s_pDemo->IsExiting() )
    {
        DoFrame();
    }

    Finalize();

    return 0;
}

//---------------------------------------------------------------------------
//! @brief  フレームを処理します。
//---------------------------------------------------------------------------
void DoFrame()
{

    // mcsHID (マウス入力キャプチャ) の更新処理。
#if defined(NW_PLATFORM_CAFE)
    nw::dev::PadDeviceCafe::GetInstance()->Update();
    nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Update();
#else
    nw::dev::JoyPadDeviceWin::GetInstance()->Update();
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->Update();
#endif

    s_Pad->Update();
    s_Mouse->Update();
    nw::snd::Application::GetInstance().UpdateInputs(s_Pad, s_Mouse);

    nw::snd::Application::GetInstance().Update();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
#if defined(NW_PLATFORM_CAFE)
        s_pDemo->BeginDrawTV();
#endif

        s_pDemo->ClearFrameBuffers();
        nw::snd::Application::GetInstance().Render();

#if defined(NW_PLATFORM_CAFE)
        s_pDemo->EndDrawTV();
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    s_pDemo->SwapBuffer();
    s_pDemo->WaitForVBlank();
}

#if defined(NW_PLATFORM_CAFE)
int InitializeSoundFunc(int intArg, void *ptrArg)
{
    InitializeAudio();

    OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_ALL);

    return 0;
}
#endif

//---------------------------------------------------------------------------
//! @brief  アプリケーションを初期化します。
//---------------------------------------------------------------------------
bool Initialize()
{
#if defined(NW_PLATFORM_CAFE)
    IMDisableDim();

    FSInit();
    FSAddClient(&s_FsClient, FS_RET_NO_ERROR);
    FSStateChangeParams stateChangeParams = {
        .userCallback = StateChangeCallback,
        .userContext  = NULL,
        .ioMsgQueue   = NULL
    };
    FSSetStateChangeNotification(&s_FsClient, &stateChangeParams);

    // Spyのデータ送信を他のFS処理と並行して実行できるようにするため
    // 専用のFSClientを用意します。
    FSAddClient(&s_FsClientSpy, FS_RET_NO_ERROR);
    FSSetStateChangeNotification(&s_FsClientSpy, &stateChangeParams);

    HIOInit();
#elif defined(NW_PLATFORM_WIN32)

    // TODO: [fs]暫定対処なので、devのFS対応が済んだら正式対処をする。
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        char buf2[cFileNameLen];
        if ( GetModuleFileNameA(NULL, buf, cFileNameLen) > 0 &&
            GetFileAttributesA((nw::ut::snprintf(buf2, cFileNameLen, "%s/../Contents", buf), buf2)) == FILE_ATTRIBUTE_DIRECTORY )
        {
            // SoundPlayer.exe と同じパスに Contents ディレクトリがあるなら NintendoSDK と判断します。

            if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
            {
                // CAFE_CONTENTS_DIR が設定されているならそちらを優先します。
                // Visual Studioでのデバッグの場合はプロパティシートで CAFE_CONTENTS_DIR が設定されます。
                SetCurrentDirectoryA(buf);
            }
            else
            {
                // Tools/Audio/SoundPlayer/SoundPlayer.exeが直接起動された場合に、
                // CAFE_CONTENTS_DIR が設定されていなければ、実行ファイルからの相対パスでCAFE_CONTENT_DIRを決定します。
                SetEnvironmentVariableA("CAFE_CONTENT_DIR", buf2);
                SetCurrentDirectoryA(buf2);
            }
        }
        else if ( GetEnvironmentVariableA("NW4F_ROOT", buf, cFileNameLen) > 0 )
        {
            nw::ut::snprintf(buf, cFileNameLen, "%s/DiscRoot", buf );
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
            SetEnvironmentVariableA("CAFE_CONTENT_DIR", buf);
        }
        else
        {
            NW_LOG("cannot read NW4F_ROOT\n");
        }
    }
#endif

#if defined(NW_PLATFORM_CAFE) || defined(NW_PLATFORM_WIN32)
    PADInit();
#endif

    s_Allocator.Initialize( MEMAllocFromDefaultHeap(HEAP_SIZE), HEAP_SIZE );

    if(!InitializeDemo())
    {
        return false;
    }

#if defined(NW_PLATFORM_CAFE)
    for ( int i = 0; i < WPAD_MAX_CONTROLLERS; i++ )
    {
        KPADSetConnectCallback( i, ConnectCallback );
    }
#endif

    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &s_Allocator;
    nw::dev::FileDeviceManager::GetInstance()->Initialize(fileDeviceArg);

#if defined(NW_USE_NINTENDO_SDK)
    nn::fs::MountHostRoot();

    // コンテントディレクトリパスを決定します。
    {
        static const int cFileNameLen = 512;
        nw::ut::FixedSafeString<cFileNameLen> buf;
        nw::ut::FixedSafeString<cFileNameLen> buf2;

        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf.getBuffer(), buf.getBufferSize()) > 0 )
        {
            // CAFE_CONTENTS_DIR が設定されているならそちらを優先します。
            // Visual Studioでのデバッグの場合はプロパティシートで CAFE_CONTENTS_DIR が設定されます。
            SetCurrentDirectoryA(buf.c_str());
            nn::fs::MountHost("content", buf.c_str());
            NW_LOG("Content directory: \"%s\"\n", buf.c_str());
        }
        else if ( GetModuleFileNameA(NULL, buf2.getBuffer(), buf2.getBufferSize()) > 0 &&
            GetFileAttributesA(buf.format("%s/../Contents", buf2.c_str()).c_str()) == FILE_ATTRIBUTE_DIRECTORY )
        {
            // Tools/Audio/SoundPlayer/SoundPlayer.exeが直接起動された場合に、
            // CAFE_CONTENTS_DIR が設定されていなければ、実行ファイルからの相対パスでCAFE_CONTENT_DIRを決定します。
            SetEnvironmentVariableA("CAFE_CONTENT_DIR", buf.c_str());
            SetCurrentDirectoryA(buf.c_str());
            nn::fs::MountHost("content", buf.c_str());
            NW_LOG("Content directory: \"%s\"\n", buf.c_str());
        }
        else
        {
            NW_LOG("cannot find Content directory.\n");
        }
    }
#endif

#if 0
    // SoundThread をコア2 で動作させるコード
    {
        OSInitRendezvous(&s_Rendezvous);

        OSThread* thread = OSGetDefaultThread(2); // coreId
        OSRunThread(
            thread,
            InitializeSoundFunc,
            0,
            NULL);

        OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_SUB2);
    }
#else
    if(!InitializeAudio())
    {
        return false;
    }
#endif

    nw::ut::Rect rect(
        0,
        0,
        static_cast<f32>(s_pDemo->GetWidth()),
        static_cast<f32>(s_pDemo->GetHeight()));

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    bool result = nw::snd::Application::GetInstance().Initialize(
            &s_Allocator,
            rect
            );
#elif defined(NW_PLATFORM_CAFE)
    bool result = nw::snd::Application::GetInstance().Initialize(
        &s_FsClient,
        &s_FsClientSpy,
        &s_Allocator,
        rect
        );
#endif

    if(!result)
    {
        return false;
    }


#if defined(NW_PLATFORM_CAFE)
#if defined(NW_SND_EDIT_USE_MCS)
    // mcsHID (マウス入力キャプチャ)を初期化します。
    nw::mcs::Mcs_Initialize();

    u32 errorCode = nw::mcs::Mcs_Open();
    if(errorCode != nw::mcs::MCS_ERROR_SUCCESS)
    {
        return false;
    }

    nw::mcs::McsHID_Initialize();
    if ( nw::mcs::McsHID_GetRegisteredBuffer() == NULL )
    {
        // 登録済みのバッファがない場合には、バッファを生成して登録します。
        s_InputDeviceBuf = reinterpret_cast<u8*>( s_Allocator.Alloc( MCS_INPUT_DEVICE_BUF_SIZE ) );
        nw::mcs::McsHID_RegisterBuffer( s_InputDeviceBuf, MCS_INPUT_DEVICE_BUF_SIZE );
    }
#endif

    // 入力デバイスを初期化します。
    {
        nw::dev::PadDeviceCafe::GetInstance()->Initialize();
        nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Initialize();
        s_Pad = new( s_Allocator.Alloc( sizeof( nw::dev::PadCafe ) ) ) nw::dev::PadCafe( PAD_CHANNEL );
        s_Mouse = new( s_Allocator.Alloc( sizeof( nw::dev::MouseCafe ) ) ) nw::dev::MouseCafe();
    }

    if(s_Pad == NULL || s_Mouse == NULL)
    {
        NW_FATAL_ERROR("Out of memory.");
        return false;
    }

    {
        s_Pad->SetPadDevice( nw::dev::PadDeviceCafe::GetInstance() );
        s_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceCafe::GetInstance() );
        s_Mouse->SetPointerBound( nw::math::VEC2( 0.f, 0.f ), nw::math::VEC2( 1280.f, 720.f ) );
    }
#else
    // 入力デバイスを初期化します。
    {
        nw::dev::JoyPadDeviceWin::GetInstance()->Initialize();
        nw::dev::KeyboardMouseDeviceWin::GetInstance()->Initialize();
        nw::dev::KeyboardMouseDeviceWin::GetInstance()->SetMainWindowHandle( s_pDemo->GetWindowHandle() );

        s_Pad = new( s_Allocator.Alloc( sizeof( nw::dev::PadWin ) ) ) nw::dev::PadWin( JOYPAD_CHANNEL );
        s_Mouse = new( s_Allocator.Alloc( sizeof( nw::dev::MouseWin ) ) ) nw::dev::MouseWin();
    }

    if(s_Pad == NULL || s_Mouse == NULL)
    {
        NW_FATAL_ERROR("Out of memory.");
        return false;
    }

    {
        s_Pad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
        s_Pad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );
        s_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
    }
#endif

    return true;
}

//---------------------------------------------------------------------------
//! @brief  デモを初期化します。
//---------------------------------------------------------------------------
bool InitializeDemo()
{
#ifdef NW_PLATFORM_CAFE
#if defined(NW_SND_EDIT_USE_MCS)
    // mcs の初期化
    {
        nw::mcs::Mcs_Initialize();

        u32 errorCode = nw::mcs::Mcs_Open();
        NW_ASSERT( errorCode == nw::mcs::MCS_ERROR_SUCCESS );
    }
#endif
#endif

#if defined(NW_PLATFORM_CAFE)
    nw::demo::DemoSystemDRC::CreateArg arg;
#else
    nw::demo::DemoSystem::CreateArg arg;
#endif
    arg.allocator = &s_Allocator;
    arg.waitVBlank = 1;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    arg.width = 1280;
    arg.height = 720;
#endif

#if defined(NW_PLATFORM_CAFE)
    nw::demo::DemoSystemDRC* pDemo =
        new(s_Allocator.Alloc(sizeof(nw::demo::DemoSystemDRC))) nw::demo::DemoSystemDRC(arg);
#else
    nw::demo::DemoSystem* pDemo =
        new(s_Allocator.Alloc(sizeof(nw::demo::DemoSystem))) nw::demo::DemoSystem(arg);
#endif

    if(pDemo == NULL)
    {
        NW_FATAL_ERROR("Out of memory.");
        return false;
    }

    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem(0, NULL);

    s_pDemo = pDemo;

    return true;
}

//---------------------------------------------------------------------------
//! @brief  オーディオを初期化します。
//---------------------------------------------------------------------------
bool InitializeAudio()
{
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_PPC );

    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize(param);
        s_pMemoryForSoundSystem = MEMAllocFromDefaultHeap(workMemSize);

        nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>(s_pMemoryForSoundSystem),
            workMemSize);
    }

    nw::snd::SoundSystem::EnableSeqPrintVar();

    return true;
}

//---------------------------------------------------------------------------
//! @brief  アプリケーションの終了処理を行います。
//---------------------------------------------------------------------------
void Finalize()
{
    nw::snd::Application::GetInstance().Finalize();

    FinalizeAudio();

    FinalizeDemo();

#if defined(NW_PLATFORM_CAFE)
#if defined(NW_SND_EDIT_USE_MCS)
    // mcs の終了処理
    {
        nw::mcs::McsHID_Finalize();
        nw::mcs::Mcs_Finalize();
    }

    // mcsHID (マウス入力キャプチャ)の終了処理
    nw::mcs::McsHID_Finalize();
    nw::mcs::Mcs_Finalize();
#endif

    nw::dev::PadDeviceCafe::GetInstance()->Finalize();
    nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Finalize();
#else
    nw::dev::JoyPadDeviceWin::GetInstance()->Finalize();
    nw::dev::KeyboardMouseDeviceWin::GetInstance()->Finalize();
#endif

    nw::ut::SafeFree(s_Pad, &s_Allocator);
    nw::ut::SafeFree(s_Mouse, &s_Allocator);

#if defined(NW_USE_NINTENDO_SDK)
    nn::fs::UnmountHostRoot();
    nn::fs::Unmount("content");
#endif

    s_Allocator.Finalize();
}

//---------------------------------------------------------------------------
//! @brief  デモの終了処理を行います。
//---------------------------------------------------------------------------
void FinalizeDemo()
{
    NW_ASSERT(s_pDemo != NULL);

    s_pDemo->FinalizeGraphicsSystem();
    s_pDemo->Finalize();

    nw::ut::SafeFree(s_pDemo, &s_Allocator);

    nw::dev::FileDeviceManager::GetInstance()->Finalize();
}

//---------------------------------------------------------------------------
//! @brief  オーディオの終了処理を行います。
//---------------------------------------------------------------------------
void FinalizeAudio()
{
    nw::snd::SoundSystem::Finalize();
}

//---------------------------------------------------------------------------
void SpeakerSetupCallback( s32 channel, s32 result )
{
    NW_LOG( "SpeakerSetupCallback    [channel] %d, [result] %d \n", channel, result );
}

void SpeakerShutdownCallback( s32 channel, s32 result )
{
    NW_LOG( "SpeakerShutdownCallback [channel] %d, [result] %d \n", channel, result );
}

//---------------------------------------------------------------------------
//! @brief  リモコン接続コールバックで SoundSystem のリモコンの初期化・終了処理を行います。
//---------------------------------------------------------------------------
void ConnectCallback( s32 channel, s32 reason )
{
#if defined( NW_PLATFORM_CAFE )
    if ( reason == WPAD_ERR_NONE )
    {
        nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Initialize( SpeakerSetupCallback );
    }
    else if ( reason == WPAD_ERR_NO_CONTROLLER )
    {
        nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Finalize( SpeakerShutdownCallback );
    }
#endif // defined( NW_PLATFORM_CAFE )
}
