﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"
#include <io/WaveformStreamReadTask.h>

#include <nw/ut.h>
#include <nw/snd/fnd/io/sndfnd_WaveformStream.h>
#include <io/IStreamWriter.h>

namespace nw {
namespace snd {

WaveformStreamReadTask::WaveformStreamReadTask() :
m_State(STATE_NOT_READY),
m_pReader(NULL),
m_pWriter(NULL),
m_pBuffer(NULL),
m_BufferMaxFrames(0),
m_BlockAlignment(0)
{
}

WaveformStreamReadTask::~WaveformStreamReadTask()
{
}

WaveformStreamReadTask::State WaveformStreamReadTask::GetState() const
{
    return m_State;
}

bool WaveformStreamReadTask::IsFinished() const
{
    switch(m_State)
    {
    case STATE_DONE:
    case STATE_CANCELED:
    case STATE_ERROR:
        return true;
    }

    return false;
}

void WaveformStreamReadTask::Update(
    internal::fnd::WaveformStreamReader* reader,
    IStreamWriter* writer,
    void* pBuffer,
    u32 size)

{
    NW_NULL_ASSERT(reader);
    NW_NULL_ASSERT(writer);
    NW_NULL_ASSERT(pBuffer);
    NW_ASSERT(reader->IsOpened() && reader->IsHeaderRead());
    NW_ASSERT(size > 0);
    NW_ASSERT(m_State != STATE_READY);

    const internal::fnd::WaveFormat& waveFormat = reader->GetWaveFormat();

    m_pReader = reader;
    m_pWriter = writer;
    m_pBuffer = pBuffer;
    m_BlockAlignment = waveFormat.channels * (waveFormat.bitsPerSample / 8);
    m_BufferMaxFrames = size / m_BlockAlignment;

    m_State = STATE_READY;
}

void WaveformStreamReadTask::Invoke()
{
    NW_ASSERT(m_State == STATE_READY);

    bool isCompleted = false;
    m_State = STATE_RUNNING;

    while(true)
    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_StateLock);

        if(m_State != STATE_RUNNING)
        {
            break;
        }

        internal::fnd::FndResult result(internal::fnd::SNDFND_RESULT_TRUE);
        u32 readFrames = m_pReader->ReadFrames(m_pBuffer, m_BufferMaxFrames, &result);

        if(result.IsFailed())
        {
            m_pWriter->Close();
            m_State = STATE_ERROR;
            break;
        }

        if(readFrames == 0)
        {
            isCompleted = true;
            break;
        }

        m_pWriter->Write(m_pBuffer, readFrames * m_BlockAlignment);
    }

    if(m_State != STATE_ERROR)
    {
        m_pWriter->EndOfStream();

        m_State = isCompleted ? STATE_DONE : STATE_CANCELED;
    }
}

void WaveformStreamReadTask::Reset()
{
    m_State = STATE_CANCELING;

    nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_StateLock);

    m_State = STATE_NOT_READY;
    m_pReader = NULL;
    m_pWriter = NULL;
    m_pBuffer = NULL;
    m_BufferMaxFrames = 0;
}

} // snd
} // nw
