﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewSound.h>
#include <controls/PreviewSoundListItem.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace snd {

const char PreviewSoundListItem::StoppedStateLabel[] = "[STOP]";
const char PreviewSoundListItem::PlayingStateLabel[] = "[PLAY]";
const char PreviewSoundListItem::PausedStateLabel[]  = "[PAUS]";
const nw::ut::Color4f& PreviewSoundListItem::StoppedColor = nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_WHITE_SMOKE());
const nw::ut::Color4f& PreviewSoundListItem::PlayingColor = nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_YELLOW_GREEN());
const nw::ut::Color4f& PreviewSoundListItem::PausedColor  = nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f::X_DARK_GREEN());

PreviewSoundListItem::PreviewSoundListItem() :
m_SoundIndex(nw::snd::SoundArchive::INVALID_ID)
{
    SetIsFocusable(true);
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetPadding(nw::internal::dw::Thickness(1.f, 0.f));

    m_Container.SetOrientation(nw::internal::dw::HORIZONTAL);

    m_Container.GetContents().AddItem(&m_StateLabel);
    m_Container.GetContents().AddItem(&m_IDLabel);
    m_Container.GetContents().AddItem(&m_Label);

    m_StateLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

    m_IDLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

    m_Label.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);
}

void PreviewSoundListItem::Initialize(PreviewSound& previewSound)
{
    m_pPreviewSound = &previewSound;
}

PreviewSound& PreviewSoundListItem::GetPreviewSound()
{
    if(m_pPreviewSound == NULL)
    {
        NW_FATAL_ERROR("PreviewSound is not initialized.");
    }

    return *m_pPreviewSound;
}

bool PreviewSoundListItem::OnUpdateMouse(const nw::math::Vector2& position, const dev::Mouse& mouse)
{
    if(mouse.IsDoubleClick())
    {
        m_pPreviewSound->Play();
        return true;
    }

    return false;
}

bool PreviewSoundListItem::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    NW_ASSERTMSG(m_pPreviewSound != NULL, "PreviewSoundListItem is not initialized.");

    if(pad.IsTrig(nw::dev::Pad::MASK_A))
    {
        m_pPreviewSound->Play();
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_B))
    {
        m_pPreviewSound->Stop();
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_Y))
    {
        m_pPreviewSound->Pause();
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_RIGHT))
    {
        int offset = 1;
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 0x100;
        }

        m_pPreviewSound->Next(offset);
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_LEFT))
    {
        int offset = -1;
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 0x100;
        }

        m_pPreviewSound->Next(offset);
        return true;
    }

    return false;
}

void PreviewSoundListItem::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    UpdateState();
    UpdateIDLabel();
    UpdateLabel();
}

void PreviewSoundListItem::UpdateState()
{
    NW_ASSERTMSG(m_pPreviewSound != NULL, "PreviewSoundListItem is not initialized.");

    switch(m_pPreviewSound->GetState())
    {
    case PreviewSound::STATE_STOPPED:
        m_StateLabel.SetTextColor(StoppedColor);
        m_StateLabel.SetText(StoppedStateLabel);
        m_IDLabel.SetTextColor(StoppedColor);
        m_Label.SetTextColor(StoppedColor);
        break;

    case PreviewSound::STATE_PAUSED:
        m_StateLabel.SetTextColor(PausedColor);
        m_StateLabel.SetText(PausedStateLabel);
        m_IDLabel.SetTextColor(PausedColor);
        m_Label.SetTextColor(PausedColor);
        break;

    case PreviewSound::STATE_PLAYING:
        m_StateLabel.SetTextColor(PlayingColor);
        m_StateLabel.SetText(PlayingStateLabel);
        m_IDLabel.SetTextColor(PlayingColor);
        m_Label.SetTextColor(PlayingColor);
        break;
    }
}

void PreviewSoundListItem::UpdateIDLabel()
{
    NW_ASSERTMSG(m_pPreviewSound != NULL, "PreviewSoundListItem is not initialized.");

    u32 soundIndex = m_pPreviewSound->GetSoundIndex();

    if(m_SoundIndex == soundIndex)
    {
        return;
    }

    m_SoundIndex = soundIndex;

    static const u32 IDLABEL_LENGTH = 8 + 1;
    char idLabel[IDLABEL_LENGTH];

    nw::ut::snprintf(idLabel, IDLABEL_LENGTH, IDLABEL_LENGTH - 1, "%06x", soundIndex);

    m_IDLabel.SetText(idLabel);
}

void PreviewSoundListItem::UpdateLabel()
{
    NW_ASSERTMSG(m_pPreviewSound != NULL, "PreviewSoundListItem is not initialized.");

    const char* pLabel = m_pPreviewSound->GetLabel();
    m_Label.SetText((pLabel == NULL || *pLabel == '\0') ? "NO LABEL" : pLabel);
}

#if defined(NW_DEBUG)

const char* PreviewSoundListItem::ToString() const
{
    ut::snprintf(const_cast<char*>(m_DebugText), sizeof(m_DebugText), sizeof(m_DebugText) - 1, "PreviewSoundListItem(%s)", m_Label.GetText());
    return m_DebugText;
}

#endif

} // snd
} // nw
