﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewSoundArchive.h>
#include <controls/PreviewSequenceListItem.h>

namespace nw {
namespace snd {

PreviewSequenceListItem::PreviewSequenceListItem()
: m_pPreviewSoundArchive(NULL)
, m_SoundId(nw::snd::SoundArchive::INVALID_ID)
, m_LastSoundLabelLength(128)
, m_LastSoundLabel(NULL)
{
    SetIsFocusable(true);
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetPadding(nw::internal::dw::Thickness(1.f, 0.f));

    m_Container.SetOrientation(nw::internal::dw::HORIZONTAL);

    m_Container.GetContents().AddItem(&m_StateLabel);
    m_Container.GetContents().AddItem(&m_IDLabel);
    m_Container.GetContents().AddItem(&m_Label);

    m_StateLabel.SetMargin(nw::internal::dw::Thickness(6.f, 0.f));
    m_IDLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_Label.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

    m_StateLabel.SetText(" ");

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);
}

void PreviewSequenceListItem::Initialize(PreviewSoundArchive& previewSoundArchive)
{
    NW_ASSERT_NOT_NULL(&previewSoundArchive);
    m_pPreviewSoundArchive = &previewSoundArchive;

    AllocSoundLabel(m_LastSoundLabelLength);
    std::memset(m_LastSoundLabel, 0, m_LastSoundLabelLength);
}

void PreviewSequenceListItem::Finalize()
{
    FreeSoundLabel();
}

bool PreviewSequenceListItem::ReadSelectedSoundInfo(nw::snd::SoundArchive::SoundInfo* info)
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    if ( !m_pPreviewSoundArchive->IsOpened() ) { return false; }
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL) { return false; }
    return pSoundArchive->ReadSoundInfo( m_SoundId, info );
}

bool PreviewSequenceListItem::OnUpdateMouse(const nw::math::Vector2& position, const dev::Mouse& mouse)
{
    if(mouse.IsDoubleClick())
    {
        return true;
    }

    return false;
}

bool PreviewSequenceListItem::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_RIGHT))
    {
        IncrSeqSoundId();
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_LEFT))
    {
        DecrSeqSoundId();
        return true;
    }

    return false;
}

void PreviewSequenceListItem::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    UpdateIDLabel();
    UpdateLabel();
}

void PreviewSequenceListItem::UpdateInfo()
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL ) { return; }

    m_SoundId = pSoundArchive->GetItemId(m_LastSoundLabel);

    if (m_SoundId == nw::snd::SoundArchive::INVALID_ID)
    {
        m_SoundId = GetSeqSoundIdMin();
    }

    UpdateBankIds();
}

void PreviewSequenceListItem::UpdateIDLabel()
{
    static const u32 IDLABEL_LENGTH = 11 + 1;
    char idLabel[IDLABEL_LENGTH];

    nw::ut::snprintf(idLabel, IDLABEL_LENGTH, IDLABEL_LENGTH - 1, "%08x", m_SoundId);

    m_IDLabel.SetText(idLabel);
}

void PreviewSequenceListItem::UpdateLabel()
{
    m_Label.SetText(GetSeqSoundString());
}

const char* PreviewSequenceListItem::GetSeqSoundString() const
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL ) { return "do not set SoundArchive"; }
    if ( !m_pPreviewSoundArchive->IsExistSeqSound() ) { return "NO ITEM"; }
    const char* result = pSoundArchive->GetItemLabel( m_SoundId );
    if ( result == NULL ) { return "..."; }
    else { return result; }
}

nw::snd::SoundArchive::ItemId PreviewSequenceListItem::GetSeqSoundIdMin() const
{
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL )
    {
        return nw::snd::SoundArchive::INVALID_ID;
    }

    nw::snd::SoundArchive::ItemId id = nw::snd::SoundArchive::INVALID_ID;
    for ( u32 i = 0; i < pSoundArchive->GetSoundCount(); i++ )
    {
        id = SOUND_ID_MIN + i;
        if ( pSoundArchive->GetSoundType( id ) == nw::snd::SoundArchive::SOUND_TYPE_SEQ )
        {
            return id;
        }
    }

    return nw::snd::SoundArchive::INVALID_ID;
}

nw::snd::SoundArchive::ItemId PreviewSequenceListItem::GetSeqSoundIdMax() const
{
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL )
    {
        return nw::snd::SoundArchive::INVALID_ID;
    }

    nw::snd::SoundArchive::ItemId id = nw::snd::SoundArchive::INVALID_ID;
    for ( int i = pSoundArchive->GetSoundCount() - 1; i >= 0; i-- )
    {
        id = SOUND_ID_MIN + i;
        if ( pSoundArchive->GetSoundType( id ) == nw::snd::SoundArchive::SOUND_TYPE_SEQ )
        {
            return id;
        }
    }

    return nw::snd::SoundArchive::INVALID_ID;
}

void PreviewSequenceListItem::IncrSeqSoundId()
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL ) { return; }
    if ( !m_pPreviewSoundArchive->IsExistSeqSound() ) { return; }

    nw::snd::SoundArchive::ItemId id = m_SoundId;
    bool isFound = false;
    for ( u32 i = (id & 0x00ffffff) + 1; i < pSoundArchive->GetSoundCount(); i++ )
    {
        id = SOUND_ID_MIN + i;
        if ( pSoundArchive->GetSoundType( id ) == nw::snd::SoundArchive::SOUND_TYPE_SEQ )
        {
            isFound = true;
            break;
        }
    }
    if ( isFound == false ) { m_SoundId = GetSeqSoundIdMin(); }
    else                    { m_SoundId = id; }

    UpdateBankIds();
}

void PreviewSequenceListItem::DecrSeqSoundId()
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL ) { return; }
    if ( !m_pPreviewSoundArchive->IsExistSeqSound() ) { return; }

    nw::snd::SoundArchive::ItemId id = m_SoundId;
    bool isFound = false;
    for ( int i = (id & 0x00ffffff) - 1; i >= 0; i-- )
    {
        id = SOUND_ID_MIN + i;
        if ( pSoundArchive->GetSoundType( id ) == nw::snd::SoundArchive::SOUND_TYPE_SEQ )
        {
            isFound = true;
            break;
        }
    }
    if ( isFound == false ) { m_SoundId = GetSeqSoundIdMax(); }
    else                    { m_SoundId = id; }

    UpdateBankIds();
}

void PreviewSequenceListItem::UpdateBankIds()
{
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL ) { return; }

    nw::snd::SoundArchive::SequenceSoundInfo info;
    bool result = pSoundArchive->ReadSequenceSoundInfo( m_SoundId, &info );

    for ( int i = 0; i < nw::snd::SoundArchive::SEQ_BANK_MAX; i++ )
    {
        m_pPreviewSoundArchive->GetPreviewBank(i).SetBankID(info.bankIds[i]);
    }

    RecordSoundLabel();
}

const SoundArchive* PreviewSequenceListItem::GetSoundArchive() const
{
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);

    if (m_pPreviewSoundArchive->IsOpened())
    {
        return &m_pPreviewSoundArchive->GetSoundArchive();
    }
    else
    {
        return NULL;
    }
}

void PreviewSequenceListItem::AllocSoundLabel(u32 length)
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

    m_LastSoundLabel = static_cast<char*>(MEMAllocFromDefaultHeap(sizeof(char) * length));
}

void PreviewSequenceListItem::FreeSoundLabel()
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

    if (m_LastSoundLabel != NULL)
    {
        MEMFreeToDefaultHeap(m_LastSoundLabel);
        m_LastSoundLabel = NULL;
    }
}

void PreviewSequenceListItem::RecordSoundLabel()
{
    NW_ASSERT_NOT_NULL(m_LastSoundLabel);
    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    const SoundArchive* pSoundArchive = GetSoundArchive();
    if ( pSoundArchive == NULL )
    {
        return;
    }

    const char* soundLabel = pSoundArchive->GetItemLabel( m_SoundId );
    if ( soundLabel == NULL )
    {
        return;
    }

    size_t labelLen = std::strlen(soundLabel) + 1;
    bool needAlloc = false;
    if (labelLen > m_LastSoundLabelLength)
    {
        m_LastSoundLabelLength = labelLen * 2;
        needAlloc = true;
    }

    if (needAlloc)
    {
        FreeSoundLabel();
        AllocSoundLabel(m_LastSoundLabelLength);
    }

    std::strncpy(m_LastSoundLabel, soundLabel, labelLen);
}

#if defined(NW_DEBUG)

const char* PreviewSequenceListItem::ToString() const
{
    ut::snprintf(const_cast<char*>(m_DebugText), sizeof(m_DebugText), sizeof(m_DebugText) - 1, "PreviewSequenceListItem(%s)", m_Label.GetText());
    return m_DebugText;
}

#endif

} // snd
} // nw
