﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewBank.h>
#include <controls/PreviewBankListItem.h>

namespace nw {
namespace snd {

PreviewBankListItem::PreviewBankListItem()
: m_pPreviewBank(NULL)
{
    SetIsFocusable(true);
    SetMargin(nw::internal::dw::Thickness(0.f));
    SetPadding(nw::internal::dw::Thickness(1.f, 0.f));

    m_Container.SetOrientation(nw::internal::dw::HORIZONTAL);

    m_Container.GetContents().AddItem(&m_StateLabel);
    m_Container.GetContents().AddItem(&m_IDLabel);
    m_Container.GetContents().AddItem(&m_Label);

    m_StateLabel.SetMargin(nw::internal::dw::Thickness(6.f, 0.f));
    m_IDLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_Label.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);
}

void PreviewBankListItem::Initialize(PreviewBank& previewBank)
{
    m_pPreviewBank = &previewBank;
}

PreviewBank& PreviewBankListItem::GetPreviewBank()
{
    if(m_pPreviewBank == NULL)
    {
        NW_FATAL_ERROR("PreviewBank is not initialized.");
    }

    return *m_pPreviewBank;
}

bool PreviewBankListItem::OnUpdateMouse(const nw::math::Vector2& position, const dev::Mouse& mouse)
{
    if(mouse.IsDoubleClick())
    {
        return true;
    }

    return false;
}

bool PreviewBankListItem::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    NW_ASSERTMSG(m_pPreviewBank != NULL, "PreviewBankListItem is not initialized.");

    if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_RIGHT))
    {
        int offset = 1;
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 0x100;
        }

        m_pPreviewBank->Next(offset);
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT) ||
        pad.IsRepeatAccel(nw::dev::Pad::MASK_LEFT))
    {
        int offset = -1;
        if(pad.IsHold(nw::dev::Pad::MASK_L))
        {
            offset *= 0x100;
        }

        m_pPreviewBank->Next(offset);
        return true;
    }

    return false;
}

void PreviewBankListItem::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    UpdateState();
    UpdateIDLabel();
    UpdateLabel();
}

void PreviewBankListItem::UpdateState()
{
    NW_ASSERTMSG(m_pPreviewBank != NULL, "PreviewBankListItem is not initialized.");

    if ( m_pPreviewBank->GetIsLoaded() )
    {
        m_StateLabel.SetText("#");
    }
    else
    {
        m_StateLabel.SetText(" ");
    }
}

void PreviewBankListItem::UpdateIDLabel()
{
    NW_ASSERTMSG(m_pPreviewBank != NULL, "PreviewBankListItem is not initialized.");

    u32 bankId = m_pPreviewBank->GetBankID();
    static const u32 IDLABEL_LENGTH = 8 + 1;
    char idLabel[IDLABEL_LENGTH];

    nw::ut::snprintf(idLabel, IDLABEL_LENGTH, IDLABEL_LENGTH - 1, "%08x", bankId);

    m_IDLabel.SetText(idLabel);
}

void PreviewBankListItem::UpdateLabel()
{
    NW_ASSERTMSG(m_pPreviewBank != NULL, "PreviewBankListItem is not initialized.");

    const char* pLabel = m_pPreviewBank->GetLabel();
    m_Label.SetText((pLabel == NULL || *pLabel == '\0') ? "NO LABEL" : pLabel);
}

#if defined(NW_DEBUG)

const char* PreviewBankListItem::ToString() const
{
    ut::snprintf(const_cast<char*>(m_DebugText), sizeof(m_DebugText), sizeof(m_DebugText) - 1, "PreviewBankListItem(%s)", m_Label.GetText());
    return m_DebugText;
}

#endif

} // snd
} // nw
