﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <controls/EffectParametersView.h>
#include <models/PreviewEffectUtility.h>

namespace nw {
namespace snd {

EffectParametersView::EffectParametersView() :
m_pEffectSet(NULL),
m_AusBusIndex(0),
m_EffectIndex(0)

{
    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_Container.GetContents().AddItem(&m_EffectLabel);
    m_Container.GetContents().AddItem(&m_NoParametersLabel);
    m_Container.GetContents().AddItem(&m_ReverbHiParameterListView);
    m_Container.GetContents().AddItem(&m_ReverbStdParameterListView);
    m_Container.GetContents().AddItem(&m_ChorusParameterListView);
    m_Container.GetContents().AddItem(&m_DelayParameterListView);
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    m_Container.GetContents().AddItem(&m_MultiChReverbParameterListView);
    m_Container.GetContents().AddItem(&m_MultiChChorusParameterListView);
    m_Container.GetContents().AddItem(&m_MultiChDelayParameterListView);
#endif

    m_EffectLabel.SetDock(nw::internal::dw::DOCK_TOP);

    m_NoParametersLabel.SetText("No Parameters");
    m_NoParametersLabel.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_NoParametersLabel.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_ReverbHiParameterListView.SetLabelWidth(140.f);
    m_ReverbHiParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_ReverbHiParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_ReverbStdParameterListView.SetLabelWidth(140.f);
    m_ReverbStdParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_ReverbStdParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_ChorusParameterListView.SetLabelWidth(140.f);
    m_ChorusParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_ChorusParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_DelayParameterListView.SetLabelWidth(140.f);
    m_DelayParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_DelayParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    m_MultiChReverbParameterListView.SetLabelWidth(140.f);
    m_MultiChReverbParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_MultiChReverbParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_MultiChChorusParameterListView.SetLabelWidth(140.f);
    m_MultiChChorusParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_MultiChChorusParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);

    m_MultiChDelayParameterListView.SetLabelWidth(140.f);
    m_MultiChDelayParameterListView.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_STRETCH);
    m_MultiChDelayParameterListView.SetVerticalAlignment(nw::internal::dw::VERTICAL_STRETCH);
#endif
}

void EffectParametersView::SetModel(PreviewEffectSet* pEffectSet, s32 ausBusIndex, s32 effectIndex)
{
    m_pEffectSet = pEffectSet;
    m_AusBusIndex = ausBusIndex;
    m_EffectIndex = effectIndex;

    if(pEffectSet == NULL)
    {
        m_ReverbHiParameterListView.SetModel(NULL);
    }
}

void EffectParametersView::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    m_NoParametersLabel.SetVisibility(nw::internal::dw::COLLAPSED);
    m_ReverbHiParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
    m_ReverbStdParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
    m_ChorusParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
    m_DelayParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    m_MultiChReverbParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
    m_MultiChChorusParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
    m_MultiChDelayParameterListView.SetVisibility(nw::internal::dw::COLLAPSED);
#endif

    if(m_pEffectSet == NULL)
    {
        m_EffectLabel.SetText("");
        m_NoParametersLabel.SetVisibility(nw::internal::dw::VISIBLE);
        return;
    }

    switch(m_pEffectSet->GetCurrent().GetEffectType())
    {
    case EFFECT_NONE:
        m_NoParametersLabel.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_REVERB_HI:
        m_ReverbHiParameterListView.SetModel(
            reinterpret_cast<PreviewEffectReverbHi*>(&m_pEffectSet->GetCurrent()));
        m_ReverbHiParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_REVERB_STD:
        m_ReverbStdParameterListView.SetModel(
            reinterpret_cast<PreviewEffectReverbStd*>(&m_pEffectSet->GetCurrent()));
        m_ReverbStdParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_CHORUS:
        m_ChorusParameterListView.SetModel(
            reinterpret_cast<PreviewEffectChorus*>(&m_pEffectSet->GetCurrent()));
        m_ChorusParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_DELAY:
        m_DelayParameterListView.SetModel(
            reinterpret_cast<PreviewEffectDelay*>(&m_pEffectSet->GetCurrent()));
        m_DelayParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    case EFFECT_MULTICH_REVERB:
        m_MultiChReverbParameterListView.SetModel(
            reinterpret_cast<PreviewEffectMultiChReverb*>(&m_pEffectSet->GetCurrent()));
        m_MultiChReverbParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_MULTICH_CHORUS:
        m_MultiChChorusParameterListView.SetModel(
            reinterpret_cast<PreviewEffectMultiChChorus*>(&m_pEffectSet->GetCurrent()));
        m_MultiChChorusParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;

    case EFFECT_MULTICH_DELAY:
        m_MultiChDelayParameterListView.SetModel(
            reinterpret_cast<PreviewEffectMultiChDelay*>(&m_pEffectSet->GetCurrent()));
        m_MultiChDelayParameterListView.SetVisibility(nw::internal::dw::VISIBLE);
        break;
#endif
    }

    char effectLabelText[EFFECT_LABEL_SIZE];
    nw::ut::snprintf(
        effectLabelText,
        EFFECT_LABEL_SIZE,
        EFFECT_LABEL_SIZE - 1,
        "Aux%c-%d : %s",
        'A' + m_AusBusIndex,
        m_EffectIndex,
        PreviewEffectUtility::ToString(m_pEffectSet->GetCurrent().GetEffectType()));

    m_EffectLabel.SetText(effectLabelText);
}

nw::internal::dw::UIElement* EffectParametersView::GetParametersView()
{
    switch(m_pEffectSet->GetCurrent().GetEffectType())
    {
    case EFFECT_NONE:
        return &m_NoParametersLabel;

    case EFFECT_REVERB_HI:
        return &m_ReverbHiParameterListView;

    case EFFECT_REVERB_STD:
        return &m_ReverbStdParameterListView;

    case EFFECT_CHORUS:
        return &m_ChorusParameterListView;

    case EFFECT_DELAY:
        return &m_DelayParameterListView;

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    case EFFECT_MULTICH_REVERB:
        return &m_MultiChReverbParameterListView;

    case EFFECT_MULTICH_CHORUS:
        return &m_MultiChChorusParameterListView;

    case EFFECT_MULTICH_DELAY:
        return &m_MultiChDelayParameterListView;
#endif
    }

    return NULL;
}

} // snd
} // nw
