﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <anim/AxisDrawer.h>
#include <nw/gfnd/gfnd_Graphics.h>

namespace nw    {
namespace snd   {

/* static */ nw::dev::DrawBuffer AxisDrawer::m_AxisXBuffer;
/* static */ nw::dev::DrawBuffer AxisDrawer::m_AxisYBuffer;
/* static */ nw::dev::DrawBuffer AxisDrawer::m_AxisZBuffer;
/* static */ int AxisDrawer::s_InitializedCount = 0;
/* static */ nw::ut::IAllocator* AxisDrawer::s_CacheAllocator = NULL;

//---------------------------------------------------------------------------
/* static */ AxisDrawer*
AxisDrawer::Create( nw::ut::IAllocator* allocator )
{
    NW_ASSERT_NOT_NULL( allocator );

    AxisDrawer* drawer = new( allocator->Alloc( sizeof( AxisDrawer ) ) ) AxisDrawer( allocator );

    if ( s_InitializedCount == 0 )
    {
        AxisDrawer::CreateCache( allocator );
    }

    ++s_InitializedCount;

    return drawer;
}

//---------------------------------------------------------------------------
void
AxisDrawer::Destroy()
{
    --s_InitializedCount;

    if ( s_InitializedCount == 0 )
    {
        AxisDrawer::DestroyCache();
    }

    nw::ut::IAllocator* allocator = this->m_Allocator;
    this->~AxisDrawer();
    allocator->Free( this );
}

//---------------------------------------------------------------------------
void
AxisDrawer::Draw() const
{
    if ( m_DrawStatus == DRAW )
    {
        NW_ASSERT_NOT_NULL( m_ProjMtx );
        NW_ASSERT_NOT_NULL( m_ViewMtx );

        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();

        renderer->SetProjectionMtx( m_ProjMtx );
        renderer->SetViewMtx( m_ViewMtx );

        nw::math::MTX34 translateMtx;
        translateMtx.SetScale( nw::math::VEC3( m_AxisLength, m_AxisLength, m_AxisLength ) );

        nw::math::MTX34 resultMtx;
        nw::math::MTX34Mult( &resultMtx, m_ModelMtx, translateMtx );
        renderer->SetModelMatrix( resultMtx );

        m_GfxContext.Apply();

        renderer->Begin();
        {
            renderer->DrawLine( m_AxisXColor, m_AxisXBuffer, m_AxisLineWidth );
            renderer->DrawLine( m_AxisYColor, m_AxisYBuffer, m_AxisLineWidth );
            renderer->DrawLine( m_AxisZColor, m_AxisZBuffer, m_AxisLineWidth );
        }
        renderer->End();
    }
}

//---------------------------------------------------------------------------
/* static */ void
AxisDrawer::CreateCache( nw::ut::IAllocator* allocator )
{
    s_CacheAllocator = allocator;

    // 軸描画用バッファを作成します。
    {
        m_AxisXBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator->Alloc( sizeof( nw::dev::Vertex ) * 2, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_AxisXBuffer.index  = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 2, GX2_INDEX_BUFFER_ALIGNMENT ) );
        m_AxisYBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator->Alloc( sizeof( nw::dev::Vertex ) * 2, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_AxisYBuffer.index  = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 2, GX2_INDEX_BUFFER_ALIGNMENT ) );
        m_AxisZBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator->Alloc( sizeof( nw::dev::Vertex ) * 2, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_AxisZBuffer.index  = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 2, GX2_INDEX_BUFFER_ALIGNMENT ) );

        {
            m_AxisXBuffer.vertex[0].pos.Set( 0.f, 0.f, 0.f );
            m_AxisXBuffer.vertex[0].uv.Set( 0.f, 0.5f );
            m_AxisXBuffer.vertex[0].color.Set( 0.f, 0.f, 0.f, 0.f );
            m_AxisXBuffer.vertex[1].pos.Set( 1.f, 0.f, 0.f );
            m_AxisXBuffer.vertex[1].uv.Set( 1.f, 0.5f );
            m_AxisXBuffer.vertex[1].color.Set( 1.f, 0.f, 0.f, 0.f );

            m_AxisYBuffer.vertex[0].pos.Set( 0.f, 0.f, 0.f );
            m_AxisYBuffer.vertex[0].uv.Set( 0.f, 0.5f );
            m_AxisYBuffer.vertex[0].color.Set( 0.f, 0.f, 0.f, 0.f );
            m_AxisYBuffer.vertex[1].pos.Set( 0.f, 1.f, 0.f );
            m_AxisYBuffer.vertex[1].uv.Set( 1.f, 0.5f );
            m_AxisYBuffer.vertex[1].color.Set( 1.f, 0.f, 0.f, 0.f );

            m_AxisZBuffer.vertex[0].pos.Set( 0.f, 0.f, 0.f );
            m_AxisZBuffer.vertex[0].uv.Set( 0.f, 0.5f );
            m_AxisZBuffer.vertex[0].color.Set( 0.f, 0.f, 0.f, 0.f );
            m_AxisZBuffer.vertex[1].pos.Set( 0.f, 0.f, 1.f );
            m_AxisZBuffer.vertex[1].uv.Set( 1.f, 0.5f );
            m_AxisZBuffer.vertex[1].color.Set( 1.f, 0.f, 0.f, 0.f );
        }

        {
            m_AxisXBuffer.index[0] = 0;
            m_AxisXBuffer.index[1] = 1;
            m_AxisYBuffer.index[0] = 0;
            m_AxisYBuffer.index[1] = 1;
            m_AxisZBuffer.index[0] = 0;
            m_AxisZBuffer.index[1] = 1;
        }

        m_AxisXBuffer.vertexNum = 2;
        m_AxisXBuffer.indexNum  = 2;
        m_AxisYBuffer.vertexNum = 2;
        m_AxisYBuffer.indexNum  = 2;
        m_AxisZBuffer.vertexNum = 2;
        m_AxisZBuffer.indexNum  = 2;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        nw::dev::PrimitiveRenderer::GetInstance()->CreateVertexBuffer( m_AxisXBuffer );
        nw::dev::PrimitiveRenderer::GetInstance()->CreateVertexBuffer( m_AxisYBuffer );
        nw::dev::PrimitiveRenderer::GetInstance()->CreateVertexBuffer( m_AxisZBuffer );
#else
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisXBuffer.vertex, sizeof( nw::dev::Vertex ) * 2 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisXBuffer.index, sizeof( u16 ) * 2 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisYBuffer.vertex, sizeof( nw::dev::Vertex ) * 2 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisYBuffer.index, sizeof( u16 ) * 2 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisZBuffer.vertex, sizeof( nw::dev::Vertex ) * 2 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_AxisZBuffer.index, sizeof( u16 ) * 2 );
#endif
    }
}

//---------------------------------------------------------------------------
/* static */ void
AxisDrawer::DestroyCache()
{
    nw::ut::SafeFree( m_AxisXBuffer.vertex, s_CacheAllocator );
    nw::ut::SafeFree( m_AxisXBuffer.index, s_CacheAllocator );
    nw::ut::SafeFree( m_AxisYBuffer.vertex, s_CacheAllocator );
    nw::ut::SafeFree( m_AxisYBuffer.index, s_CacheAllocator );
    nw::ut::SafeFree( m_AxisZBuffer.vertex, s_CacheAllocator );
    nw::ut::SafeFree( m_AxisZBuffer.index, s_CacheAllocator );

    s_CacheAllocator = NULL;
}

} // namespace snd
} // namespace nw
