﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/font.h>
#if defined(NW_ENABLE_SNDEDIT)
#include <nw/sndedit.h>
#endif
#if defined(NW_ENABLE_SNDCTRL)
#include <nw/sndctrl.h>
#endif
#if defined(NW_SND_CONFIG_ENABLE_DEV)
#include <nw/snd/spy/fnd/hio/sndspyfnd_HioInitializer.h>
#include <nw/snd/spy/sndspy_SpyController.h>
#include <nw/snd/spy/sndspy_NwSndSpy.h>
#endif
#include <nw/snd/fnd/io/sndfnd_File.h>
#include <nw/dw/dw_Controls.h>
#include <resources/GlobalResourceProvider.h>
#include <models/PreviewSoundArchive.h>
#include <models/PreviewEffectManager.h>
#include <mcs/HioManager.h>
#include <mcs/handler/ConnectHandler.h>
#include <mcs/handler/OpenSoundArchiveHandler.h>
#include <mcs/handler/CloseSoundArchiveHandler.h>
#include <mcs/handler/EchoHandler.h>
#include <mcs/handler/StopAllHandler.h>
#include <mcs/handler/SetOutputModeHandler.h>
#include <mcs/handler/SelectSoundsHandler.h>
#include <mcs/handler/GetSoundInformationsHandler.h>
#include <mcs/handler/PlaySoundHandler.h>
#include <mcs/handler/PauseSoundHandler.h>
#include <mcs/handler/StopSoundHandler.h>
#include <mcs/handler/SetSoundParametersHandler.h>
#include <mcs/handler/SetEffectParametersHandler.h>
#include <mcs/handler/Load3DModelHandler.h>
#include <mcs/handler/Load3DAnimationHandler.h>
#include <mcs/handler/LoadAnimSoundHandler.h>
#include <mcs/handler/PlayAnimationHandler.h>
#include <mcs/handler/StopAnimationHandler.h>
#include <mcs/handler/GetAnimationFrameHandler.h>
#include <mcs/handler/SetAnimationFrameHandler.h>
#include <mcs/handler/SetAnimationRateHandler.h>
#include <mcs/handler/SetAnimationLoopHandler.h>
#include <mcs/handler/SetAnimationRangeHandler.h>
#include <mcs/handler/EditAnimSoundEventHandler.h>
#include <mcs/handler/PingHandler.h>
#include <mcs/handler/UpdateAllEventIDHandler.h>
#include <mcs/handler/SetSequenceVariablesHandler.h>
#include <mcs/handler/GetSequenceVariablesHandler.h>
#include <PanelManager.h>
#include <anim/AnimSoundController.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief SoundPlayer アプリケーションクラスです。
//---------------------------------------------------------------------------
class Application
{
private:
    typedef u32 PORT;

private:
    static const u32   MAX_FILE_PATH = 256;
    static const char* FONT_BINARY_FILENAME;
#if defined( NW_PLATFORM_CAFE )
    static const char* FONT_SHADER_FILENAME;
    static const char* PRIMITIVE_RENDERER_SHADER_FILENAME;
#endif

    static const u32   FONT_SHADER_BINARY_ALIGNMENT = 128;
    static const u32   FONT_DATA_ALIGNMENT = nw::font::RESOURCE_ALIGNMENT;

    static const char  SOUND_ARCHIVE_FILEPATH[];
#if defined( NW_USE_NINTENDO_SDK )
    static const s32   SOUND_HEAP_SIZE = 48 * 1024 * 1024;
#elif defined( NW_PLATFORM_CAFE )
    static const s32   SOUND_HEAP_SIZE = 300 * 1024 * 1024;
#else
    static const s32   SOUND_HEAP_SIZE = 48 * 1024 * 1024;
#endif
    static const s32   SNDEDIT_HEAP_SIZE = 48 * 1024 * 1024;

    static const s32   EFFECT_BUFFER_SIZE = 2 * 1024 * 1024 * PreviewEffectManager::AUX_BUS_COUNT;

    static const u32   CHANNEL_BUFFER_SIZE      = 64 * 1024;
    static const u32   PING_CHANNEL_BUFFER_SIZE = 256;

    static const u32   SOUND_EDIT_UPDATE_CACHES_THREAD_STACK_SIZE = 32 * 1024;

#ifdef NW_SND_CONFIG_ENABLE_DEV
    static const size_t SPY_DATA_BUFFER_LENGTH = 32 * 1024 * 1024;
#endif

    enum PortIndex
    {
        PORT_INDEX_SYNC   = 0,
        PORT_INDEX_FUNC   = 1,
        PORT_INDEX_CTRL   = 2,
        PORT_INDEX_TOOL   = 3,
        PORT_INDEX_VIEWER = 4,
        PORT_INDEX_PING   = 5,
        PORT_INDEX_NUM
    };

private:
    Application();

public:
    static Application& GetInstance()
    {
        static Application instance;
        return instance;
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    bool Initialize(
        nw::ut::IAllocator* pAllocator,
        const nw::ut::Rect& viewport);
#elif defined(NW_PLATFORM_CAFE)
    bool Initialize(
        FSClient* fsClient,
        FSClient* fsClientSpy,
        nw::ut::IAllocator* pAllocator,
        const nw::ut::Rect& viewport);
#endif

    void Finalize();

    bool IsAvailable() const;

    void UpdateInputs(const nw::dev::Pad* pPad, const nw::dev::Mouse* pMouse);
    void Update();
    void Render();

    // 暫定：ハンドラから呼び出すために、アクセサを用意する。
    PreviewSoundArchive& GetSoundArchive() { return m_PreviewSoundArchive; }
    AnimSoundController& GetAnimSoundCtrl() { return m_AnimSoundController; }
    PreviewEffectManager& GetPreviewEffectManager();
    PanelManager& GetPanelManager() { return m_PanelManager; }

    void SetCustomPort(u32 index, PORT port) { NW_ASSERT(index < PORT_INDEX_NUM); m_Port[index] = port; }

    static void SoundArchiveOpenedCallback();
    void OnSoundArchiveOpened();

private:
    bool InitializePassiveChannel(nw::ut::IAllocator& allocator);
    bool InitializeActiveChannel(nw::ut::IAllocator& allocator);
    bool InitializePingChannel(nw::ut::IAllocator& allocator);
    void InitializeMessageHandlers();
    bool InitializeFont();
    bool InitializeTextWriter();
    bool InitializePrimitiveRenderer();
    bool InitializeUI();
#if defined(NW_SND_CONFIG_ENABLE_DEV)
    bool InitializeSpy(nw::snd::spy::SpyController::OpenArg& arg);
#endif
    bool InitializeSoundArchiveEditor(nw::ut::IAllocator* pAllocator);
    bool InitializeSndEdit(nw::ut::IAllocator* pAllocator);

    void FinalizePassiveChannel();
    void FinalizeActiveChannel();
    void FinalizePingChannel();
    void FinalizeMessageHandlers();
    void FinalizeFont();
    void FinalizeTextWriter();
    void FinalizePrimitiveRenderer();
    void FinalizeUI();
    void FinalizeSpy();
    void FinalizeSoundArchiveEditor();
    void FinalizeSndEdit();

    void UpdateViewport();

    void OnConnected();

    void* LoadFile(
        const char* path,
        u32 alignment = internal::fnd::FileTraits::IO_BUFFER_ALIGNMENT,
        u32* fileSize = NULL);

    HioChannel::ChannelType GetChannelInfo(HioChannelType channel) const;

private:
    nw::ut::IAllocator* m_pAllocator;

#if defined(NW_PLATFORM_CAFE)
    FSClient*   m_FsClient;
    FSCmdBlock* m_FsCmdBlock;
    char        m_FsHfioMountPath[FS_MAX_MOUNTPATH_SIZE];

    FSClient*   m_FsClientSpy;
#endif

    GlobalResourceProvider m_ResourceProvider;

    nw::ut::Rect m_Viewport;
    nw::internal::dw::UIElementTreeContext m_UITreeContext;
    nw::internal::dw::NwUIRenderer  m_UIRenderer;
    nw::internal::dw::NwTextRenderer m_TextRenderer;

    nw::font::ResFont m_Font;
    void* m_pFontBinary;
    void* m_pFontShaderBinary;
    void* m_pPrimitiveRendererShaderBinary;
    u32   m_FontBinarySize;
    u32   m_FontShaderBinarySize;

    PanelManager m_PanelManager;

    PreviewSoundArchive m_PreviewSoundArchive;
    PreviewEffectManager m_PreviewEffectManager;
    PreviewEffectManager m_PreviewEffectManagerDrc;

    PORT m_Port[PORT_INDEX_NUM];

#if defined(NW_ENABLE_SNDEDIT)
    edit::SoundEditSession      m_SoundEditSession;
    edit::SoundArchiveEditor    m_SoundArchiveEditor;
    edit::UpdateCachesHelper    m_SoundEditUpdateCachesHelper;
    void* m_SoundEditSessionBuffer;
    void* m_SoundArchiveEditorBuffer;
    u8* m_SoundEditUpdateCachesThreadStack;
#endif

#if defined(NW_ENABLE_SNDCTRL)
    ctrl::SoundControlSession   m_SoundControlSession;
    ctrl::SoundObjectController m_SoundObjectController;
    void* m_SoundControlSessionBuffer;
#endif

#if defined(NW_SND_CONFIG_ENABLE_DEV)
    nw::snd::spy::SpyController m_SpyController;
    void* m_pSpyBuffer;
    u32 m_AppFrame;
#endif

    HioManager m_HioManager;
    HioChannel m_PassiveChannel;
    HioChannel m_ActiveChannel;
    HioChannel m_PingChannel;

    ConnectHandler::InvokedEventHandler m_ConnectionObserver;

    ConnectHandler                m_ConnectHandler;
    OpenSoundArchiveHandler       m_OpenSoundArchiveHandler;
    CloseSoundArchiveHandler      m_CloseSoundArchiveHandler;
    EchoHandler                   m_EchoHandler;
    StopAllHandler                m_StopAllHandler;
    SetOutputModeHandler          m_SetOutputModeHandler;
    SelectSoundsHandler           m_SelectSoundsHandler;
    GetSoundInformationsHandler   m_GetSoundInformationsHandler;
    PlaySoundHandler              m_PlaySoundHandler;
    PauseSoundHandler             m_PauseSoundHandler;
    StopSoundHandler              m_StopSoundHandler;
    SetSoundParametersHandler     m_SetSoundParametersHandler;
    SetEffectParametersHandler    m_SetEffectParametersHandler;
    Load3DModelHandler            m_Load3DModelHandler;
    Load3DAnimationHandler        m_Load3DAnimationHandler;
    LoadAnimSoundHandler          m_LoadAnimSoundHandler;
    PlayAnimationHandler          m_PlayAnimationHandler;
    StopAnimationHandler          m_StopAnimationHandler;
    GetAnimationFrameHandler      m_GetAnimationFrameHandler;
    SetAnimationFrameHandler      m_SetAnimationFrameHandler;
    SetAnimationRateHandler       m_SetAnimationRateHandler;
    SetAnimationLoopHandler       m_SetAnimationLoopHandler;
    SetAnimationRangeHandler      m_SetAnimationRangeHandler;
    EditAnimSoundEventHandler     m_EditAnimSoundEventHandler;
    PingHandler                   m_PingHandler;
    UpdateAllEventIDHandler       m_UpdateAllEventIDHandler;
    SetSequenceVariablesHandler   m_SetSequenceVariablesHandler;
    GetSequenceVariablesHandler   m_GetSequenceVariablesHandler;

    u8 m_EffectBuffer[EFFECT_BUFFER_SIZE];
    u8 m_EffectBufferDrc[EFFECT_BUFFER_SIZE];

    bool m_IsAnimSound;
    AnimSoundController m_AnimSoundController;
};

} // nw::snd
} // nw
