﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SOUNDPLAYER_HIO_CHANNEL_H_
#define NW_SOUNDPLAYER_HIO_CHANNEL_H_

#if defined(NW_SND_EDIT_USE_MCS)
#include <nw/mcs.h>
#else
#include <nw/snd/fnd/io/sndfnd_HioStream.h>
#endif

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_LinkList.h>
#include <mcs/HioMessageHandler.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    HIOチャンネルを示すクラスです。
//!
//! @details  複数のHIOメッセージハンドラを管理するとともに、
//!           指定チャンネルに流れてきたパケットを適合したハンドラに処理を振り分けます。
//---------------------------------------------------------------------------
class HioChannel
{
public:
#if defined(NW_SND_EDIT_USE_MCS)
    typedef u16 ChannelType;
#else
    typedef nw::snd::internal::fnd::HioStream::ChannelType ChannelType;
#endif

private:
    typedef nw::ut::LinkList<HioMessageHandler::Reference, offsetof(HioMessageHandler::Reference, node)> HioMessageHandlerRefList;

public:
    //! @brief LinkList 用の参照クラスです。
    struct Reference
    {
        Reference(HioChannel& value) : value(value) { }

        HioChannel& value;          //!< HioChannel への参照です。
        nw::ut::LinkListNode node;  //!< LinkListNode です。
    };


public:
    /* ctor */ HioChannel();
    /* dtor */ ~HioChannel();

public:
    //! @brief  初期化します。
    bool Initialize(nw::ut::IAllocator& allocator, u32 bufferSize);

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  初期化の有無を取得します。
    bool IsInitialized() const { return m_Allocator != NULL; }

#if !defined(NW_SND_EDIT_USE_MCS)
    //! @brief  チャンネルが開かれているかどうかを取得します。
    bool IsOpened() const { return m_Stream.IsOpened(); }
#endif

    //! @brief  指定チャンネルを開きます。
    bool Open(ChannelType channel);

    //! @brief  開かれているチャンネルを閉じます。
    void Close();

    //! @brief  HIOメッセージハンドラを登録します。
    void RegisterMessageHandler(HioMessageHandler& handler);

    //! @brief  HIOメッセージハンドラを登録します。
    void UnregisterMessageHandler(HioMessageHandler& handler);

    //! @brief  ストリームを取得します。
    nw::ut::IoStream& GetStream() { return m_Stream; }

    //! @brief  HIOチャンネルを更新します。
    void Update();

#if !defined(NW_SND_EDIT_USE_MCS)
    //! @brief ポーリング処理を実行します。
    void Polling();
#endif

    //---------------------------------------------------------------------------
    //! @brief         LinkList 用の参照を取得します。
    //!
    //! @details       この関数の戻り値である HioChannel::Reference を利用して、
    //!                HioChannel を格納する LinkList を構築できます。
    //!
    //! @return        HioChannel::Reference を返します。
    //---------------------------------------------------------------------------
    Reference& GetReference() { return m_Reference; }

private:
    //! @brief  指定HIOメッセージヘッダに適合するハンドラを取得します。
    HioMessageHandler* GetMessageHandler(const HioMessageHeader& header) const;

private:
#if defined(NW_SND_EDIT_USE_MCS)
    mcs::McsIoStream m_Stream;                      //!< 送受信用ストリームです。
#else
    nw::snd::internal::fnd::HioStream m_Stream;     //!< 送受信用ストリームです。
#endif
    HioMessageHandlerRefList m_MessageHandlerRefs;  //!< HIOメッセージハンドラ参照リストです。

    nw::ut::IAllocator* m_Allocator;                //!< アロケータです。
    void* m_Buffer;                                 //!< 受信用バッファです。
    u32   m_BufferSize;                             //!< 受信用バッファサイズです。
#if !defined(NW_SND_EDIT_USE_MCS)
    void* m_WorkBuffer;                             //!< Hio通信に使用する作業バッファです。
    u32   m_WorkBufferSize;                         //!< Hio通信に使用する作業バッファのサイズです。
#endif

    Reference m_Reference;                          //!< LinkList 用の参照です。
};

} // namespace nw::snd
} // namespace nw

#endif // NW_SOUNDPLAYER_HIO_CHANNEL_H_
