﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <controls/U32ParameterListItem.h>

namespace nw {
namespace snd {

template<class TModel>
U32ParameterListItem<TModel>::U32ParameterListItem() :
m_DefaultValue(0),
m_ValueChange(1),
m_ValueLargeChange(10),
m_MinimumValue(0),
m_MaximumValue(100),
m_pGetValueHandler(NULL),
m_pSetValueHandler(NULL),
m_pPrintValueHandler(NULL)
{
}

template<class TModel>
void U32ParameterListItem<TModel>::Initialize(
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    Initialize(NULL, NULL, pGetValueHandler, pSetValueHandler, pPrintValueHandler);
}

template<class TModel>
void U32ParameterListItem<TModel>::Initialize(
    GetIsCheckedHandler pGetIsCheckedHandler,
    SetIsCheckedHandler pSetIsCheckedHandler,
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    NW_ASSERTMSG(m_pGetValueHandler == NULL, "U32ParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pSetValueHandler == NULL, "U32ParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pPrintValueHandler == NULL, "U32ParameterListItem is already initialized.");
    NW_ASSERT(pGetValueHandler != NULL);
    NW_ASSERT(pSetValueHandler != NULL);
    NW_ASSERT(pPrintValueHandler != NULL);

    m_pGetValueHandler = pGetValueHandler;
    m_pSetValueHandler = pSetValueHandler;
    m_pPrintValueHandler = pPrintValueHandler;

    Base::Initialize(pGetIsCheckedHandler, pSetIsCheckedHandler);
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetValue() const
{
    NW_ASSERTMSG(m_pGetValueHandler != NULL, "U32ParameterListItem is not initialized.");
    return (*m_pGetValueHandler)(GetModel());
}

template<class TModel>
void U32ParameterListItem<TModel>::SetValue(u32 value)
{
    NW_ASSERTMSG(m_pSetValueHandler != NULL, "U32ParameterListItem is not initialized.");
    (*m_pSetValueHandler)(GetModel(), value);
}

template<class TModel>
void U32ParameterListItem<TModel>::ResetValue()
{
    SetValue(m_DefaultValue);
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetDefaultValue() const
{
    return m_DefaultValue;
}

template<class TModel>
void U32ParameterListItem<TModel>::SetDefaultValue(u32 value)
{
    m_DefaultValue = value;
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetValueChange() const
{
    return m_ValueChange;
}

template<class TModel>
void U32ParameterListItem<TModel>::SetValueChange(u32 value)
{
    m_ValueChange = value;
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetValueLargeChange() const
{
    return m_ValueLargeChange;
}

template<class TModel>
void U32ParameterListItem<TModel>::SetValueLargeChange(u32 value)
{
    m_ValueLargeChange = value;
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetMinimumValue() const
{
    return m_MinimumValue;
}

template<class TModel>
void U32ParameterListItem<TModel>::SetMinimumValue(u32 value)
{
    m_MinimumValue = value;
}

template<class TModel>
u32 U32ParameterListItem<TModel>::GetMaximumValue() const
{
    return m_MaximumValue;
}

template<class TModel>
void U32ParameterListItem<TModel>::SetMaximumValue(u32 value)
{
    m_MaximumValue = value;
}

template<class TModel>
void U32ParameterListItem<TModel>::NextValue(bool isLargeChange)
{
    u32 change = isLargeChange ? m_ValueLargeChange : m_ValueChange;
    u32 value = GetValue();

    if(change > m_MaximumValue - value)
    {
        SetValue(m_MaximumValue);
    }
    else
    {
        SetValue(value + change);
    }
}

template<class TModel>
void U32ParameterListItem<TModel>::PreviousValue(bool isLargeChange)
{
    u32 change = isLargeChange ? m_ValueLargeChange : m_ValueChange;
    u32 value = GetValue();

    if(m_MinimumValue + change > value)
    {
        SetValue(m_MinimumValue);
    }
    else
    {
        SetValue(value - change);
    }
}

template<class TModel>
void U32ParameterListItem<TModel>::PrintValue(char* pText, u32 bufferLength)
{
    NW_ASSERTMSG(m_pPrintValueHandler != NULL, "U32ParameterListItem is not initialized.");
    (*m_pPrintValueHandler)(pText, bufferLength, GetValue());
}

} // snd
} // nw
