﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <controls/SelectableParameterListItem.h>

namespace nw {
namespace snd {

template<class TModel, typename TValue>
SelectableParameterListItem<TModel,TValue>::SelectableParameterListItem(TValue defaultValue) :
m_DefaultValue(defaultValue),
m_pValues(NULL),
m_ValueCount(0),
m_pSetValueHandler(NULL),
m_pPrintValueHandler(NULL)
{
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::Initialize(
    const TValue* pValues,
    u32 valueCount,
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    Initialize(NULL, NULL, pValues, valueCount, pGetValueHandler, pSetValueHandler, pPrintValueHandler);
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::Initialize(
    GetIsCheckedHandler pGetIsCheckedHandler,
    SetIsCheckedHandler pSetIsCheckedHandler,
    const TValue* pValues,
    u32 valueCount,
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    NW_ASSERTMSG(m_pValues == NULL, "SelectableParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pGetValueHandler == NULL, "SelectableParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pSetValueHandler == NULL, "SelectableParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pPrintValueHandler == NULL, "SelectableParameterListItem is already initialized.");
    NW_ASSERT(pValues != NULL);
    NW_ASSERT(pGetValueHandler != NULL);
    NW_ASSERT(pSetValueHandler != NULL);
    NW_ASSERT(pPrintValueHandler != NULL);

    m_pValues = pValues;
    m_ValueCount = valueCount;
    m_pGetValueHandler = pGetValueHandler;
    m_pSetValueHandler = pSetValueHandler;
    m_pPrintValueHandler = pPrintValueHandler;

    Base::Initialize(pGetIsCheckedHandler, pSetIsCheckedHandler);
}

template<class TModel, typename TValue>
TValue SelectableParameterListItem<TModel,TValue>::GetValue() const
{
    NW_ASSERTMSG(m_pGetValueHandler != NULL, "SelectableParameterListItem is not initialized.");
    return (*m_pGetValueHandler)(GetModel());
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::SetValue(TValue value)
{
    NW_ASSERTMSG(m_pSetValueHandler != NULL, "SelectableParameterListItem is not initialized.");
    (*m_pSetValueHandler)(GetModel(), value);
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::ResetValue()
{
    SetValue(m_DefaultValue);
}

template<class TModel, typename TValue>
TValue SelectableParameterListItem<TModel,TValue>::GetDefaultValue() const
{
    return m_DefaultValue;
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::SetDefaultValue(TValue value)
{
    m_DefaultValue = value;
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::NextValue(bool isLargeChange)
{
    TValue currentValue = GetValue();
    TValue newValue = currentValue;
    bool isCurrentValueFound = false;

    for(u32 i = 0; i < m_ValueCount; ++i)
    {
        if(isCurrentValueFound)
        {
            newValue = m_pValues[i];
            break;
        }

        isCurrentValueFound = currentValue == m_pValues[i];
    }

    SetValue(newValue);
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::PreviousValue(bool isLargeChange)
{
    TValue currentValue = GetValue();
    TValue newValue = currentValue;

    for(u32 i = 0; i < m_ValueCount; ++i)
    {
        if(currentValue == m_pValues[i])
        {
            break;
        }

        newValue = m_pValues[i];
    }

    SetValue(newValue);
}

template<class TModel, typename TValue>
void SelectableParameterListItem<TModel,TValue>::PrintValue(char* pText, u32 bufferLength)
{
    NW_ASSERTMSG(m_pPrintValueHandler != NULL, "SelectableParameterListItem is not initialized.");
    (*m_pPrintValueHandler)(pText, bufferLength, GetValue());
}

} // snd
} // nw
