﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_GRID_DRAWER_H_
#define NW_SND_GRID_DRAWER_H_

#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_Delegate.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/gfnd/gfnd_GraphicsContext.h>

namespace nw    {
namespace snd  {

//---------------------------------------------------------------------------
//! @brief        グリッド描画用のユーティリティです。
//---------------------------------------------------------------------------
class GridDrawer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ GridDrawer();


    //---------------------------------------------------------------------------
    //! @brief        グリッドの幅を設定します。
    //!
    //!               CreateCache 後に SetGridInterval を行った場合には、
    //!               UpdateCache を行いキャッシュを更新する必要があります。
    //!
    //! @param[in]    interval  グリッドの幅です。
    //---------------------------------------------------------------------------
    void    SetGridInterval(f32 interval) { this->m_GridInterval = interval; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドの幅を取得します。
    //!
    //! @return       グリッドの幅を返します。
    //---------------------------------------------------------------------------
    f32     GetGridInterval() const { return this->m_GridInterval; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドの数を設定します。
    //!
    //!               CreateCache 後に SetGridCount を行った場合には、
    //!               UpdateCache を行いキャッシュを更新する必要があります。
    //!
    //! @param[in]    count     グリッドの数です。
    //---------------------------------------------------------------------------
    void    SetGridCount(u32 count) { this->m_GridCount = count; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドの数を取得します。
    //!
    //! @return       グリッドの数を返します。
    //---------------------------------------------------------------------------
    u32     GetGridCount() const { return this->m_GridCount; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドカラーを設定します。
    //!
    //! @param[in]    color     グリッドカラーです。
    //---------------------------------------------------------------------------
    void    SetGridColor(nw::ut::Color4u8 color) { this->m_GridColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドカラーを取得します。
    //!
    //! @return       グリッドカラーを返します。
    //---------------------------------------------------------------------------
    nw::ut::Color4u8 GetGridColor() const { return this->m_GridColor; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドの線幅を設定します。
    //!
    //! @param[in]    lineWidth グリッドの線幅です。
    //---------------------------------------------------------------------------
    void    SetGridLineWidth(f32 lineWidth) { this->m_GridLineWidth = lineWidth; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドの線幅を取得します。
    //!
    //! @return       グリッドの線幅を返します。
    //---------------------------------------------------------------------------
    f32     GetGridLineWidth() const { return this->m_GridLineWidth; }


    //---------------------------------------------------------------------------
    //! @brief        グリッドを描画します。
    //---------------------------------------------------------------------------
    void    Draw() const;

    //---------------------------------------------------------------------------
    //! @brief        描画用の行列を設定します。
    //!
    //! @param[in]    projMtx   プロジェクション行列です。
    //! @param[in]    viewMtx   ビュー行列です。
    //! @param[in]    modelMtx  変換行列です。
    //---------------------------------------------------------------------------
    void    SetMatrix(
        const nw::math::MTX44& projMtx,
        const nw::math::MTX34& viewMtx,
        const nw::math::MTX34& modelMtx = nw::math::MTX34::Identity()
    )
    {
        m_ProjMtx = &projMtx;
        m_ViewMtx = &viewMtx;
        m_ModelMtx = modelMtx;
    }

    //---------------------------------------------------------------------------
    //! @brief        描画用のキャッシュを構築します。
    //!
    //! @param[in]    allocator アロケーターです。
    //---------------------------------------------------------------------------
    void CreateCache( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        描画用のキャッシュを破棄します。
    //---------------------------------------------------------------------------
    void DestroyCache();

    //---------------------------------------------------------------------------
    //! @brief        描画用のキャッシュを更新します。
    //---------------------------------------------------------------------------
    void UpdateCache();

private:
    enum Status
    {
        HIDE,
        DRAW
    };

    //! @brief コンフィグが更新されたときに呼ばれます。
    void OnUpdateConfig();

    f32                 m_GridInterval;         //!< グリッドの幅です。
    u32                 m_GridCount;            //!< グリッド数です。
    nw::ut::Color4u8    m_GridColor;            //!< グリッドカラーです。
    f32                 m_GridLineWidth;        //!< グリッドの線幅です。
    nw::dev::DrawBuffer m_GridBuffer;           //!< グリッド描画用のバッファです。

    const nw::math::MTX44* m_ProjMtx;           //!< プロジェクション行列です。
    const nw::math::MTX34* m_ViewMtx;           //!< ビュー行列です。
    nw::math::MTX34        m_ModelMtx;          //!< モデル行列です。

    nw::gfnd::GraphicsContext m_GfxContext;     //!< グラフィックスコンテキストです。
    nw::ut::IAllocator* m_Allocator;            //!< アロケーターです。

    u32 m_DrawStatus;
};

} // namespace snd
} // namespace nw

#endif //  NW_SND_GRID_DRAWER_H_
