﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut.h>

#include "anim/RingBuffer.h"

namespace nw {
namespace snd {

//-------------------------------------------------------------------------------
template < typename TCache, typename TKeyType, u32 CacheMax = 8 >
class CacheRepository
{
public:
    //---------------------------------------------------------------------------
    CacheRepository()
    : m_Allocator(NULL)
    , m_CacheArray()
    {
    }

    //---------------------------------------------------------------------------
    virtual ~CacheRepository()
    {
    }

public:
    //---------------------------------------------------------------------------
    void Initialize(nw::ut::IAllocator* allocator)
    {
        m_Allocator = allocator;
    }

    //---------------------------------------------------------------------------
    void Finalize()
    {
        for (u32 i = 0; i < m_CacheArray.Size(); ++i)
        {
            if (m_CacheArray[i] == NULL)
            {
                continue;
            }

            Destroy(m_CacheArray[i], NULL);
        }

        m_CacheArray.Clear();
    }

    //---------------------------------------------------------------------------
    TCache* Add(const TCache& cache)
    {
        NW_NULL_ASSERT(m_Allocator);

        // 新しいキャッシュを生成
        void* memory = m_Allocator->Alloc(sizeof(TCache));
        NW_NULL_ASSERT(memory);
        std::memset(memory, 0, sizeof(TCache));
        TCache* newCache = new(memory) TCache;

        // 値コピー
        *newCache = cache;

        // コレクションにポインタを追加
        TCache* oldestCache = m_CacheArray.Push(newCache);

        // あふれたものがあれば解放
        if (oldestCache != NULL)
        {
            Destroy(oldestCache, newCache);
        }

#if defined(NW_DEBUG)
        Dump();
#endif

        return newCache;
    }

    //---------------------------------------------------------------------------
    TCache* Find(TKeyType key)
    {
        for (u32 i = 0; i < m_CacheArray.Size(); ++i)
        {
            if (m_CacheArray[i] == NULL)
            {
                continue;
            }

            if (IsCacheHit(*m_CacheArray[i], key))
            {
                return m_CacheArray[i];
            }
        }

        return NULL;
    }

protected:
    //---------------------------------------------------------------------------
    // @note newCacheにNULLが来た場合には必ず破棄処理を行うことを期待しています。
    virtual void OnCacheDestory(TCache& oldestCache, TCache* newCache) = 0;

    //---------------------------------------------------------------------------
    virtual bool IsCacheHit(TCache& cache, TKeyType key) = 0;

    nw::ut::IAllocator* GetAllocator() { return m_Allocator; }

private:
    //---------------------------------------------------------------------------
    void Destroy(TCache* oldestCache, TCache* newCache)
    {
        NW_NULL_ASSERT(oldestCache);
        NW_NULL_ASSERT(m_Allocator);

        OnCacheDestory(*oldestCache, newCache);
        m_Allocator->Free(oldestCache);
    }

    //---------------------------------------------------------------------------
    void Dump()
    {
        for (u32 i = 0; i < m_CacheArray.Size(); ++i)
        {
            NW_LOG("[%d]", i);
            if (m_CacheArray[i] == NULL)
            {
                NW_LOG("NULL");
            }
            else
            {
                NW_LOG("%p", m_CacheArray[i]);
            }
            NW_LOG("\n");
        }
    }

private:
    nw::ut::IAllocator* m_Allocator;

    RingBuffer<TCache, CacheMax> m_CacheArray;
};

} // namespace snd
} // namespace nw

//eof
