﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_AXIS_DRAWER_H_
#define NW_SND_AXIS_DRAWER_H_

#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_Delegate.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/gfnd/gfnd_GraphicsContext.h>

namespace nw    {
namespace snd  {

//---------------------------------------------------------------------------
//! @brief        XYZ 軸描画用のユーティリティです。
//---------------------------------------------------------------------------
class AxisDrawer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        AxisDrawer を生成します。
    //---------------------------------------------------------------------------
    static AxisDrawer* Create( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        AxisDrawer を破棄します。
    //---------------------------------------------------------------------------
    void               Destroy();


    //---------------------------------------------------------------------------
    //! @brief        軸の長さを設定します。
    //!
    //! @param[in]    length    軸の長さです。
    //---------------------------------------------------------------------------
    void    SetAxisLength(f32 length) { m_AxisLength = length; }

    //---------------------------------------------------------------------------
    //! @brief        軸の長さを取得します。
    //!
    //! @return       軸の長さを返します。
    //---------------------------------------------------------------------------
    f32     GetAxisLength() const { return m_AxisLength; }

    //---------------------------------------------------------------------------
    //! @brief        軸の線幅を設定します。
    //!
    //! @param[in]    lineWidth 軸の線幅です。
    //---------------------------------------------------------------------------
    void    SetAxisLineWidth(f32 lineWidth) { m_AxisLineWidth = lineWidth; }

    //---------------------------------------------------------------------------
    //! @brief        軸の線幅を取得します。
    //!
    //! @return       軸の線幅を返します。
    //---------------------------------------------------------------------------
    f32     GetAxisLineWidth() const { return m_AxisLineWidth; }

    //---------------------------------------------------------------------------
    //! @brief        各軸の色を設定します。
    //!
    //! @param[in]    axisXColor X 軸の色です。
    //! @param[in]    axisYColor Y 軸の色です。
    //! @param[in]    axisZColor Z 軸の色です。
    //---------------------------------------------------------------------------
    void    SetAxisColor(nw::ut::Color4u8 axisXColor, nw::ut::Color4u8 axisYColor, nw::ut::Color4u8 axisZColor)
    {
        m_AxisXColor = axisXColor;
        m_AxisYColor = axisYColor;
        m_AxisZColor = axisZColor;
    }

    //---------------------------------------------------------------------------
    //! @brief        X 軸の色を設定します。
    //!
    //! @param[in]    color     X 軸の色です。
    //---------------------------------------------------------------------------
    void    SetAxisXColor(nw::ut::Color4u8 color)
    {
        m_AxisXColor = color;
    }

    //---------------------------------------------------------------------------
    //! @brief        Y 軸の色を設定します。
    //!
    //! @param[in]    color     Y 軸の色です。
    //---------------------------------------------------------------------------
    void    SetAxisYColor(nw::ut::Color4u8 color)
    {
        m_AxisYColor = color;
    }

    //---------------------------------------------------------------------------
    //! @brief        Z 軸の色を設定します。
    //!
    //! @param[in]    color     Z 軸の色です。
    //---------------------------------------------------------------------------
    void    SetAxisZColor(nw::ut::Color4u8 color)
    {
        m_AxisZColor = color;
    }

    //---------------------------------------------------------------------------
    //! @brief        各軸の色を取得します。
    //!
    //! @param[out]   axisXColor X 軸の色です。
    //! @param[out]   axisYColor Y 軸の色です。
    //! @param[out]   axisZColor Z 軸の色です。
    //---------------------------------------------------------------------------
    void    GetAxisColor(nw::ut::Color4u8* axisXColor, nw::ut::Color4u8* axisYColor, nw::ut::Color4u8* axisZColor) const
    {
        if ( axisXColor )
        {
            *axisXColor = m_AxisXColor;
        }
        if ( axisYColor )
        {
            *axisYColor = m_AxisYColor;
        }
        if ( axisZColor )
        {
            *axisZColor = m_AxisZColor;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        X 軸の色を取得します。
    //!
    //! @return       X 軸の色を返します。
    //---------------------------------------------------------------------------
    nw::ut::Color4u8 GetAxisXColor() const
    {
        return m_AxisXColor;
    }

    //---------------------------------------------------------------------------
    //! @brief        Y 軸の色を取得します。
    //!
    //! @return       Y 軸の色を返します。
    //---------------------------------------------------------------------------
    nw::ut::Color4u8 GetAxisYColor() const
    {
        return m_AxisYColor;
    }

    //---------------------------------------------------------------------------
    //! @brief        Z 軸の色を取得します。
    //!
    //! @return       Z 軸の色を返します。
    //---------------------------------------------------------------------------
    nw::ut::Color4u8 GetAxisZColor() const
    {
        return m_AxisZColor;
    }


    //---------------------------------------------------------------------------
    //! @brief        グリッドを描画します。
    //---------------------------------------------------------------------------
    void Draw() const;

    //---------------------------------------------------------------------------
    //! @brief        描画用の行列をセットします。
    //!
    //! @param[in]    projMtx   プロジェクション行列です。
    //! @param[in]    viewMtx   ビュー行列です。
    //! @param[in]    modelMtx  モデル行列です。
    //---------------------------------------------------------------------------
    void SetMatrix(
        const nw::math::MTX44& projMtx,
        const nw::math::MTX34& viewMtx,
        const nw::math::MTX34& modelMtx = nw::math::MTX34::Identity()
        )
    {
        m_ProjMtx = &projMtx;
        m_ViewMtx = &viewMtx;
        m_ModelMtx = modelMtx;
    }


private:
    enum Status
    {
        HIDE,
        DRAW
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    allocator アロケーターです。
    //---------------------------------------------------------------------------
    /* ctor */ AxisDrawer( nw::ut::IAllocator* allocator )
        : m_AxisLength( 3.f )
        , m_AxisLineWidth( 2.f )
        , m_AxisXColor( 0xFF, 0x00, 0x00, 0xFF )
        , m_AxisYColor( 0x00, 0xFF, 0x00, 0xFF )
        , m_AxisZColor( 0x00, 0x00, 0xFF, 0xFF )
        , m_ProjMtx( NULL )
        , m_ViewMtx( NULL )
        , m_Allocator( allocator )
        , m_DrawStatus( DRAW )
    {}

    //---------------------------------------------------------------------------
    //! @brief        描画用のキャッシュを構築します。
    //!
    //! @param[in]    allocator アロケーターです。
    //---------------------------------------------------------------------------
    static void CreateCache( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        描画用のキャッシュを破棄します。
    //---------------------------------------------------------------------------
    static void DestroyCache();

    //! @brief コンフィグが更新されたときに呼ばれます。
    void OnUpdateConfig();

    f32                 m_AxisLength;        //!< 軸の長さです。
    f32                 m_AxisLineWidth;     //!< 軸の線幅です。
    nw::ut::Color4u8    m_AxisXColor;        //!< X 軸の色です。
    nw::ut::Color4u8    m_AxisYColor;        //!< Y 軸の色です。
    nw::ut::Color4u8    m_AxisZColor;        //!< Z 軸の色です。

    const nw::math::MTX44* m_ProjMtx;        //!< プロジェクション行列です。
    const nw::math::MTX34* m_ViewMtx;        //!< ビュー行列です。
    nw::math::MTX34        m_ModelMtx;       //!< モデル行列です。

    nw::gfnd::GraphicsContext m_GfxContext;  //!< グラフィックスコンテキストです。
    nw::ut::IAllocator* m_Allocator;         //!< アロケーターです。

    static nw::dev::DrawBuffer m_AxisXBuffer;       //!< X 軸描画用のバッファです。
    static nw::dev::DrawBuffer m_AxisYBuffer;       //!< Y 軸描画用のバッファです。
    static nw::dev::DrawBuffer m_AxisZBuffer;       //!< Z 軸描画用のバッファです。
    static int s_InitializedCount;                  //!< 初期化回数の参照カウントです。
    static nw::ut::IAllocator* s_CacheAllocator;    //!< 描画用キャッシュのアロケーターです。

    u32 m_DrawStatus;
};

} // namespace snd
} // namespace nw

#endif //  NW_SND_AXIS_DRAWER_H_
