﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <anim/IAnimSoundTarget.h>

#include <nw/ut.h>
#include <nw/dev.h>
#include <nw/g3d.h>

#include <anim/CacheRepository.h>
#include <anim/ModelUtility.h>
#include <anim/ViewerCamera.h>
#include <anim/GridDrawer.h>
#include <anim/AxisDrawer.h>

namespace nw {
namespace snd {

class AnimSoundTargetCafe
    : public IAnimSoundTarget
{
public:
    //! @brief Create に渡す生成パラメータです。
    struct CreateArg
    {
        nw::ut::IAllocator* allocator;            //!< アロケータです。
        u8*                 shaderBinary;         //!< シェーダーバイナリへのポインタです。
        u32                 shaderBinarySize;     //!< シェーダーバイナリのサイズです。
        u8*                 fontBinary;           //!< フォントバイナリへのポインタです。
        u32                 fontBinarySize;       //!< フォントバイナリのサイズです。
        s32                 screenWidth;          //!< スクリーンの幅です。
        s32                 screenHeight;         //!< スクリーンの高さです。

        //! @brief コンストラクタです。
        CreateArg() :
            allocator( NULL ),
            shaderBinary( NULL ),
            shaderBinarySize( 0 ),
            fontBinary( NULL ),
            fontBinarySize( 0 )
        {}
    };

public:
    static AnimSoundTargetCafe* Create(const CreateArg& arg);
    virtual void Destroy();

    virtual u32 OnLoadModel(const char* filePath, bool isReload);
    virtual u32 OnLoadAnimation(const char* filePath, bool isReload);
    virtual void OnUnload();
    virtual void OnUpdateInputs(const dev::Pad* pad, const nw::dev::Mouse* mouse);
    virtual void OnCalc();
    virtual void OnDraw();


    virtual void OnPlayAnim();
    virtual void OnStopAnim();
    virtual void OnSetLoop(bool flg);
    virtual void OnSetRange(f32 start, f32 end);
    virtual f32  OnGetRangeStart();
    virtual f32  OnGetRangeEnd();
    virtual void OnSetRate(f32 rate);
    virtual f32  OnGetRate();
    virtual void OnSetFrame(f32 frame);
    virtual f32  OnGetFrame();

    virtual bool IsPlayingAnim();
    virtual void SetCameraEnabled(bool enabled);

private:
    AnimSoundTargetCafe(const CreateArg& arg);
    ~AnimSoundTargetCafe();

    void InitializeShader();
    void FinalizeShader();

    void UpdateInputsImpl(const dev::Pad* pad, const nw::dev::Mouse* mouse);

    // dwの中にプレビューを表示するため、処理/描画関数の実体を分ける
    void CalcImpl();
    void DrawImpl();

private:
    nw::ut::IAllocator* m_Allocator;
    bool m_IsCameraEnabled;

    f32 m_RangeStart;
    f32 m_RangeEnd;

    // ビュー設定
    ViewerProjection m_Projection;
    ViewerCamera     m_Camera;

    SimpleShader m_Shader;
    SimpleModel* m_LatestModel;

    GridDrawer m_GridDrawer;
    AxisDrawer* m_AxisDrawer;

    //----------------------------------------------------------
    void* LoadBinaryFile(const char* path);

    //----------------------------------------------------------
    nw::g3d::AnimFrameCtrl m_FrameCtrl;
    bool m_IsAnimEnable;

    //----------------------------------------------------------
    // キャッシュ関係の定義
    static const u32 CACHE_MAX = 8;
    static const int PATH_MAX = 260;

    //----------------------------------------------------------
    struct ModelCacheInfo
    {
        char path[PATH_MAX];
        SimpleModel model;
    };
    class ModelCache : public CacheRepository<ModelCacheInfo, const char*, CACHE_MAX>
    {
    public:
        ModelCache();
        ~ModelCache();

    protected:
        virtual void OnCacheDestory(ModelCacheInfo& oldestCache, ModelCacheInfo* newCache);
        virtual bool IsCacheHit(ModelCacheInfo& cache, const char* key);
    };

    //----------------------------------------------------------
    struct AnimCacheInfo
    {
        char path[PATH_MAX];
        SimpleAnimSet animSet;
    };
    class AnimCache : public CacheRepository<AnimCacheInfo, const char*, CACHE_MAX>
    {
    public:
        AnimCache();
        ~AnimCache();

    public:
        void SetCurrent(AnimCacheInfo* current) { m_Current = current;}
        AnimCacheInfo* Current() { return m_Current; }

    protected:
        virtual void OnCacheDestory(AnimCacheInfo& oldestCache, AnimCacheInfo* newCache);
        virtual bool IsCacheHit(AnimCacheInfo& cache, const char* key);

    private:
        AnimCacheInfo* m_Current;
    };

    //----------------------------------------------------------
    ModelCache m_ModelCache;
    AnimCache m_AnimCache;
};

} // namespace snd
} // namespace nw

// eof
