﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Configuration;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

// MEFを使うには以下の2つが必要です。
// 参照設定にもSystem.ComponentModel.Compositionを追加する必要があります。
using System.ComponentModel.Composition;
using System.ComponentModel.Composition.Hosting;

namespace NintendoWare.SoundMaker
{
    // 参照設定に、SoundMakerPlugin を含める必要があります。
    using NintendoWare.SoundMakerPlugin;

    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMaker.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Resources;
    using Runtime = NintendoWare.SoundRuntime;

    // SoundMaker.exe から使うための通常の Export
    [Export(typeof(ISoundMakerPlugin))]
    // SoundMakerPluginCafe.dll から使うための Export
    [Export("NintendoWare.SoundMaker.SoundMakerPluginGeneric", typeof(ISoundMakerPlugin))]
    public class SoundMakerPluginGeneric : ISoundMakerPlugin
    {
        private AppSettingsSection appSettings;
        private IDocumentFactory[] documentFactories;
        private IDocumentReader[] documentReaders;
        private CommunicationInfo communicationInfo;

        public SoundMakerPluginGeneric()
        {
            Configuration config = ConfigurationManager.OpenExeConfiguration(this.GetType().Assembly.Location);
            this.appSettings = (AppSettingsSection)config.GetSection("appSettings");
            this.communicationInfo = new CommunicationInfo((this as ISoundMakerPlugin).Platform);
            this.CommunicationInfoInitialize();
        }

        string ISoundMakerPlugin.Platform
        {
            get
            {
                return "Generic";
            }
        }

        // Preview

        int ISoundMakerPlugin.RuntimeSoundSystem_GetActiveVoiceCount()
        {
            return Runtime.AtkSoundSystem.GetActiveVoiceCount();
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_LockSoundThread()
        {
            Runtime.AtkSoundSystem.LockSoundThread();
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_UnlockSoundThread()
        {
            Runtime.AtkSoundSystem.UnlockSoundThread();
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_StopAllVoices()
        {
            Runtime.AtkSoundSystem.StopAllVoices();
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_SoundThreadPause(bool pauseFlag)
        {
            Runtime.AtkSoundSystem.SoundThreadPause(pauseFlag);
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_SoundFrameProcess()
        {
            Runtime.AtkSoundSystem.SoundFrameProcess();
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_InitSoundSystem(int soundThreadPriority, int dvdThreadPriority)
        {
            Runtime.AtkSoundSystem.InitSoundSystem(soundThreadPriority, dvdThreadPriority);
        }

        void ISoundMakerPlugin.RuntimeSoundSystem_ShutdownSoundSystem()
        {
            Runtime.AtkSoundSystem.ShutdownSoundSystem();
        }

        bool ISoundMakerPlugin.IsNwRendererEnabled
        {
            get { return false; }
        }

        // Runtime.Global

        int ISoundMakerPlugin.RuntimeGlobal_DefaultSampleRate
        {
            get { return Runtime.Global.DefaultSampleRate; }
        }

        int ISoundMakerPlugin.RuntimeGlobal_CHANNEL_COUNT
        {
            get
            {
                return Runtime.Global.ChannelCount;
            }
        }

        int ISoundMakerPlugin.RuntimeGlobal_MSEC_PER_FRAME
        {
            get
            {
                return Runtime.Global.MsecPerFrame;
            }
        }

        int ISoundMakerPlugin.RuntimeGlobal_AXSynthesize(short[] buffer, int samplingRate)
        {
            return Runtime.Global.AXSynthesize(buffer, samplingRate);
        }

        void ISoundMakerPlugin.InitializeAudioDevice(string deviceName)
        {
            Nintendo.AudioTool.Audio.InitializeAudioDevice(deviceName);
        }

        IEnumerable<string> ISoundMakerPlugin.EnumerateAudioDeviceNames()
        {
            return Nintendo.AudioTool.Audio.EnumerateAudioDevices();
        }

        void ISoundMakerPlugin.RuntimeGlobal_AXInit()
        {
        }

        void ISoundMakerPlugin.RuntimeGlobal_AXQuit()
        {
        }

        // Runtime.SequenceSoundPlayerCafe

        short ISoundMakerPlugin.RuntimeSequenceSoundPlayer_GetGlobalVariable(int varNo)
        {
            return Runtime.SequenceSoundPlayerCafe.GetGlobalVariable(varNo);
        }

        void ISoundMakerPlugin.RuntimeSequenceSoundPlayer_SetGlobalVariable(int varNo, short var)
        {
            Runtime.SequenceSoundPlayerCafe.SetGlobalVariable(varNo, var);
        }

        // Create

        IPreviewSequenceChannel ISoundMakerPlugin.CreatePreviewSequenceChannel(string bankFileName, Instrument instrument, ref IRuntimeNoteOnInfo noteOnInfo)
        {
            Runtime.NoteOnInfo noInfo = (noteOnInfo as RuntimeNoteOnInfo).NoteOnInfo;

            return new PreviewSequenceChannelGeneric(bankFileName, instrument, ref noInfo);
        }

        IPreviewSequenceChannel ISoundMakerPlugin.CreatePreviewSequenceChannel(string bankFileName, ref IRuntimeNoteOnInfo noteOnInfo, RendererType rendererType)
        {
            Runtime.ChannelCafe.RendererType type = this.RendererType2SoundRuntimeRendererType(rendererType);
            Runtime.NoteOnInfo noInfo = (noteOnInfo as RuntimeNoteOnInfo).NoteOnInfo;

            return new PreviewSequenceChannelGeneric(bankFileName, ref noInfo, type);
        }

        IPreviewWaveChannel ISoundMakerPlugin.CreatePreviewWaveChannel(WaveSoundBase sound, RendererType rendererType, bool forceNoLoop)
        {
            Runtime.ChannelCafe.RendererType type = this.RendererType2SoundRuntimeRendererType(rendererType);
            return new PreviewWaveChannelGeneric(sound, type, forceNoLoop);
        }

        IPreviewStreamChannel ISoundMakerPlugin.CreatePreviewStreamChannel(
            StreamSoundTrackBase streamTrack,
            StreamSoundTrackBase baseTrack,
            int channelIndexStart,
            float volume,
            float pitch,
            PanMode panMode,
            PanCurve panCurve,
            float mainSend,
            RendererType rendererType,
            bool forceNoLoop)
        {
            Runtime.ChannelCafe.RendererType type = this.RendererType2SoundRuntimeRendererType(rendererType);
            return new PreviewStreamChannelGeneric(streamTrack, baseTrack, channelIndexStart, volume, pitch, panMode, panCurve, mainSend, type, forceNoLoop);
        }

        ISequenceSoundPlayer ISoundMakerPlugin.CreateSequenceSoundPlayer()
        {
            return new SequenceSoundPlayerGeneric();
        }

        ISeqFileReader ISoundMakerPlugin.CreateSeqFileReader(IntPtr seqData)
        {
            return new SeqFileReaderGeneric(seqData);
        }

        ISequenceSoundBinaryReader ISoundMakerPlugin.CreateSequenceSoundBinaryReader()
        {
            return new NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SequenceSoundBinaryReader();
        }

        IMidiManager ISoundMakerPlugin.CreateMidiManager()
        {
            return new MidiManagerGeneric();
        }


        private Runtime.ChannelCafe.RendererType RendererType2SoundRuntimeRendererType(RendererType rendererType)
        {
            Runtime.ChannelCafe.RendererType type;

            switch (rendererType)
            {
                default:
                case RendererType.k32KHz:
                    type = Runtime.ChannelCafe.RendererType.k32KHz;
                    break;

                case RendererType.k48KHz:
                    type = Runtime.ChannelCafe.RendererType.k48KHz;
                    break;
            }

            return type;
        }


        // Converter 関連

        CreateSoundProjectConverterDelegate ISoundMakerPlugin.GetCreateSoundProjectConverterDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return this.CreateSoundProjectConverter;
#else
            return null;
#endif
        }

        private ISoundProjectConverter2 CreateSoundProjectConverter(SoundProjectConversionTraits traits)
        {
            return new SoundProjectConverter(traits);
        }

        CreateFileIDAggregateFactoryDelegate ISoundMakerPlugin.GetCreateFileIDAggregateFactoryDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return this.CreateFileIDAggregateFactory;
#else
            return null;
#endif
        }

        private IFileIDAggregateFactory CreateFileIDAggregateFactory(string basePath)
        {
            return new NintendoWare.SoundFoundation.Conversion.NintendoWareBinary.FileIDAggregateFactory(basePath);
        }

        CreateFileIDDelegate ISoundMakerPlugin.GetCreateFileIDDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return this.CreateFileID;
#else
            return null;
#endif
        }

        private IFileID CreateFileID(string value)
        {
            return new NintendoWare.SoundFoundation.Conversion.NintendoWareBinary.FileID(value);
        }

        CreateSoundArchiveBinaryXmlDelegate ISoundMakerPlugin.GetCreateSoundArchiveBinaryXmlDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return this.CreateSoundArchiveBinaryXml;
#else
            return null;
#endif
        }

        private ISoundArchiveBinaryXml CreateSoundArchiveBinaryXml(Stream stream)
        {
            NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveBinaryXml soundArchiveBinaryXml;

            if (stream == null)
            {
                soundArchiveBinaryXml = new NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveBinaryXml()
                {
                    Sounds = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.SoundXml>(),
                    SoundSetBanks = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.SoundSetBankXml>(),
                    SoundGroups = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.SoundGroupXml>(),
                    Groups = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.GroupXml>(),
                    WaveArchives = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.WaveArchiveXml>(),
                    Players = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.PlayerXml>(),
                    Files = new List<NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements.FileXml>(),
                };
            }
            else
            {
                soundArchiveBinaryXml = new XmlSerializer(typeof(NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveBinaryXml)).Deserialize(stream) as NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveBinaryXml;
            }

            return new NintendoWare.SoundMaker.FileFormats.NintendoWareBinary.SoundArchiveBinaryXmlCafe(soundArchiveBinaryXml);
        }

        CreateWaveFileInfoDelegate ISoundMakerPlugin.GetCreateWaveFileInfoDelegate()
        {
            return this.CreateWaveFileInfo;
        }

        private IWaveFileInfo CreateWaveFileInfo()
        {
            return new WaveFileInfo();
        }

        CreateSoundIDCppHeaderExporterDelegate ISoundMakerPlugin.GetCreateSoundIDCppHeaderExporterDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return (() => this.SoundIDCppHeaderExporter;);
#else
            return null;
#endif
        }

        private ISoundIDCppHeaderExporter SoundIDCppHeaderExporter
        {
            get
            {
                return new NintendoWare.SoundFoundation.Conversion.NintendoWareBinary.SoundIDCppHeaderExporter();
            }
        }

        CreateSoundArchiveBinaryMapExporterDelegate ISoundMakerPlugin.GetCreateSoundArchiveBinaryMapExporterDelegate()
        {
#if USE_PLUGIN_CONVERTER
            return (() => this.SoundArchiveBinaryMapExporter);
#else
            return null;
#endif
        }

        private ISoundArchiveBinaryMapExporter SoundArchiveBinaryMapExporter
        {
            get
            {
                return new NintendoWare.SoundFoundation.Conversion.NintendoWareBinary.SoundArchiveBinaryMapExporter();
            }
        }

        // 接続関連インターフェース
        CommunicationInfo ISoundMakerPlugin.CommunicationInfo
        {
            get
            {
                return this.communicationInfo;
            }
        }

        private void CommunicationInfoInitialize()
        {
            ConnectionInfo[] connectionInfos =
                {
#if USE_NWSND
                    new ConnectionInfo("PCApplication_nwsnd", MessageResource.PCApplication_nwsnd, false, null, null, null, null, null, null, null, null),
#endif
#if USE_PCSOUNDPLAYER
                    new ConnectionInfo("PCSoundPlayer_nwsnd", MessageResource.PCSoundPlayer_nwsnd, false, null, null, null, null, null, null, null, null),
#endif
                    new ConnectionInfo("PCApplication", MessageResource.PCApplication, false, null, null, null, null, null, null, null, null, "Generic-Windows"),
                };

            foreach (ConnectionInfo info in connectionInfos)
            {
                this.communicationInfo.SetConnectionInfo(info.TargetName, info);
            }
        }

        // その他
        void ISoundMakerPlugin.Initialize(SoundSetComponentFactory soundSetComponentFactory)
        {
            this.documentFactories = new IDocumentFactory[] { };
            this.documentReaders = new IDocumentReader[] { };
        }

        IEnumerable<IDocumentFactory> ISoundMakerPlugin.DocumentFactories
        {
            get { return this.documentFactories; }
        }

        IEnumerable<IDocumentReader> ISoundMakerPlugin.DocumentReaders
        {
            get { return this.documentReaders; }
        }

        string ISoundMakerPlugin.GetExpandedAppSetting(string key)
        {
            if (this.appSettings.Settings.AllKeys.Contains(key) == false)
            {
                return string.Empty;
            }

            return this.ExpandMacro(this.appSettings.Settings[key].Value);
        }

        // コントロール関係
        string ISoundMakerPlugin.PlatformLabel
        {
            get
            {
                return "PC";
            }
        }

        bool ISoundMakerPlugin.IsValidateWaveEncodingValue(WaveEncoding encoding)
        {
            switch (encoding)
            {
                case WaveEncoding.Pcm16:
                case WaveEncoding.Adpcm:
                case WaveEncoding.NoConvert:
                    return true;

                default:
                    return false;
            }
        }

        // 実機 Player 関係（Genenric は PC なので関係ない）
        ITargetPlayer ISoundMakerPlugin.TargetPlayer
        {
            get
            {
                return null;
            }
        }

        private string ExpandMacro(string value)
        {
            {
                string executingDirectory = Path.GetFullPath(Path.GetDirectoryName(Assembly.GetEntryAssembly().Location));
                value = value.Replace("$(SoundMakerDirectory)", executingDirectory);
            }
            return value;
        }

        //////
#if DEBUG
        string ISoundMakerPlugin.GetMessage()
        {
            return "SoundMaker Generic !!";
        }
#endif
    }
}
