﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMakerPlugin;
    using Runtime = NintendoWare.SoundRuntime;

    public class PreviewWaveChannelGeneric : PreviewChannelGeneric, IPreviewWaveChannel
    {
        private Runtime.ChannelCafe _channel = null;
        private Runtime.WaveInfo _waveInfo;

        public PreviewWaveChannelGeneric(WaveSoundBase sound, Runtime.ChannelCafe.RendererType rendererType, bool forceNoLoop)
        {
            Initialize(sound, rendererType, forceNoLoop);
        }

        public override Runtime.ChannelCafe Channel
        {
            get { return _channel; }
        }

        public void ChannelSetUserVolume(float volume)
        {
            _channel.SetUserVolume(volume);
        }

        protected override Runtime.WaveInfo WaveInfo
        {
            get { return _waveInfo; }
        }

        private void Initialize(WaveSoundBase sound, Runtime.ChannelCafe.RendererType rendererType, bool forceNoLoop)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            if (!File.Exists(sound.FilePath))
            {
                throw new FileNotFoundException("file not found.", sound.FilePath);
            }

            bool? isLoop = null;
            if (forceNoLoop == true)
            {
                isLoop = false;
            }

            WaveFileInfo waveFileInfo = (WaveFileInfo)WaveFileManager.Instance.LoadFile(sound.FilePath, isLoop, null, null, sound.GetTargetSampleRate(), sound.GetTargetChannelCount());

            if (null == waveFileInfo) { throw new Exception(); }


            Runtime.ChannelCafe newChannel = new Runtime.ChannelCafe();

            bool result = newChannel.AllocChannel(waveFileInfo.WaveFile.ChannelCount, sound.ChannelPriority, rendererType);
            if (!result) { throw new Exception(); }

            newChannel.SetUserVolume(sound.Volume / 127.0f);
            newChannel.SetUserPitchRatio(sound.Pitch);
            newChannel.SetUserPan((sound.Pan - 64) / 63.0f);
            newChannel.SetRelease(sound.ReleaseEnvelope);
            newChannel.SetMainSend(sound.Sends.MainSend / 127.0f - 1.0f);
            newChannel.SetPanMode(sound.PanMode.ToRuntimePanMode());
            newChannel.SetPanCurve(sound.PanCurve.ToRuntimePanCurve());
            newChannel.SetBiquadFilter(sound.BiquadType.ToRuntimeBiquadFilterType(),
                                        (float)sound.Biquad / 127.0F);
            newChannel.SetUserLpfFreq((float)(sound.LPF - 64) / 64.0F);

            _channel = newChannel;
            _waveInfo = waveFileInfo.ToWaveInfo(0, waveFileInfo.WaveFile.ChannelCount);
        }
    }
}
