﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.ToolDevelopmentKit;
    using Runtime = NintendoWare.SoundRuntime;

    public class PreviewSequenceChannelGeneric : PreviewChannelGeneric, IPreviewSequenceChannel
    {
        private const float VELOCITY_MAX_R = 1.0f / 127.0f;
        private const float INSTRUMENT_VOLUME_CRITERION_R = 1.0f / 127.0f;

        private Runtime.ChannelCafe _channel;
        private Runtime.WaveInfo _waveInfo;
        private int _noteLength = 0;

        public PreviewSequenceChannelGeneric(string bankFileName, ref Runtime.NoteOnInfo noteOnInfo,
                                          Runtime.ChannelCafe.RendererType rendererType)
        {
            Initialize(bankFileName, ref noteOnInfo, rendererType);
        }

        public PreviewSequenceChannelGeneric(string bankFileName, Instrument instrument,
                                          ref Runtime.NoteOnInfo noteOnInfo)
            : this(bankFileName, instrument, ref noteOnInfo, Runtime.ChannelCafe.RendererType.k32KHz)
        {
        }

        public PreviewSequenceChannelGeneric(string bankFileName, Instrument instrument,
                                          ref Runtime.NoteOnInfo noteOnInfo,
                                          Runtime.ChannelCafe.RendererType rendererType)
        {
            Initialize(bankFileName, instrument, ref noteOnInfo, rendererType);
        }

        public override Runtime.ChannelCafe Channel
        {
            get { return _channel; }
        }

        public void SetInstrumentVolume(float volume)
        {
            _channel.SetInstrumentVolume(volume);
        }

        object IPreviewSequenceChannel.Channel
        {
            get { return (object)_channel; }
        }

        protected override Runtime.WaveInfo WaveInfo
        {
            get { return _waveInfo; }
        }

        protected override int PlaybackLength
        {
            get { return _noteLength; }
        }

        private XmlInstrument FindXmlInstrument(XmlBankFile bankXmlFile, int programNo)
        {
            foreach (XmlInstrument instrumentXml in bankXmlFile.Body.Bank.Items)
            {
                if (!instrumentXml.IsInstrumentEnabled())
                {
                    continue;
                }

                XmlProgramNoParam programNoParam =
                    this.GetParameter(instrumentXml.Parameters, XmlParameterNames.ProgramNo)
                    as XmlProgramNoParam;

                if (programNoParam == null)
                {
                    continue;
                }

                if (programNoParam.Value == programNo)
                {
                    return instrumentXml;
                }
            }

            return null;
        }

        private XmlInstrument FindXmlInstrument(XmlBankFile bankXmlFile, string instrumentName)
        {
            foreach (XmlInstrument instrumentXml in bankXmlFile.Body.Bank.Items)
            {
                if (!instrumentXml.IsInstrumentEnabled())
                {
                    continue;
                }

                if (instrumentName == instrumentXml.Name)
                {
                    return instrumentXml;
                }
            }
            return null;
        }

        private XmlKeyRegion FindXmlKeyRegion(XmlInstrument instrumentXml, int key)
        {
            foreach (XmlKeyRegion keyRegionXml in instrumentXml.Items)
            {
                XmlKeyMaxParam keyMaxParam =
                    this.GetParameter(keyRegionXml.Parameters, XmlParameterNames.KeyMax)
                    as XmlKeyMaxParam;
                XmlKeyMinParam keyMinParam =
                    this.GetParameter(keyRegionXml.Parameters, XmlParameterNames.KeyMin)
                    as XmlKeyMinParam;

                if (keyMinParam.Value <= key && key <= keyMaxParam.Value)
                {
                    return keyRegionXml;
                }
            }
            return null;
        }

        private XmlVelocityRegion FindXmlVelocityRegion(XmlKeyRegion keyRegionXml, int velocity)
        {
            foreach (XmlVelocityRegion velocityRegionXml in keyRegionXml.Items)
            {
                XmlVelocityMaxParam velocityMaxParam =
                   this.GetParameter(velocityRegionXml.Parameters, ProjectParameterNames.VelocityRegion.VelocityMax)
                   as XmlVelocityMaxParam;
                XmlVelocityMinParam velocityMinParam =
                    this.GetParameter(velocityRegionXml.Parameters, ProjectParameterNames.VelocityRegion.VelocityMin)
                    as XmlVelocityMinParam;

                if (velocityMinParam.Value <= velocity && velocity <= velocityMaxParam.Value)
                {
                    return velocityRegionXml;
                }
            }
            return null;
        }

        private KeyRegion FindXmlKeyRegion(Instrument instrument, int key)
        {
            foreach (KeyRegion keyRegion in instrument.Children)
            {
                if (keyRegion.KeyMin <= key && key <= keyRegion.KeyMax)
                {
                    return keyRegion;
                }
            }
            return null;
        }

        private VelocityRegion FindXmlVelocityRegion(KeyRegion keyRegion, int velocity)
        {
            foreach (VelocityRegion velocityRegion in keyRegion.Children)
            {
#if false
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsEnabled)
                {
                    continue;
                }
#endif

                if (velocityRegion.VelocityMin <= velocity && velocity <= velocityRegion.VelocityMax)
                {
                    return velocityRegion;
                }
            }
            return null;
        }

        private XmlParameter GetParameter(IEnumerable<XmlParameter> parameters, string name)
        {
            Assertion.Argument.NotNull(parameters);
            Assertion.Argument.NotNull(name);

            foreach (XmlParameter parameter in parameters)
            {
                if (parameter.Name == name)
                {
                    return parameter;
                }
            }

            return null;
        }

        private IDictionary<string, XmlParameter> CreateParameterDictionary(IEnumerable<XmlParameter> parameters)
        {
            Assertion.Argument.NotNull(parameters);

            Dictionary<string, XmlParameter> dictionary = new Dictionary<string, XmlParameter>();

            foreach (XmlParameter parameter in parameters)
            {
                dictionary.Add(parameter.Name, parameter);
            }

            return dictionary;
        }

        private void Initialize(string bankFileName, ref Runtime.NoteOnInfo noteOnInfo, Runtime.ChannelCafe.RendererType rendererType)
        {
            if (null == bankFileName) { throw new ArgumentNullException("bankFileName"); }

            BankFileManager.BankFileInfo bankFileInfo = BankFileManager.Instance.LoadFile(bankFileName);
            if (bankFileInfo == null) { return; }

            XmlInstrument instrumentXml = FindXmlInstrument(bankFileInfo.XmlBank, noteOnInfo.prgNo);
            if (instrumentXml == null) { return; }

            XmlKeyRegion keyRegionXml = FindXmlKeyRegion(instrumentXml, noteOnInfo.key);
            if (keyRegionXml == null) { return; }

            XmlVelocityRegion velocityRegionXml = FindXmlVelocityRegion(keyRegionXml, noteOnInfo.velocity);
            if (velocityRegionXml == null) { return; }

            IDictionary<string, XmlParameter> instrumentParams = this.CreateParameterDictionary(instrumentXml.Parameters);
            IDictionary<string, XmlParameter> keyRegionParams = this.CreateParameterDictionary(keyRegionXml.Parameters);
            IDictionary<string, XmlParameter> velocityRegionParams = this.CreateParameterDictionary(velocityRegionXml.Parameters);

            string waveFileName = Path.Combine(
                Path.GetDirectoryName(bankFileName),
                velocityRegionParams[XmlParameterNames.FilePath].XmlValue.ToString());

            bool isResampleEnabled;
            instrumentXml.FindParameterValue(XmlParameterNames.IsResampleEnabled, out isResampleEnabled);
            bool isDownMixEnabled;
            instrumentXml.FindParameterValue(XmlParameterNames.IsDownMixEnabled, out isDownMixEnabled);

            int? sampleRate = null;
            if (isResampleEnabled)
            {
                instrumentXml.FindParameterValue(XmlParameterNames.SampleRate, out sampleRate);
            }

            int? channelCount = null;
            if (isDownMixEnabled == true)
            {
                channelCount = 1; // モノラル化
            }

            WaveFileInfo waveFileInfo = (WaveFileInfo)WaveFileManager.Instance.LoadFile(waveFileName, null, null, null, sampleRate, channelCount);

            if (waveFileInfo == null) { return; }


            Runtime.ChannelCafe channel = new Runtime.ChannelCafe();
            channel.AllocChannel(
                waveFileInfo.WaveFile.ChannelCount,
                noteOnInfo.priority,
                noteOnInfo.channelCallback,
                noteOnInfo.channelCallbackData,
                rendererType
            );

            XmlEnvelopeParam envelopeXml = null;

            switch (
                InstrumentEnvelopeModeEx.Parse(
                    instrumentParams[XmlParameterNames.InstrumentEnvelopeMode].XmlValue.ToString())
                )
            {
                case InstrumentEnvelopeMode.Instrument:
                    envelopeXml =
                        instrumentParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    break;

                case InstrumentEnvelopeMode.KeyRegion:
                    envelopeXml =
                        keyRegionParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    break;

                case InstrumentEnvelopeMode.VelocityRegion:
                    envelopeXml =
                        velocityRegionParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    break;

                case InstrumentEnvelopeMode.Lower:
                    envelopeXml =
                        instrumentParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    if (envelopeXml != null)
                    {
                        break;
                    }

                    envelopeXml =
                        keyRegionParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    if (envelopeXml != null)
                    {
                        break;
                    }

                    envelopeXml =
                        velocityRegionParams[XmlParameterNames.EnvelopeParams] as XmlEnvelopeParam;
                    break;
            }

            if (envelopeXml != null)
            {
                IDictionary<string, XmlParameter> envelopeParams = this.CreateParameterDictionary(envelopeXml.Parameters);

                channel.SetEnvelope(
                    (envelopeParams[XmlParameterNames.Envelope.Attack] as XmlEnvelopeAttackParam).Value,
                    (envelopeParams[XmlParameterNames.Envelope.Hold] as XmlEnvelopeHoldParam).Value,
                    (envelopeParams[XmlParameterNames.Envelope.Decay] as XmlEnvelopeDecayParam).Value,
                    (envelopeParams[XmlParameterNames.Envelope.Sustain] as XmlEnvelopeSustainParam).Value,
                    (envelopeParams[XmlParameterNames.Envelope.Release] as XmlEnvelopeReleaseParam).Value
                );
            }

            channel.SetKey(
                noteOnInfo.key,
                (velocityRegionParams[XmlParameterNames.OriginalKey]
                as XmlOriginalKeyParam).Value);

            int instrumentVolume = CalculateInstrumentVolume(
                (instrumentParams[XmlParameterNames.Volume] as XmlVolumeParam).Value,
                (velocityRegionParams[XmlParameterNames.Volume] as XmlVolumeParam).Value);
            channel.SetInstrumentVolume(instrumentVolume * INSTRUMENT_VOLUME_CRITERION_R);
            channel.SetVelocity(noteOnInfo.velocity * VELOCITY_MAX_R);


            double tune = 1.0;
            tune *= Math.Pow(2.0, (instrumentParams[XmlParameterNames.PitchCents] as XmlPitchCentsParam).Value / 1200.0);
            tune *= Math.Pow(2.0, (velocityRegionParams[XmlParameterNames.PitchCents] as XmlPitchCentsParam).Value / 1200.0);
            tune *= Math.Pow(2.0, (instrumentParams[XmlParameterNames.PitchSemitones] as XmlPitchSemitonesParam).Value / 12.0);
            tune *= Math.Pow(2.0, (velocityRegionParams[XmlParameterNames.PitchSemitones] as XmlPitchSemitonesParam).Value / 12.0);
            channel.SetTune((float)tune);

            int pan = noteOnInfo.initPan;
            if (keyRegionParams.ContainsKey(XmlParameterNames.Pan))
            {
                pan += (keyRegionParams[XmlParameterNames.Pan] as XmlPanParam).Value - 64;
            }
            pan += (velocityRegionParams[XmlParameterNames.Pan] as XmlPanParam).Value - 64;
            channel.SetInitPan(pan / 63.0f);

            _channel = channel;
            _waveInfo = (waveFileInfo as WaveFileInfo).ToWaveInfo(0, waveFileInfo.WaveFile.ChannelCount);
            _noteLength = noteOnInfo.length;
        }

        private void Initialize(string bankFileName, Instrument instrument, ref Runtime.NoteOnInfo noteOnInfo, Runtime.ChannelCafe.RendererType rendererType)
        {
            if (null == bankFileName) { throw new ArgumentNullException("bankFileName"); }
            if (null == instrument) { throw new ArgumentNullException("instrument"); }

            KeyRegion keyRegion = FindXmlKeyRegion(instrument, noteOnInfo.key);
            if (null == keyRegion) { return; }

            VelocityRegion velocityRegion = FindXmlVelocityRegion(keyRegion, noteOnInfo.velocity);
            if (null == velocityRegion) { return; }

            string filePath = Path.Combine(Path.GetDirectoryName(bankFileName), velocityRegion.FilePath);

            int? sampleRate = null;
            if (instrument.IsResampleEnabled)
            {
                sampleRate = instrument.SampleRate;
            }

            int? channelCount = null;
            if (instrument.IsDownMixEnabled == true)
            {
                channelCount = 1; // モノラル化
            }

            WaveFileInfo waveFileInfo = (WaveFileInfo)WaveFileManager.Instance.LoadFile(filePath, null, null, null, sampleRate, channelCount);

            if (waveFileInfo == null) { return; }


            Runtime.ChannelCafe channel = new Runtime.ChannelCafe();
            channel.AllocChannel(
                waveFileInfo.WaveFile.ChannelCount,
                noteOnInfo.priority,
                noteOnInfo.channelCallback,
                noteOnInfo.channelCallbackData,
                rendererType
            );

            Envelope envelope = null;

            switch (instrument.EnvelopeMode)
            {
                case InstrumentEnvelopeMode.Instrument:
                    envelope = instrument.Envelope;
                    break;

                case InstrumentEnvelopeMode.VelocityRegion:
                    envelope = (Envelope)velocityRegion.Envelope;
                    break;

                case InstrumentEnvelopeMode.Lower:
                    if (velocityRegion.Envelope != null)
                    {
                        envelope = (Envelope)velocityRegion.Envelope;
                    }
                    else
                    {
                        envelope = (Envelope)instrument.Envelope;
                    }
                    break;
            }
            if (envelope != null)
            {
                channel.SetEnvelope(
                    envelope.Attack,
                    envelope.Hold,
                    envelope.Decay,
                    envelope.Sustain,
                    envelope.Release
                );
            }

            channel.SetKey(noteOnInfo.key, velocityRegion.OriginalKey);
            int instrumentVolume = CalculateInstrumentVolume(instrument.Volume, velocityRegion.Volume);
            channel.SetInstrumentVolume(instrumentVolume * INSTRUMENT_VOLUME_CRITERION_R);
            channel.SetVelocity(noteOnInfo.velocity * VELOCITY_MAX_R);

            double tune = 1.0;
            tune *= Math.Pow(2.0, instrument.PitchCents / 1200.0);
            tune *= Math.Pow(2.0, velocityRegion.PitchCents / 1200.0);
            tune *= Math.Pow(2.0, instrument.PitchSemitones / 12.0);
            tune *= Math.Pow(2.0, velocityRegion.PitchSemitones / 12.0);
            channel.SetTune((float)tune);

            int pan = noteOnInfo.initPan;
            pan += velocityRegion.Pan - 64;
            channel.SetInitPan(pan / 63.0f);

            if (velocityRegion.PercussionMode)
            {
                channel.SetReleaseIgnore(true);
            }

            channel.SetAlternateAssignId(velocityRegion.KeyGroup);

            channel.SetInterpolationType((int)velocityRegion.InterpolationType);

            _channel = channel;
            _waveInfo = (waveFileInfo as WaveFileInfo).ToWaveInfo(0, waveFileInfo.WaveFile.ChannelCount);
            _noteLength = noteOnInfo.length;
        }

        private int CalculateInstrumentVolume(int instrumentVolume, int velocityRegionVolume)
        {
            double rate = ((double)instrumentVolume / 127.0f) * ((double)velocityRegionVolume / 127.0f);
            int volume = (int)(127.0f * rate);
            volume = Math.Min(volume, 255);
            volume = Math.Max(volume, 0);

            return volume;
        }
    }
}
