﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.FileFormats
{
    using System;
    using System.Collections;
    using System.Diagnostics;
    using System.IO;
    using System.Runtime.InteropServices;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using SoundFoundation.Conversion.NintendoWareBinary;
    using Nintendo.Foundation.Contracts;
    using SoundFoundation.Resources;
    using SoundFoundation.Conversion;

    public class WaveFileInfo : IDisposable, IWaveFileInfo
    {
        private const int WaveBufferAlignment = 64;

        private bool isDisposed = false;
        private WaveFile waveFile;
        private DateTime lastModifiedTime;
        private string filePath;
        private GCHandle[] waveDataPtrArray = null;
        private int[] waveDataOffsetArray = null;

        ~WaveFileInfo()
        {
            this.Dispose();
        }

        public WaveFile WaveFile
        {
            get { return waveFile; }
        }
        public DateTime LastModifiedTime
        {
            get { return lastModifiedTime; }
        }
        public string FilePath
        {
            get { return filePath; }
        }

        public bool IsLoop { get; private set; }

        /// <summary>
        /// ループ開始フレームを取得します。（プリプロセス後）
        /// </summary>
        public long LoopStartFrame { get; private set; }

        /// <summary>
        /// ループ終了フレームを取得します。（プリプロセス後）
        /// </summary>
        public long LoopEndFrame { get; private set; }

        public Int32 GetWaveDataAddr(int channel)
        {
            return waveDataPtrArray[channel].AddrOfPinnedObject().ToInt32() + waveDataOffsetArray[channel];
        }

        public void Invalidate()
        {
            FreeWaveData();
            waveDataPtrArray = null;
            waveDataOffsetArray = null;
        }

        public void Open(string filePath)
        {
            Open(filePath, null, null, null, null, null);
        }

        public void Open(string filePath, bool? isLoop, int? loopStartFrame, int? loopEndFrame, int? sampleRate, int? channelCount)
        {
            lastModifiedTime = File.GetLastWriteTime(filePath);

            var targetFilePath = this.Preprocess(filePath, isLoop, loopStartFrame, loopEndFrame, sampleRate, channelCount);

            using (WaveFileReader reader = WaveFileReader.CreateInstance(targetFilePath))
            {
                waveFile = reader.Open(targetFilePath);

                Ensure.True(waveFile.FrameCount > 0, () => new ApplicationException("FrameCount == 0"));

                WaveFormat waveFormat = new WaveFormat()
                {
                    BitsPerSample = waveFile.SampleBit,
                    ChannelCount = waveFile.ChannelCount,
                    Encoding = Encoding.Pcm,
                    HasLoop = this.IsLoop,
                    IsLittleEndian = waveFile is WaveFileWav,
                    IsSigned = !((waveFile is WaveFileWav) && waveFile.SampleBit == 8),
                    OriginalLoopStartFrame = (int)this.LoopStartFrame,
                    OriginalLoopEndFrame = (int)this.LoopEndFrame,
                    LoopStartFrame = (int)this.LoopStartFrame,
                    LoopEndFrame = (int)this.LoopEndFrame,
                    SamplingRate = waveFile.SampleRate,
                };

                NintendoWareWaveEncoder encoder = NintendoWareWaveEncoderFactory.Instance.Create(Encoding.Pcm);

                using (var waveStream = new WaveStream(waveFormat, reader.OpenDataStream()))
                {
                    CodecOutput[] codecOutputs = encoder.Run(waveStream);

                    // Dirty チェックに利用されるので、プリプロセス前のパスを入れる
                    this.filePath = filePath;

                    // エミュレータ用データの作成
                    this.FreeWaveData();

                    waveDataPtrArray = new GCHandle[codecOutputs.Length];
                    waveDataOffsetArray = new int[codecOutputs.Length];

                    for (int i = 0; i < codecOutputs.Length; i++)
                    {
                        var data = new byte[codecOutputs[i].Data.Length + WaveBufferAlignment];
                        var handle = GCHandle.Alloc(data, GCHandleType.Pinned);

                        int offset = WaveBufferAlignment - (int)(handle.AddrOfPinnedObject().ToInt64() % WaveBufferAlignment);
                        if (offset == WaveBufferAlignment)
                        {
                            offset = 0;
                        }

                        Array.Copy(codecOutputs[i].Data, 0, data, offset, codecOutputs[i].Data.Length);
                        waveDataPtrArray[i] = handle;
                        waveDataOffsetArray[i] = offset;
                    }
                }
            }
        }

        public void Dispose()
        {
            if (this.isDisposed)
            {
                return;
            }

            this.isDisposed = true;

            this.FreeWaveData();
        }

        private void FreeWaveData()
        {
            if (this.waveDataPtrArray != null)
            {
                foreach (GCHandle handle in this.waveDataPtrArray)
                {
                    handle.Free();
                }

                this.waveDataPtrArray = null;
            }
        }

        private string Preprocess(string filePath, bool? isLoop, int? loopStartFrame, int? loopEndFrame, int? sampleRate, int? channelCount)
        {
            var result = filePath;

            using (var reader = WaveFileReader.CreateInstance(filePath))
            {
                WaveFile srcFile;

                srcFile = reader.Open(filePath);

                int sampleRateValue = srcFile.SampleRate;
                int channelCountValue = srcFile.ChannelCount;

                // サンプルレートが指定されている場合
                if (sampleRate.HasValue == true)
                {
                    sampleRateValue = sampleRate.Value;

                    // ダウンサンプルになっていない（アップサンプルは禁止）
                    if (srcFile.SampleRate < sampleRate)
                    {
                        throw new ApplicationException(string.Format(
                            MessageResourceCommon.Message_UpsampleIsNotSupported,
                            sampleRate.Value,
                            srcFile.SampleRate));
                    }

                    // サンプルレートが範囲外
                    if (sampleRate < ConversionTraits.MinSampleRate || ConversionTraits.MaxSampleRate < sampleRate)
                    {
                        var message = string.Format(
                            MessageResourceCommon.Message_SampleRateMustBeValidRange,
                            ConversionTraits.MinSampleRate,
                            ConversionTraits.MaxSampleRate);
                        throw new ApplicationException(message);
                    }

                    // 指定されたループ情報をそのまま利用する or 波形のループ情報をサンプルレートでスケールして利用する
                    if (isLoop == false)
                    {
                        this.IsLoop = false;
                    }
                    else if (isLoop == true && 0 <= loopStartFrame && loopStartFrame < loopEndFrame)
                    {
                        this.LoopStartFrame = loopStartFrame.Value;
                        this.LoopEndFrame = loopEndFrame.Value;
                        this.IsLoop = true;
                    }
                    else if (srcFile.IsLoop && 0 <= srcFile.LoopStartFrame && srcFile.LoopStartFrame < srcFile.LoopEndFrame)
                    {
                        this.LoopStartFrame = srcFile.LoopStartFrame * sampleRate.Value / srcFile.SampleRate;
                        this.LoopEndFrame = srcFile.LoopEndFrame * sampleRate.Value / srcFile.SampleRate;
                        this.IsLoop = true;
                    }
                }
                else
                {
                    // 指定されたループ情報 or 波形のループ情報をそのまま利用する
                    if (isLoop == false)
                    {
                        this.IsLoop = false;
                    }
                    else if (isLoop == true && 0 <= loopStartFrame && loopStartFrame < loopEndFrame)
                    {
                        this.LoopStartFrame = (long)loopStartFrame.Value;
                        this.LoopEndFrame = (long)loopEndFrame.Value;
                        this.IsLoop = true;
                    }
                    else if (srcFile.IsLoop && 0 <= srcFile.LoopStartFrame && srcFile.LoopStartFrame < srcFile.LoopEndFrame)
                    {
                        this.LoopStartFrame = srcFile.LoopStartFrame;
                        this.LoopEndFrame = srcFile.LoopEndFrame;
                        this.IsLoop = true;
                    }
                }

                // チャンネル数が指定されている場合（＝現時点ではモノラル化が指定されている場合）
                if (channelCount.HasValue == true)
                {
                    // 現時点では Assert のチェックをするだけ。モノラル化のみ。
                    Debug.Assert(channelCount == 1);

                    channelCountValue = channelCount.Value;
                }

                if (sampleRate.HasValue == true || channelCount.HasValue == true)
                {
                    if (string.IsNullOrEmpty(WaveFileManager.Instance.PreprocessExePath))
                    {
                        throw new ApplicationException(MessageResourceCommon.Message_SoxExePathIsEmpty);
                    }

                    if (!File.Exists(WaveFileManager.Instance.PreprocessExePath))
                    {
                        throw new ApplicationException(string.Format(
                            MessageResourceCommon.Message_SoxExeNotFound,
                            WaveFileManager.Instance.PreprocessExePath));
                    }

                    var outputFilePath = this.GetTempFileName("wav");
                    SoxExecutor.Run(WaveFileManager.Instance.PreprocessExePath, filePath, outputFilePath, sampleRateValue, channelCountValue);

                    result = outputFilePath;
                }
            }

            return result;
        }

        private string GetTempFileName(string extension)
        {
            while (true)
            {
                var filePath = Path.ChangeExtension(Path.GetTempFileName(), extension);

                if (!File.Exists(filePath))
                {
                    return filePath;
                }
            }
        }
    }
}
