﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "Atk.h"
#include <nn/audio.h>
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#include <nn/audio/audio_AudioRendererApi.private.h>
#include <nn/atkTool/detail/HardwareManagerWrapper.h>

namespace {
nn::os::Event g_FastRenderedEvent(nn::os::EventClearMode_AutoClear);
void* g_WorkBuffer = nullptr;
}

namespace NintendoWare { namespace SoundRuntime {

int Global::AXSynthesize( array<short>^ buffer, int samplingRate )
{
    // １オーディオフレーム レンダリングする
    {
        nn::atk::detail::driver::SoundThreadLock lock;
        nn::atkTool::detail::HardwareManagerWrapper::RequestUpdateAudioRenderer();
    }

    nn::audio::TriggerRendering();
    g_FastRenderedEvent.Wait();

    // CircularBufferSink のバッファ状態を更新するために、RequestUpdateAudioRenderer()
    {
        nn::atk::detail::driver::SoundThreadLock lock;
        nn::atkTool::detail::HardwareManagerWrapper::RequestUpdateAudioRenderer();
    }

    auto size = nn::atk::SoundSystem::GetCircularBufferSinkBufferSize();
    if((int)size < buffer->Length)
    {
        throw gcnew System::Exception();
    }

    pin_ptr<short> pinedBuffer = &buffer[0];

    return nn::atk::SoundSystem::ReadCircularBufferSink(
        pinedBuffer,
        System::Math::Min(size, buffer->Length * sizeof(short)));
}

void AtkSoundSystem::InitSoundSystem(int soundThreadPriority, int dvdThreadPriority)
{
    nn::atk::SoundSystem::SoundSystemParam param;

    param.soundThreadPriority = soundThreadPriority;
    param.taskThreadPriority = dvdThreadPriority;
    param.enableCircularBufferSink = true;
    param.recordingAudioFrameCount = 32;

    size_t workMemSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_WorkBuffer = new uint8_t[ workMemSize ];

    if (nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_WorkBuffer ),
        workMemSize) == false)
    {
        throw gcnew System::Exception("Can not Initialize. nn::atk::SoundSystem::Initialize() == false");
    }
    nn::audio::SetMemoryPoolErrorCheckEnabled(&nn::atk::SoundSystem::GetAudioRendererConfig(), false);
}

void AtkSoundSystem::ShutdownSoundSystem()
{
    nn::atk::SoundSystem::Finalize();
    delete[] reinterpret_cast<uint8_t*>( g_WorkBuffer );
}

int AtkSoundSystem::GetActiveVoiceCount()
{
    return nn::atk::detail::driver::MultiVoiceManager::GetInstance().GetActiveCount();
}

void AtkSoundSystem::StopAllVoices()
{
    nn::atk::detail::driver::MultiVoiceManager::GetInstance().StopAllVoices();
}

void AtkSoundSystem::LockSoundThread()
{
    nn::atk::detail::driver::SoundThread::GetInstance().Lock();
}

void AtkSoundSystem::UnlockSoundThread()
{
    nn::atk::detail::driver::SoundThread::GetInstance().Unlock();
}

void AtkSoundSystem::SoundFrameProcess()
{
    nn::atk::detail::driver::SoundThread::GetInstance().UpdateElapsedFrameCount();
    nn::atk::detail::driver::SoundThread::GetInstance().UpdateLowLevelVoices();
    nn::atk::detail::driver::SoundThread::GetInstance().FrameProcess();
    nn::atk::detail::driver::SoundThread::GetInstance().UpdateLowLevelVoicesParam();
}

void AtkSoundSystem::SoundThreadPause( bool pauseFlag )
{
    if(pauseFlag)
    {
        nn::atk::detail::driver::SoundThread::GetInstance().Pause( true );

        // SoundThread を利用せずに FrameProcess 処理を回せるように SoundThread が確実に Pause するタイミングまで待機する
        // 最大 2 オーディオフレーム分のレンダリング処理が完了するまで待てば OK
        nn::atkTool::detail::HardwareManagerWrapper::WaitAudioRendererEvent();
        nn::atkTool::detail::HardwareManagerWrapper::WaitAudioRendererEvent();

        // FastRenderingMode に切り替えて、CircularBufferSink のバッファをクリアしておく
        nn::audio::SetFastRenderingMode(true, &g_FastRenderedEvent);
        g_FastRenderedEvent.Wait();

        {
            nn::atk::detail::driver::SoundThreadLock lock;
            nn::atkTool::detail::HardwareManagerWrapper::RequestUpdateAudioRenderer();
        }

        uint8_t work[1024];
        while(nn::atk::SoundSystem::ReadCircularBufferSink(work, sizeof(work)) > 0) { }
    }
    else
    {
        nn::audio::SetFastRenderingMode(false, nullptr);

        nn::atk::detail::driver::SoundThread::GetInstance().Pause( false );
    }
}

}}
