﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <crtdbg.h>
#include <iomanip>
#include <fstream>  // ofstream

#include <nn/utilTool/utilTool_CommandFramework.h>

#include <sndlib/path.h>
#include <sndlib/stream.h>

#include "smfconv.h"
#include "SmfParser.h"

#define VERSION "0.1.1"
#define EXT_NAME ".fseq"

#define PATTERN_EVENT_MIN 8 // パターン小節の最低イベント数

extern void test();

using namespace sndlib;

// TODO: put ではパラメータの範囲に注意すること
//       ２つ以上のコマンドを出力しなければならないこともある


// TODO: 曲の先頭休符を削除しないオプション
// TODO: パターン圧縮をしないオプション

namespace // anonymous
{

bool CompareEvent( const Event* lhs, const Event* rhs )
{
    if ( lhs->tick == rhs->tick ) {
        if ( lhs->isMetaEvent() && ! rhs->isMetaEvent() ) return true;
        return false;
    }

    return lhs->tick < rhs->tick;
}

} // end of anonymous namespace


SmfParser* sParser = NULL;

ChannelEventException::ChannelEventException(const std::string& _S, int _channel, int _tick)
: ParseException(_S)
{
    if ( sParser == NULL )
    {
        channel = _channel;
        measure = 0;
        beat = 0;
        tick = _tick;
    }
    else
    {
        BeatInfo info;
        sParser->mBeatStream.tickToBeat( _tick, &info );
        channel = _channel;
        measure = info.measure;
        beat = info.beat;
        tick = info.tick;
    }
}

MetaEventException::MetaEventException(const std::string& _S, int _tick)
: ParseException(_S)
{
    BeatStream bs;
    BeatInfo info;
    bs.tickToBeat( _tick, &info );
    measure = info.measure;
    beat = info.beat;
    tick = info.tick;
}

ControlChangeEventException::ControlChangeEventException(const std::string& _S, int _cc, int _value, int _tick)
: ParseException(_S)
{
    BeatStream bs;
    BeatInfo info;
    bs.tickToBeat( _tick, &info );
    measure = info.measure;
    beat = info.beat;
    tick = info.tick;
    controlChange = _cc;
    value = _value;
}

const std::string MakeTrackLabelPrefix( const std::string& prefix, int number )
{
    std::stringstream s;
    s << prefix << "_Track_" << number;
    return s.str();
}

const std::string MakeLocalTrackLabelPrefix( int number )
{
    std::stringstream s;
    s << "Track_" << number;
    return s.str();
}

class CommandlineParameters
{
public:
    CommandlineParameters() : m_TimeBase(0), m_IsHelp(false), m_IsUpdate(false), m_IsVerbose(false)
    {
    }

    void Initialize(nn::utilTool::SingleCommandInterface &commandInterface)
    {
        commandInterface.AddHiPriorityFlagArgument(
            nn::utilTool::FlagArgument(
                std::shared_ptr<nn::utilTool::FlagArgumentStore>(new nn::utilTool::BoolFlagArgumentStore(&m_IsHelp)),
                nn::utilTool::KeywordArgumentName('h', "help")),
            nn::utilTool::KeywordArgumentDocument('h', "help", false, "", "Display help.", false));

        commandInterface.AddKeywordArgument(&m_IncludeDirectoryPaths, "include-dir", "add the directory to the #include<> search path", false);
        commandInterface.AddKeywordArgument(&m_OutputFilePath, 'o', "output", "specify output filename", false);
        commandInterface.AddKeywordArgumentWithDefaultValue(&m_TimeBase, "timebase", "specify timebase", 48);
        commandInterface.AddFlagArgument(&m_IsUpdate, 'u', "update", "update only when input file is newer than output file");
        commandInterface.AddFlagArgument(&m_IsVerbose, 'v', "verbose", "explain what is being done");
        commandInterface.SetVariableArguments(&m_InputFilePaths, "files", "intput files", true);
    }

    bool IsHelp() const { return m_IsHelp; }
    bool IsUpdate() const { return m_IsUpdate; }
    bool IsVerbose() const { return m_IsVerbose; }

    int GetTimeBase() const { return m_TimeBase; }

    const std::vector<std::string>& GetInputFilePaths() const { return m_InputFilePaths; }
    const std::vector<std::string>& GetIncludeDirectoryPaths() const { return m_IncludeDirectoryPaths; }
    const std::string& GetOutputFilePath() const { return m_OutputFilePath; }

private:
    std::vector<std::string> m_InputFilePaths;
    std::vector<std::string> m_IncludeDirectoryPaths;
    std::string m_OutputFilePath;

    int m_TimeBase;

    bool m_IsHelp;
    bool m_IsUpdate;
    bool m_IsVerbose;
};

void PrintUsage(nn::utilTool::SingleCommandInterface &commandInterface)
{
    std::string helpText = MakePlainTextCommandDocument(commandInterface.GetDocument());
    std::cout << helpText.c_str();
}

int Main( int argc, char* argv[] )
{
#ifdef _DEBUG
    test();
#endif

    setlocale( LC_ALL, "" );

    CommandLineArg cmdinfo;

    {
        nn::utilTool::DefaultCommandLogObserver logObserver;
        nn::utilTool::RegisterCommandLogObserver(logObserver);
        NN_UTIL_SCOPE_EXIT{ UnregisterCommandLogObserver(logObserver); };

        CommandlineParameters params;
        nn::utilTool::SingleCommandInterface commandInterface;

        commandInterface.SetName("SmfConverter.exe");
        commandInterface.SetSummary("");

        params.Initialize(commandInterface);

        bool result = commandInterface.TryParse(argc - 1, const_cast<const char**>(&argv[1]));
        if (!result)
        {
            PrintUsage(commandInterface);
            return 1;
        }

        if (params.IsHelp())
        {
            PrintUsage(commandInterface);
            return 0;
        }

        for (auto it = params.GetInputFilePaths().begin();
            it != params.GetInputFilePaths().end();
            ++it)
        {
            cmdinfo.files.push_back(*it);
        }

        cmdinfo.out_file = params.GetOutputFilePath();

        cmdinfo.timebase = params.GetTimeBase();

        cmdinfo.update_flag = params.IsUpdate();
        cmdinfo.verbose_flag = params.IsVerbose();

        if (cmdinfo.verbose_flag)
        {
            logObserver.SetLevel(nn::utilTool::CommandLogLevel_Verbose);
        }
        else
        {
            logObserver.SetLevel(nn::utilTool::CommandLogLevel_Error);
        }
    }

    std::vector<std::string>::const_iterator it;
    for( it = cmdinfo.files.begin() ; it != cmdinfo.files.end() ; ++it )
    {
        const std::string infile = *it;
        const std::string basename = infile.substr(0, infile.rfind('.'));
        const std::string outfile = ( cmdinfo.out_file.empty() ) ? basename + EXT_NAME : cmdinfo.out_file ;
        std::string labelPrefix = path::GetBaseName( infile );
        for( std::string::iterator p = labelPrefix.begin() ; p != labelPrefix.end() ; p++ ) {
            if ( *p != '_' && ! isalnum( *p ) ) {
                *p = '_';
            }
        }
        if ( labelPrefix[0] != '_' ) labelPrefix = "_" + labelPrefix;
        labelPrefix = "SMF" + labelPrefix;

        if ( cmdinfo.update_flag && util::CompareFileTime( infile, outfile ) > 0 ) {
            if ( cmdinfo.verbose_flag ) {
                std::cout << "smfconv: " << outfile << " is update." << std::endl;
            }
            continue;
        }

        SmfParser parser;

        sParser = &parser;

        // SMFの解析
        try {
            parser.parse( infile.c_str(), cmdinfo, labelPrefix );
        }
        catch ( ParseException &e) {
            e.filename = infile;
            throw;
        }

        std::ofstream out( outfile.c_str() );
        if ( ! out ) {
            std::stringstream s;
            s << "cannot open " << outfile << " for writing";
            throw my_exception( s.str() );
        }

        try {
            unsigned long trackBitMask = ( 1 << 0 );

            tick_t head_wait = ~0UL;
            for(int ch = 0 ; ch < CHANNEL_MAX ; ch++ )
            {
                EventStream& strm = parser.mMidiEventStream[ ch ];

                bool isAvailable = false;
                for( EventStream::const_iterator p = strm.begin();
                     p != strm.end() ; p++ ) {
                    if ( (*p)->isImportantEvent() ) {
                        isAvailable = true;
                        break;
                    }
                }

                if ( ! isAvailable ) continue;

                trackBitMask |= ( 1 << ch );

                // ソート
                stable_sort( strm.begin(), strm.end(), CompareEvent );

                // 先頭の空白を検出
                for( EventStream::const_iterator p = strm.begin() ; p != strm.end() ; ++p )
                {
                    Event* e = *p;

                    if ( e->isFixedEvent() )
                    {
                        if ( head_wait > e->tick ) head_wait = e->tick;
                        break;
                    }
                }
            }
            head_wait = head_wait * cmdinfo.timebase / parser.mDivison;


            // ヘッダ
            out
                << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                << ";" << std::endl
                << "; " << outfile << std::endl
                << ";     generated by smfconv" << std::endl
                << ";" << std::endl
                << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                << std::endl
                << labelPrefix << "_Begin:" << std::endl;
            ;

            // トラックリスト出力
            {
                if ( trackBitMask != 0x0001 ) {
                    out
                        << "    alloctrack 0x"
                        << std::hex << std::setw(4) << std::setfill('0') << trackBitMask << std::dec
                        << std::endl;
                }

                out << labelPrefix << "_Start:" << std::endl;

                if ( cmdinfo.timebase != 48 ) {
                    out << "    timebase " << cmdinfo.timebase << std::endl;
                }

                for(int ch = 1 ; ch < CHANNEL_MAX ; ch++ )
                {
                    if ( trackBitMask & ( 1 << ch ) ) {
                        out << "    opentrack " << ch << ", " << MakeTrackLabelPrefix( labelPrefix, ch ) << std::endl;
                    }
                }
            }

            // MIDIチャンネル毎の処理
            for(int ch = 0 ; ch < CHANNEL_MAX ; ch++ )
            {
                if ( ( trackBitMask & ( 1 << ch ) ) == 0 ) continue;

                EventStream& strm = parser.mMidiEventStream[ ch ];

                tick_t head_wait_rest = 0;
                if ( 1 ) {
                    head_wait_rest = head_wait;
                }

                BeatInfo beat;

                // TODO: グローバルイベント結合のため、再確認が必要
                if ( strm.back()->tick > strm.lastTick ) strm.lastTick = strm.back()->tick;

                // 小節数の取得
                parser.mBeatStream.tickToBeat( strm.lastTick, &beat);
                const unsigned int measure_count = beat.measure + 1;

                SmfTrack track( measure_count );

                // wait イベントを挿入しながら、小節毎にイベントを分割する

                EventStream::const_iterator event_p = strm.begin();
                BeatStream::const_iterator beat_p = parser.mBeatStream.begin();
                const BeatEvent* cur_beat = *beat_p; // Valid because stream is NOT empty
                unsigned int rest_measure = cur_beat->measure;
                for ( unsigned int measure = 0; measure < measure_count ; ++measure, --rest_measure )
                {
                    // ビート情報を更新します。
                    while ( rest_measure == 0 ) {
                        ++beat_p;
                        cur_beat = *beat_p;
                        rest_measure = cur_beat->measure;
                    }

                    unsigned int measure_wait =
                        cur_beat->division * cur_beat->numerator
                        * cmdinfo.timebase / parser.mDivison;
                    unsigned int cur_wait = 0;
                    const tick_t cur_tick = parser.mBeatStream.measureToTick( measure );
                    if ( measure == measure_count - 1 )
                    {
                        tick_t diff = strm.lastTick - cur_tick;
                        measure_wait = diff * cmdinfo.timebase / parser.mDivison;
                    }

                    while ( event_p != strm.end() ){
                        const Event* event = *event_p;
                        parser.mBeatStream.tickToBeat( event->tick, &beat );
                        if ( beat.measure != measure) break;

                        unsigned int ofs = event->tick - cur_tick;
                        ofs = ofs * cmdinfo.timebase / parser.mDivison;
                        if ( ofs > measure_wait ) ofs = measure_wait;

                        unsigned int diff = ofs - cur_wait;
                        if ( diff > 0 ) {
                            if ( head_wait_rest < static_cast<unsigned long>(diff) ) {
                                track[ measure ].push_back( new WaitEvent( cur_tick + cur_wait * parser.mDivison / cmdinfo.timebase, diff - head_wait_rest ) );
                                head_wait_rest = 0;
                            }
                            else {
                                head_wait_rest -= diff;
                            }
                            cur_wait += diff;
                        }

                        // MIDIイベントのコピー
                        Event* new_event = event->clone();

                        // ノートコマンドの長さを調整します。
                        new_event->updateDuration( parser.mDivison, cmdinfo.timebase );

                        track[ measure ].push_back( new_event );

                        ++event_p;
                    }

                    const unsigned int rest_wait = measure_wait - cur_wait;

                    // 小節の残りの部分のウェイトコマンド
                    if ( rest_wait > 0 ) {
                        if ( head_wait_rest < rest_wait ) {
                            track[ measure ].push_back( new WaitEvent( cur_tick + cur_wait * parser.mDivison / cmdinfo.timebase, rest_wait - head_wait_rest ) );
                            head_wait_rest = 0;
                        }
                        else{
                            head_wait_rest -= rest_wait;
                        }
                    }

                } // End of 小節毎の処理


                /////////////////
                // パターン圧縮
                /////////////////
                PatternArray patternArray;
                if ( 1 )
                {

                    int pattern_no = 0;

                    for( unsigned int m1 = 0;
                         m1 < measure_count - 1 ; // 最後の小節は探索対象が無いのではじく
                         ++m1 )
                    {
                        Measure& lhs = track[ m1 ];
                        if ( lhs.isPattern ) continue;

                        if ( lhs.size() < PATTERN_EVENT_MIN ) continue;

                        Pattern* pattern = NULL;

                        for ( unsigned int m2 = m1 + 1 ; m2 < measure_count ; ++m2 )
                        {
                            Measure& rhs = track[ m2 ];
                            if ( rhs.isPattern ) continue;

                            if (! lhs.isEqual( rhs ) ) continue;

                            if ( pattern == NULL ) {
                                std::ostringstream str;
                                str << MakeTrackLabelPrefix( labelPrefix, ch ) << "_" << pattern_no;
                                pattern_no++;
                                pattern = new Pattern( str.str().c_str() );
                                pattern->assign( lhs.begin(), lhs.end() );
                                patternArray.push_back( pattern );
                            }

                            // パターンイベントに置き換え
                            for_each( rhs.begin(), rhs.end(), util::DeleteObject());
                            rhs.clear();
                            rhs.push_back( new PatternEvent(0, pattern) );
                            rhs.isPattern = true;
                        }

                        if (pattern != NULL)
                        {
                            // パターンイベントに置き換え
                            // 　イベントの所有権は、Pattern オブジェクトに移行済みのため
                            // 　delete の必要は無し
                            lhs.clear();
                            lhs.push_back( new PatternEvent(0, pattern) );
                            lhs.isPattern = true;
                        }
                    }
                }

                ///////////////
                // waitの結合
                ///////////////

                // 正順処置

                EventStream ostrm;
                WaitEvent* wait = NULL;
                for ( unsigned int measure = 0 ; measure < measure_count ; ++measure )
                {
                    std::ostringstream str;
                    str << "; Measure " << measure+1 << " ----------------------------------";
                    ostrm.push_back( new TextEvent( 0, str.str() ) );

                    Measure& m = track[ measure ];
                    for(Measure::iterator p = m.begin() ; p != m.end(); ++p )
                    {
                        if ( WaitEvent* e = dynamic_cast<WaitEvent*>(*p) )
                        {
                            if ( wait == NULL )
                            {
                                wait = e->clone();
                                ostrm.push_back( wait );
                            }
                            else
                            {
                                wait->duration += e->duration;
                            }
                        }
                        else
                        {
                            ostrm.push_back( (*p)->clone() );
                            wait = NULL;
                        }
                    }

                }

                ///////////////
                // テキスト出力
                ///////////////
                out
                    << std::endl
                    << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                    << "; Track " << ch << std::endl
                    << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                    << std::endl
                    << MakeTrackLabelPrefix( labelPrefix, ch ) << ":" << std::endl
                    ;

                // イベント出力
                out << "    notewait_off" << std::endl;
                for ( EventStream::const_iterator p = ostrm.begin() ; p != ostrm.end() ; ++p )
                {
                    out << **p << std::endl;
                }
                out << "    fin" << std::endl;

                if (! patternArray.empty() )
                {
                    out
                        << std::endl
                        << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                        << "; Patterns" << std::endl
                        << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl
                        << std::endl;
                }

                for (
                    PatternArray::const_iterator patt = patternArray.begin() ;
                    patt != patternArray.end() ;
                    ++patt
                )
                {
                    const Pattern* pattern = *patt;

                    out << pattern->label << ":" << std::endl;

                    for ( Pattern::const_iterator p = pattern->begin() ; p != pattern->end() ; ++p )
                    {
                        out << **p << std::endl;
                    }

                    out << "    ret" << std::endl << std::endl;
                }


            }

            out
                << std::endl
                << labelPrefix << "_End:" << std::endl
                ;

        }
        catch( ... ) {
            out.close();
            remove( outfile.c_str() );
            throw;
        }
    }

    return 0;
}

int main(int argc, char* argv[])
{
    int retCode = 0;

    int tmpDbgFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
    tmpDbgFlag |= _CRTDBG_DELAY_FREE_MEM_DF;
    tmpDbgFlag |= _CRTDBG_LEAK_CHECK_DF;
    _CrtSetDbgFlag(tmpDbgFlag);

    try {
        retCode = Main( argc, argv );
    }
    catch ( const MetaEventException& e ) {
        std::cerr << "smfconv: " << e.filename << ": " << e.what() << std::endl
             << "         [ measure:" << e.measure
             << " beat:" << e.beat
             << " tick:" << e.tick
             << " ]" << std::endl
             ;
        retCode = 1;
    }
    catch ( const ChannelEventException& e ) {
        std::cerr << "smfconv: " << e.filename << ": " << e.what() << std::endl
             << "         [ channel:" << e.channel
             << " measure:" << e.measure
             << " beat:" << e.beat
             << " tick:" << e.tick
             << " ]" << std::endl
             ;
        retCode = 1;
    }
    catch ( const ControlChangeEventException& e ) {
        std::cerr << "smfconv: " << e.filename << ": " << e.what()
             << " (CC:" << e.controlChange
             << " value:" << e.value << ")" << std::endl
             << "         [ measure:" << e.measure
             << " beat:" << e.beat
             << " tick:" << e.tick
             << " ]" << std::endl
             ;
        retCode = 1;
    }
    catch ( const ParseException& e ) {
        std::cerr << "smfconv: " << e.filename << ": " << e.what() << std::endl;
        retCode = 1;
    }
    catch ( const CommandLineArgParserException& ) {
        // メッセージは Nintendo::Foundation::IO::CommandLineParser が出力してくれます。
        retCode = 1;
    }
    catch ( const my_exception &e) {
        std::cerr
            << "smfconv: " << e.what() << std::endl
            ;

        retCode = 1;
    }
    catch ( const std::exception& e ) {
        std::cerr << "smfconv: Unexpected exception" << std::endl
             << " message:" << e.what() << std::endl;
        retCode = 1;
    }
    catch(...) {
        std::cerr << "smfconv: Unexpected exception" << std::endl;
        retCode = 1;
    }

    return retCode;
}

