﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sndlib/path.h"
#include "sndlib/utillity.h"


namespace sndlib
{


using namespace std;

namespace // anonymous
{

typedef list< string > DirList;

struct PathInfo
{
    bool absolute;
    bool unc;
    string drive;
    string uncHost;
    string fileBase;
    string fileExt;
    DirList dirList;

    const string GetFilePath() const;
    const string GetDirPath() const;
    const string GetFileName() const;
    const string GetBasePath() const;
    const string GetBaseName() const;

private:
    string ConstructDirPath() const;
};

const string PathInfo::GetFilePath() const
{
    string path = ConstructDirPath();

    path += fileBase;
    path += fileExt;

    return path;
}

const string PathInfo::GetDirPath() const
{
    string path = ConstructDirPath();

    if ( path.empty() ) return "./";

    return path;
}

const string PathInfo::GetFileName() const
{
    string path = fileBase;
    path += fileExt;

    return path;
}

const string PathInfo::GetBasePath() const
{
    string path = ConstructDirPath();

    return path + fileBase;
}

const string PathInfo::GetBaseName() const
{
    return fileBase;
}

string PathInfo::ConstructDirPath() const
{
    string path;

    if ( unc ) {
        path += "//";
        path += uncHost;
        path += "/";
    }
    else if ( absolute ) {
        path += drive;
        path += "/";
    }

    for( DirList::const_iterator p = dirList.begin() ;
         p != dirList.end() ; ++p )
    {
        path += *p;
        path += '/';
    }

    return path;
}

// 最後に'/'を付加する
string FixDirectoryName( const string& dirPath )
{
    if ( dirPath.empty() ) return "./";

    if ( dirPath.find_last_of( "\\/" ) != dirPath.length()-1 ) {
        return dirPath + '/';
    }

    return dirPath;
}

void ParseFilePath( const string& filePath, PathInfo& pathInfo )
{
    char drive[_MAX_DRIVE];
    char dir[_MAX_DIR];
    char fname[_MAX_FNAME];
    char ext[_MAX_EXT];

    _splitpath( filePath.c_str(), drive, dir, fname, ext );

    pathInfo.drive = drive;

    pathInfo.absolute = pathInfo.drive.empty() ? false : true ;
    pathInfo.unc = false;
    pathInfo.uncHost.clear();
    pathInfo.dirList.clear();

    const string dirPath( dir );

    string::size_type prev = 0;
    string::size_type next = 0;

    // 絶対パスの解析
    while( next < dirPath.length() )
    {
        if ( dirPath[ next ] != '\\' && dirPath[ next ] != '/' ) break;

        if ( next == 0 ) pathInfo.absolute = true;
        else if ( next == 1 ) pathInfo.unc = true;

        next++;
    }
    prev = next;

    // ディレクトリの解析
    bool bFindUncHost = false;
    while( ( next = dirPath.find_first_of( "\\/", prev ) ) != string::npos )
    {
        const string& dirName = dirPath.substr( prev, next-prev );

        if ( ! dirName.empty() && dirName != "." )
        {
            if ( dirName == ".." && ! pathInfo.dirList.empty() && pathInfo.dirList.back() != ".." ) {
                pathInfo.dirList.pop_back();
            }
            else {
                if ( pathInfo.unc && ! bFindUncHost ) {
                    pathInfo.uncHost = dirName;
                    bFindUncHost = true;
                }
                else {
                    pathInfo.dirList.push_back( dirName );
                }
            }
        }

        prev = next+1;
    }

    // ファイル名
    pathInfo.fileBase = fname;
    pathInfo.fileExt = ext;
}

}

namespace path
{

string getExtName( const string& path )
{
    string::size_type p = path.find_last_of('.');
    if ( p == string::npos ) return string();
    return path.substr( p );
}

string getBaseName( const string& path )
{
    string baseName = path;
    string::size_type p = baseName.find_last_of("\\/:");
    if ( p != string::npos ) baseName = baseName.substr( p+1 );
    p = baseName.find_last_of('.');
    if ( p == string::npos ) return baseName;
    return baseName.substr( 0, p );
}

string getBasePath( const string& path )
{
    string::size_type p = path.find_last_of('.');
    if ( p == string::npos ) return path;
    return path.substr( 0, p );
}

string getModuleDir()
{
    TCHAR moduleFileName[ _MAX_PATH ];
    GetModuleFileName( NULL, moduleFileName, _MAX_PATH );
    const std::string modname = moduleFileName;
    const std::string moddir = modname.substr(0, modname.rfind('\\'));
    return moddir;
}

bool isDirectory( const string& path )
{
    DWORD dwAttr = GetFileAttributes( path.c_str() );
    if ( dwAttr == -1 ) return false;
    if ( dwAttr & FILE_ATTRIBUTE_DIRECTORY ) return true;
    return false;
}

bool MakeDirectory( const string& dirPath )
{
    PathInfo pathInfo;

    ParseFilePath( FixDirectoryName( dirPath ), pathInfo );

    string path;

    if ( pathInfo.unc ) {
        path += "//";
        path += pathInfo.uncHost;
        path += "/";
    }
    else if ( pathInfo.absolute ) {
        path += pathInfo.drive;
        path += "/";
    }

    DirList::const_iterator p;
    for ( p = pathInfo.dirList.begin(); p != pathInfo.dirList.end(); ++p )
    {
        path += *p;

        if ( ! isDirectory( path ) ) {
            if ( _mkdir( path.c_str() ) != 0 ) {
                return false;
            }
        }

        path += '/';
    }

    return true;
}

unsigned long getFileSize( const string& s )
{
    HANDLE h = CreateFile(
        s.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return 0;

    DWORD fileSize = GetFileSize( h, NULL );

    CloseHandle( h );

    return fileSize;
}

int CompareFileTime( const string& s1, const string& s2 )
{
    FILETIME t1;
    FILETIME t2;

    HANDLE h = CreateFile(
        s1.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return -1;

    if ( ! GetFileTime( h, NULL, NULL, &t1 ) ) {
        CloseHandle( h );
        return -1;
    }
    CloseHandle( h );

    h = CreateFile(
        s2.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return -1;

    if ( ! GetFileTime( h, NULL, NULL, &t2 ) ) {
        CloseHandle( h );
        return -1;
    }

    CloseHandle( h );

    return CompareFileTime( &t2, &t1 );
}

const string NormalizeFilePath( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return pathInfo.GetFilePath();
}


const string GetDirPath( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return pathInfo.GetDirPath();
}

const string GetFileName( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return pathInfo.GetFileName();
}

const string GetBasePath( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return pathInfo.GetBasePath();
}

const string GetBaseName( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return pathInfo.GetBaseName();
}

const string GetExtName( const string& filePath )
{
    char ext[_MAX_EXT];

    _splitpath( filePath.c_str(), NULL, NULL, NULL, ext );

    return ext;
}

bool IsRelative( const string& filePath )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    return ! pathInfo.absolute;
}

const string AppendBaseDir( const string& filePath, const string& baseDir )
{
    PathInfo pathInfo;

    ParseFilePath( filePath, pathInfo );

    if ( pathInfo.absolute ) {
        return pathInfo.GetFilePath();
    }

    const string absFilePath = baseDir + "/" + filePath;
    ParseFilePath( absFilePath, pathInfo );

    return pathInfo.GetFilePath();
}

const string GetRelativeFilePath( const string& filePath, const string& baseDirPath )
{
    char curDir[ _MAX_PATH ];
    GetCurrentDirectory( sizeof( curDir ), curDir );

    PathInfo pathInfo;
    PathInfo baseDirPathInfo;

    ParseFilePath( filePath, pathInfo );
    ParseFilePath( FixDirectoryName( baseDirPath ), baseDirPathInfo );

    if ( ! pathInfo.absolute ) {
        ParseFilePath( string( curDir ) + "/" + filePath, pathInfo );
    }
    if ( ! baseDirPathInfo.absolute ) {
        ParseFilePath( string( curDir ) + "/" + FixDirectoryName( baseDirPath ), baseDirPathInfo );
    }

    if ( pathInfo.unc ) {
        if ( ! baseDirPathInfo.unc ) return pathInfo.GetFilePath();
        if ( pathInfo.uncHost != baseDirPathInfo.uncHost ) return pathInfo.GetFilePath();
    }
    else {
        if ( baseDirPathInfo.unc ) return pathInfo.GetFilePath();
        if ( util::compareNoCase(pathInfo.drive,baseDirPathInfo.drive) != 0 ) return pathInfo.GetFilePath();
    }

    DirList::iterator p = pathInfo.dirList.begin();
    DirList::iterator bp = baseDirPathInfo.dirList.begin();

    while( p != pathInfo.dirList.end() && bp != baseDirPathInfo.dirList.end() ) {
        if ( util::compareNoCase(*p,*bp) != 0 ) break;
        p++;
        bp++;
    }

    pathInfo.dirList.erase( pathInfo.dirList.begin(), p );
    baseDirPathInfo.dirList.erase( baseDirPathInfo.dirList.begin(), bp );

    for( DirList::reverse_iterator bp = baseDirPathInfo.dirList.rbegin();
         bp != baseDirPathInfo.dirList.rend(); bp++ )
    {
        pathInfo.dirList.push_front("..");
    }

    pathInfo.absolute = false;
    pathInfo.unc = false;
    return pathInfo.GetFilePath();
}

} // end of namespace path

} // namespace sndlib

