﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sndlib/lexer.h"
#include "sndlib/path.h"
#include "sndlib/ExprParser.h"

#include <windows.h>

namespace sndlib
{

using namespace std;

#define INCLUDE_DEPTH_MAX 256

namespace
{
int todigit( char c )
{
    assert( isxdigit( c ) ) ;

    if ( isdigit( c ) ) return c - '0';
    if ( 'a' <= c && c <= 'f' ) return c - 'a' + 10;
    if ( 'A' <= c && c <= 'F' ) return c - 'A' + 10;

    return 0;
}

char escape( char c )
{
    switch( c ) {
    case '0': return '\0';
    case 'a': return '\a';
    case 'b': return '\b';
    case 't': return '\t';
    case 'n': return '\n';
    case 'v': return '\v';
    case 'f': return '\f';
    case 'r': return '\r';
    }

    return c;
}
}

Lexer::Lexer( std::istream& in, const string& fileName )
: mLastToken( RETURN ),
  mLastTokenString(),
  mNumberValue(0),
  mStringValue(),
  mLastTokenAvailable(false),
  mExceptionFlag(false),
  mIgnoreIfState(false)
{
    SmartPtr<File> file( new File( in, path::NormalizeFilePath( fileName ) ) );

    mFileStack.push_back( file );
    mDependencyFileList.insert( path::NormalizeFilePath( fileName ));
}



bool Lexer::File::GetChar( char& ch )
{
    istream& in = mInput;

    if ( mLinePos >= mLineBuffer.length() )
    {
        if ( in.eof() ) return false;

        getline( in, mLineBuffer );

        mLinePos = 0;
        mLineNumber++;

        mLineBuffer += '\n';
    }

    ch = mLineBuffer[ mLinePos ];
    mLinePos++;

    return true;
}

void Lexer::File::UnGet()
{
    assert( mLinePos > 0 );
    mLinePos--;
}

bool Lexer::GetChar( char& ch )
{
#if 1
    if ( mFileStack.empty() ) {
        if ( mExceptionFlag ) throw Failure();
        return false;
    }

    if ( ! mFileStack.back()->GetChar( ch ) )
    {
        if ( mFileStack.size() == 1 && ! mIfStateStack.empty() )
        {
            stringstream s;
            s << "Mismatched #endif";
            throw Exception( s.str(), *this);
        }
        mFileStack.pop_back();

        ch = '\n';
    }

    return true;
#else
    while( ! mFileStack.empty() && ! mFileStack.back()->GetChar( ch ) )
    {
        if ( mFileStack.size() == 1 && ! mIfStateStack.empty() )
        {
            stringstream s;
            s << "Mismatched #endif";
            throw Exception( s.str(), *this);
        }
        mFileStack.pop_back();
    }

    if ( mFileStack.empty() ) {
        if ( mExceptionFlag ) throw Failure();
        return false;
    }

    return true;
#endif
}

bool Lexer::GetChar_SkipSpace( char& ch )
{
    do {
        if ( ! GetChar( ch ) ) return false;

        // コメント
        if ( ch == ';' ) {
            do {
                if ( ! GetChar( ch ) ) return false;
            } while ( ch != '\n' );
        }
    } while ( ch != '\n' && isspace(ch) );

    return true;
}

void Lexer::UnGet( )
{
    assert( ! mFileStack.empty() );
    SmartPtr<File>& file = mFileStack.back();
    file->UnGet();
}

bool Lexer::CheckIfState() const
{
    if ( mIgnoreIfState ) return true;

    IfStateStack::const_iterator p;
    for ( p = mIfStateStack.begin() ; p != mIfStateStack.end() ; p++ )
    {
        if ( p->flag && p->elseFlag ) return false;
        if ( ! p->flag && ! p->elseFlag ) return false;
    }
    return true;
}

Lexer::Token Lexer::ParseDirective()
{
    char ch;

    if ( ! GetChar_SkipSpace( ch ) ) {
        return mLastToken = END;
    }

    if ( ch == '\n' ) {
        // # のみ
        return mLastToken = RETURN;
    }

    // ディレクティブ名取得
    if ( ! isalpha( ch ) && ch != '_' ) {
        stringstream s;
        s << "Expected preprocessor directive, found \'" << ch << "\'";
        throw Exception( s.str(), *this);
    }
    string directive;
    directive = ch;
    while( GetChar( ch ) ) {
        if ( ! isalnum( ch ) && ch != '_' ) {
            UnGet();
            break;
        }
        directive += ch;
    }

    // ディレクティブ毎の処理
    if ( directive == "if" ) {
        if ( ! GetChar( ch ) || ! isspace(ch) || ch == '\n' ) {
            stringstream s;
            s << "Expected space after a #if";
            throw Exception( s.str(), *this);
        }
        ExprParser exprParser;
        mIgnoreIfState = true;
        bool flag = exprParser.Parse( *this ) ? true : false;
        mIgnoreIfState = false;
        mIfStateStack.push_back( IfState( flag ) );
    }
    else if ( directive == "ifdef" || directive == "ifndef" ) {
        if ( ! GetChar( ch ) || ! isspace(ch) || ch == '\n' ) {
            stringstream s;
            s << "Expected space after a #" << directive;
            throw Exception( s.str(), *this);
        }
        if ( ! GetChar_SkipSpace( ch ) || ch == '\n' ) {
            stringstream s;
            s << "No identifier appears after a #" << directive;
            throw Exception( s.str(), *this);
        }
        if ( ! isalpha( ch ) && ch != '_') {
            stringstream s;
            s << "Expected identifier, found \'" << ch << "\'";
            throw Exception( s.str(), *this);
        }

        string macroName;
        macroName = ch;
        while( GetChar( ch ) ) {
            if ( ! isalnum( ch ) && ch != '_' ) {
                UnGet();
                break;
            }
            macroName += ch;
        }
        bool flag = ( mMacroTable.find( macroName ) != mMacroTable.end() );
        if ( directive == "ifndef" ) flag = ! flag;
        mIfStateStack.push_back( IfState( flag ) );

        ReadToken();
    }
    else if ( directive == "elif" ) {
        if ( mIfStateStack.empty() || mIfStateStack.back().elseFlag ) {
            throw Lexer::Exception( "Unexpected #elif", *this );
        }

        if ( ! GetChar( ch ) || ! isspace(ch) || ch == '\n' ) {
            stringstream s;
            s << "Expected space after a #elif";
            throw Exception( s.str(), *this);
        }

        mIfStateStack.back().elseFlag = true;

        ExprParser exprParser;
        mIgnoreIfState = true;
        bool flag = exprParser.Parse( *this ) ? true : false;
        mIgnoreIfState = false;
        mIfStateStack.push_back( IfState( flag, true ) );
    }
    else if ( directive == "else" ) {
        if ( mIfStateStack.empty() || mIfStateStack.back().elseFlag ) {
            throw Lexer::Exception( "Unexpected #else", *this );
        }
        mIfStateStack.back().elseFlag = true;

        ReadToken();
    }
    else if ( directive == "endif" ) {
        if ( mIfStateStack.empty() ) {
            throw Lexer::Exception( "Unexpected #endif", *this );
        }
        while( ! mIfStateStack.empty() ) {
            if ( ! mIfStateStack.back().elifFlag ) {
                mIfStateStack.pop_back();
                break;
            }
            mIfStateStack.pop_back();
        }

        ReadToken();
    }
    else {
        if ( ! CheckIfState() ) {
            // skip line
            char ch;
            while ( GetChar_SkipSpace( ch ) && ch != '\n' ) {
            }
            if ( ch == '\n' ) mLastToken = RETURN;
            else mLastToken = END;
        }
        else if ( directive == "define" ) {
            // #define
            if ( ! GetChar( ch ) || ! isspace(ch) || ch == '\n' ) {
                stringstream s;
                s << "Expected space after a #define";
                throw Exception( s.str(), *this);
            }

            if ( ! GetChar_SkipSpace( ch ) || ch == '\n' ) {
                stringstream s;
                s << "No identifier appears after a #define";
                throw Exception( s.str(), *this);
            }
            if ( ! isalpha( ch ) && ch != '_') {
                stringstream s;
                s << "Expected identifier, found \'" << ch << "\'";
                throw Exception( s.str(), *this);
            }

            string macroName;
            macroName = ch;
            while( GetChar( ch ) ) {
                if ( ! isalnum( ch ) && ch != '_' ) {
                    UnGet();
                    break;
                }
                macroName += ch;
            }

            SmartPtr< Macro > macro( new Macro );

            Token token;
            while ( ( token = ReadToken() ) != END ) {
                if ( token == RETURN ) break;

                if ( token == SYMBOL ) {
                    // 再帰置換チェック
                    MacroTable::const_iterator p = mMacroTable.find( mStringValue );
                    if ( p != mMacroTable.end() ) {
                        const Macro& m = * (p->second);
                        for( Macro::const_iterator p = m.begin();
                             p != m.end() ; ++p )
                        {
                            macro->push_back( *p );
                        }
                    }
                }

                macro->push_back( MakeCurrentTokenInfo() );
            }

            MacroTable::const_iterator p = mMacroTable.find( macroName );
            if ( p != mMacroTable.end() )
            {
                // ２重定義

                SmartPtr<Macro> m = p->second;
                if ( m->size() != macro->size() ||
                     ! std::equal( m->begin(), m->end(), macro->begin(), IsEqual ) )
                {
                    stringstream s;
                    s << "Multiple defined macro \"" << macroName << "\"";
                    throw Exception( s.str(), *this);
                }
            }

            mMacroTable[ macroName ] = macro;
        }
        else if ( directive == "undef" ) {
            // #undef
            if ( ! GetChar( ch ) || ! isspace(ch) || ch == 'ch' ) {
                stringstream s;
                s << "Expected space after a #undef";
                throw Exception( s.str(), *this);
            }
            if ( ! GetChar_SkipSpace( ch ) || ch == '\n' ) {
                stringstream s;
                s << "No identifier appears after a #undef";
                throw Exception( s.str(), *this);
            }
            if ( ! isalpha( ch ) && ch != '_') {
                stringstream s;
                s << "Expected identifier, found \'" << ch << "\'";
                throw Exception( s.str(), *this);
            }
            string macroName;
            macroName = ch;
            while( GetChar( ch ) ) {
                if ( ! isalnum( ch ) && ch != '_' ) {
                    UnGet();
                    break;
                }
                macroName += ch;
            }
            if ( mMacroTable.find( macroName ) != mMacroTable.end() ) {
                mMacroTable.erase( macroName );
            }

            ReadToken();
        }
        else if ( directive == "include" ) {
            // #include
            Token t = ReadToken();

            if ( t != STRING && t != LT ) {
                stringstream s;
                s << "Expected filename, found \'" << ch << "\'";
                throw Exception( s.str(), *this);
            }
            if ( mFileStack.size() > INCLUDE_DEPTH_MAX ) {
                stringstream s;
                s << "#include nesting overflow";
                throw Exception( s.str(), *this);
            }

            bool bSystemPath = ( t != STRING ) ? true : false;

            // ファイル名の取得
            string fileName;
            if ( ! bSystemPath ) {
                fileName = GetStringValue();
            }
            else {
                while ( GetChar( ch ) )
                {
                    if ( ch == '\\' ) { // エスケープ
                        if ( ! GetChar( ch ) ) {
                            throw Exception("Syntax error '\"'", *this);
                        }
                        ch = escape( ch );
                    }
                    else {
                        if ( ch == '>' ) {
                            break;
                        }
                        if ( ch == '\n' ) {
                            throw Exception("Newline in constant", *this);
                        }
                    }
                    fileName += ch;
                }
            }

            // パスの変換
            IncludeFile* pFile = NULL;
            string filePath;
            bool bSuccess = false;
            int retry = 1;
            if ( bSystemPath && path::IsRelative( fileName ) ) {
                retry = static_cast<int>( mSearchPathList.size() );
            }
            while( ! bSuccess )
            {
                if ( retry <= 0 )
                {
                    stringstream s;
                    s << "Cannot open file \"" << fileName << "\"";
                    throw Exception( s.str(), *this );
                }

                filePath = fileName;
                if ( path::IsRelative( fileName ) )
                {
                    assert( ! mFileStack.empty() );
                    string dirPath;
                    if ( t == STRING ) {
                        dirPath = path::GetDirPath( mFileStack.back()->GetFileName() );
                    }
                    else {
                        dirPath = mSearchPathList[ mSearchPathList.size() - retry ];
                        dirPath += '\\';
                    }
                    filePath = dirPath + fileName;
                }
                filePath = path::NormalizeFilePath( filePath );

                //
                pFile = new IncludeFile( filePath );
                if ( pFile->IsOpen() ) {
                    bSuccess = true;
                }
                else {
                    delete pFile;
                    pFile = NULL;
                    retry--;
                }
            }

            try {
                ReadToken(); // read return
            }
            catch( ... ) {
                delete pFile;
                throw;
            }

            mFileStack.push_back( SmartPtr<IncludeFile>( pFile ) );
            mDependencyFileList.insert( path::NormalizeFilePath( filePath ));
        }
        else {
            stringstream s;
            s << "Invalid preprocessor command string \'" << directive << "\'";
            throw Exception( s.str(), *this);
        }
    }

    if ( mLastToken != RETURN ) {
        stringstream s;
        s << "Unexpected tokens following preprocessor directive - expected a newline";
        throw Exception( s.str(), *this);
    }

    return mLastToken;
}


Lexer::Token Lexer::ReadToken()
{
    char ch;

    while( 1 )
    {
        if ( mLastTokenAvailable ) {
            mLastTokenAvailable = false;
            return mLastToken;
        }

        if ( ! mTokenStack.empty() ) {
            TokenInfo info = mTokenStack.top();

            mLastToken = info.mToken;
            mLastTokenString = info.mTokenString;
            mStringValue = info.mStringValue;
            mNumberValue = info.mNumberValue;
            mTokenStack.pop();
            return mLastToken;
        }

        if ( ! GetChar_SkipSpace( ch ) ) {
            return mLastToken = END;
        }

        mLastTokenString = ch;
        mNumberValue = 0;
        mStringValue.clear();

        if ( ch == '#' ) {
            if ( mLastToken != RETURN ) {
                stringstream s;
                s << "The # sign of a preprocessor directive must be the first charactor on a line";
                throw Exception( s.str() , *this);
            }
            return mLastToken = ParseDirective();
        }

        if ( ! CheckIfState() ) continue;

        switch( ch ) {

        case '\\':
            throw Exception("Illegal escape sequence", *this);

        case '\n':
        case '-':
        case '+':
        case '*':
        case '/':
        case ':':
        case ',':
        case '.':
        case '(':
        case ')':
        case '{':
        case '}':
        case '[':
        case ']':
        case '|':
        case '&':
            return mLastToken = Token( ch );

        case '=':
            if ( ! GetChar( ch ) ) {
                return mLastToken = EQUAL;
            }
            switch( ch ) {
            case '=':
                return mLastToken = EQ;

            default:
                UnGet();
                return mLastToken = EQUAL;
            }

        case '<':
            if ( ! GetChar( ch ) ) {
                return mLastToken = LT;
            }
            switch( ch ) {
            case '=':
                return mLastToken = LE;
            case '<':
                return mLastToken = LSHIFT;

            default:
                UnGet();
                return mLastToken = LT;
            }

        case '>':
            if ( ! GetChar( ch ) ) {
                return mLastToken = GT;
            }
            switch( ch ) {
            case '=':
                return mLastToken = GE;
            case '>':
                return mLastToken = RSHIFT;

            default:
                UnGet();
                return mLastToken = GT;
            }

        case '"': // 文字列
            mStringValue.clear();
            while ( GetChar( ch ) ) {
                if ( ch == '\\' ) { // エスケープ
                    if ( ! GetChar( ch ) ) {
                        throw Exception("Syntax error '\"'", *this);
                    }
                    ch = escape( ch );
                }
                else {
                    if ( ch == '"' ) {
                        return mLastToken = STRING;
                    }
                    if ( ch == '\n' ) {
                        throw Exception("Newline in constant", *this);
                    }
                }
                mStringValue += ch;
            }
            throw Exception("Syntax error '\"'", *this);

        case '\'': // 文字定数
            if ( ! GetChar( ch ) ) {
                throw Exception("Syntax error '\''", *this);
            }
            if ( ch == '\'' ) {
                throw Exception("Empty character constant", *this);
            }
            if ( ch == '\\' ) { // エスケープ
                if ( ! GetChar( ch ) ) {
                    throw Exception("Syntax error '\''", *this);
                }
                ch = escape( ch );
            }
            mNumberValue = ch;
            if ( ! GetChar( ch ) ) {
                throw Exception("Syntax error '\''", *this);
            }
            if ( ch != '\'' ) {
                throw Exception("Too many characters in constant", *this);
            }
            return mLastToken = NUMBER;

        case '0': // 数字
            if ( ! GetChar( ch ) ) {
                mNumberValue = 0;
                return mLastToken = NUMBER;
            }
            if ( ch == 'x' || ch == 'X' ) {
                // 0x 16進数モード
                if ( ! GetChar( ch ) ) {
                    throw Exception("Hexadecimal constants are illegal.", *this);
                }
                if ( ! isxdigit( ch ) ) {
                    throw Exception("Hexadecimal constants are illegal.", *this);
                }
                mNumberValue = todigit( ch );
                while ( GetChar( ch ) ) {
                    if ( ! isxdigit( ch ) ) {
                        UnGet();
                        break;
                    }
                    if ( mNumberValue >> 60 ) {
                        throw Exception("Constant too big", *this);
                    }
                    mNumberValue <<= 4;
                    mNumberValue |= todigit( ch );
                }
                return mLastToken = NUMBER;
            }
            if ( ch == 'b' || ch == 'B' ) {
                // 0b 2進数モード
                if ( ! GetChar( ch ) ) {
                    throw Exception("Binary number constants are illegal.", *this);
                }
                if ( ch != '0' && ch != '1' ) {
                    throw Exception("Binary number constants are illegal.", *this);
                }
                mNumberValue = todigit( ch );
                while ( GetChar( ch ) ) {
                    if ( ch != '0' && ch != '1' ) {
                        UnGet();
                        break;
                    }
                    if ( mNumberValue >> 60 ) {
                        throw Exception("Constant too big", *this);
                    }
                    mNumberValue <<= 1;
                    mNumberValue |= todigit( ch );
                }
                return mLastToken = NUMBER;
            }
            UnGet();
            ch = '0';
            // continue;

        case '1': case '2': case '3': case '4':
        case '5':  case '6': case '7': case '8': case '9':
            mNumberValue = ch - '0';
            while ( GetChar( ch ) ) {
                if ( ! isdigit( ch ) ) {
                    UnGet();
                    break;
                }
                if ( mNumberValue >> 60 ) {
                    throw Exception("Constant too big", *this);
                }
                mNumberValue *= 10;
                mNumberValue += ch - '0';
            }

            return mLastToken = NUMBER;


        case '@': // @ディレクティブ
            mStringValue.clear();
            if ( ! GetChar( ch ) ||
                 ! isalpha( ch ) && ch != '_' )
            {
                throw Exception("Empty directive name", *this);
            }
            mStringValue = ch;
            while ( GetChar( ch ) ) {
                if ( ! isalpha(ch) && ! isdigit( ch ) && ch != '_' ) {
                    UnGet();
                    break;
                }
                mStringValue += ch;
            }
            return mLastToken = DIRECTIVE;

        default:
            if ( isalpha( ch ) || ch == '_' ) { // シンボル
                mStringValue = ch;
                while ( GetChar( ch ) ) {
                    if ( ! isalpha(ch) && ! isdigit( ch ) && ch != '_' ) {
                        UnGet();
                        break;
                    }
                    mStringValue += ch;
                }

                MacroTable::const_iterator p = mMacroTable.find( mStringValue );
                if ( p == mMacroTable.end() ) {
                    return mLastToken = SYMBOL;
                }

                // マクロ置換
                const Macro& macro = * (p->second);
                for( Macro::const_reverse_iterator p = macro.rbegin();
                     p != macro.rend() ; ++p )
                {
                    mTokenStack.push( *p );
                }

                // one more time
                break;
            }

            // エラー
            stringstream s;
            s << "Unknown charactor \'" << ch << "\'";
            throw Exception( s.str() , *this);
        }
    }
}

Lexer::Token Lexer::PeekToken( )
{
    Token token = ReadToken();

#if 1
    mLastTokenAvailable = true;
#else
    mTokenStack.push( MakeCurrentTokenInfo() );
#endif

    return token;
}

bool Lexer::IsNumberValueAvailable() const
{
    if ( mLastToken == NUMBER ) return true;
    return false;
}
bool Lexer::IsStringValueAvailable() const
{
    if ( mLastToken == SYMBOL ) return true;
    if ( mLastToken == STRING ) return true;
    if ( mLastToken == DIRECTIVE ) return true;

    return false;
}

__int64 Lexer::GetNumberValue() const
{
    assert( IsNumberValueAvailable() );
    return mNumberValue;
}

std::string Lexer::GetStringValue() const
{
    assert( IsStringValueAvailable() );
    return mStringValue;
}

int Lexer::GetLineNumber() const
{
    if ( mFileStack.empty() ) return 0;
    const SmartPtr<File>& file = mFileStack.back();
    return file->GetLineNumber();
}

std::string Lexer::GetFileName() const
{
    if ( mFileStack.empty() ) return string();
    const SmartPtr<File>& file = mFileStack.back();
    return file->GetFileName();
}

std::string Lexer::GetLastLine() const
{
    if ( mFileStack.empty() ) return string();
    const SmartPtr<File>& file = mFileStack.back();
    return file->GetLastLine();
}

LexerState Lexer::GetState() const
{
    LexerState state;
    state.fileName = GetFileName();
    state.lineNumber = GetLineNumber();
    state.lastLine = GetLastLine();

    return state;
}

Lexer::TokenInfo Lexer::MakeCurrentTokenInfo() const
{
    TokenInfo info;
    info.mToken = mLastToken;
    info.mTokenString = mLastTokenString;
    info.mNumberValue = 0; // 未初期化を避けます
    if ( IsStringValueAvailable() ) {
        info.mStringValue = mStringValue;
    }
    if ( IsNumberValueAvailable() ) {
        info.mNumberValue = mNumberValue;
    }
    return info;
}

void Lexer::PushSearchPath( const std::string& path )
{
    char curDir[ _MAX_PATH ];
    ::GetCurrentDirectory( sizeof( curDir ), curDir );

    const std::string searchPath = path::AppendBaseDir( path, curDir );

    mSearchPathList.push_back( searchPath);
}


bool Lexer::IsEqual( const TokenInfo& lhs, const TokenInfo& rhs )
{
    if ( lhs.mToken != rhs.mToken ) return false;

    switch( lhs.mToken ) {
    case SYMBOL:
    case STRING:
    case DIRECTIVE:
        if ( lhs.mStringValue != rhs.mStringValue ) return false;
        break;
    case NUMBER:
        if ( lhs.mNumberValue != rhs.mNumberValue ) return false;
        break;
    default:
        break;
    }

    return true;
}

void Lexer::VerifyToken( Token token, const string& expectedToken ) const
{
    if ( mLastToken != token ) {
        stringstream s;
        s << "Expected " << expectedToken << ", but found \"" << mLastTokenString << "\"";
        throw Lexer::Exception( s.str(), *this );
    }
}

void Lexer::VerifyMinMaxValue( const std::string& name, __int64 value, __int64 min, __int64 max ) const
{
    if ( min <= value && value <= max ) return;

    stringstream s;
    s << name << " is out of range, " << min << " <= " << value << " <= " << max << " not satisfied.";
    throw Exception( s.str(), *this );
}

Lexer::Exception::Exception( const string& message, const Lexer& lexer )
: std::exception("")
{
    stringstream s;
    s << lexer.GetFileName() << ":" << lexer.GetLineNumber() << ": "
      << message << endl
      << lexer.GetLastLine();
    ;
    mStr = s.str();
}
Lexer::Exception::Exception( const string& message, const LexerState& state )
: std::exception("")
{
    stringstream s;
    s << state.fileName << ":" << state.lineNumber << ": "
      << message << endl
      << state.lastLine;
    ;
    mStr = s.str();
}
const char* Lexer::Exception::what() const
{
    return mStr.c_str();
}

namespace lexer
{

bool ParseKeyString( const string& str, int& key )
{
    static const int key_offset[] = {
        9, 11, 0, 2, 4, 5, 7
    };
    string::const_iterator p = str.begin();
    char c = *p++;
    if ( 'a' <= c && c <= 'g' ) {
        key = key_offset[ c - 'a' ];
    }
    else if ( 'A' <= c && c <= 'G' ) {
        key = key_offset[ c - 'A' ];
    }
    else {
        return false;
    }

    if ( p == str.end() ) return false;
    c = *p;
    switch( c ) {
    case 'n': case 'N':
        p++;
        break;
    case 's': case 'S':
        key++;
        p++;
        break;
    case 'f': case 'b': case 'F': case 'B':
        key--;
        p++;
        break;
    }

    if ( p == str.end() ) return false;
    c = *p;

    bool minus_flag = false;
    if ( c == 'm' || c == 'M' ) {
        minus_flag = true;
        p++;
    }

    if ( p == str.end() ) return false;
    c = *p++;
    if (! isdigit( c ) ) return false;

    int octave = c - '0';
    while( p != str.end() ) {
        c = *p++;
        if (! isdigit( c ) ) return false;
        octave *= 10;
        octave += c - '0';
    }

    if ( minus_flag ) octave = - octave;

    key += ( octave + 1 ) * 12;
    return true;
}
}

} // namespace sndlib

