﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(AFX_UTILLITY_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_)
#define AFX_UTILLITY_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <ctime>
#include <windows.h>
#include <direct.h>
#include <vector>
#include <set>
#include <iostream>
#include <string>
#include "stream.h"

namespace sndlib {
namespace util {

using namespace std;


struct DeleteObject
{
    template<typename T>
    void operator()(T* ptr) const { delete ptr; }
};

inline bool ciCharLessNoCase( char c1, char c2 ) {
    return
        tolower( static_cast<unsigned char>( c1 ) ) <
        tolower( static_cast<unsigned char>( c2 ) ) ;
}

struct StringLessNoCase
    : public binary_function< string, string, bool >
{
    bool operator()( const string& s1, const string& s2 ) const {
        return lexicographical_compare(
            s1.begin(), s1.end(),
            s2.begin(), s2.end(),
            ciCharLessNoCase
        );
    }
};

inline int compareNoCase( const string& s1, const string& s2 )
{
    string::const_iterator p1;
    string::const_iterator p2;
    for( p1 = s1.begin() , p2 = s2.begin() ; p1 != s1.end(); p1++, p2++ )
    {
        if ( p2 == s2.end() ) return 1;

        char c1 = tolower(*p1);
        char c2 = tolower(*p2);
        if ( c1 == c2 ) continue;
        return ( c1 < c2 ) ? -1 : 1;
    }

    if ( p2 != s2.end() ) return -1;

    return 0;
}

inline string getQuotedFilePath( const string& filePath )
{
    if ( filePath.find(' ') != std::string::npos ) {
        return "\"" + filePath + "\"";
    }
    return filePath;
}

inline string getBaseName( const string& path )
{
    string baseName = path;
    string::size_type p = baseName.find_last_of("\\/:");
    if ( p != string::npos ) baseName = baseName.substr( p+1 );
    p = baseName.find_last_of('.');
    if ( p == string::npos ) return baseName;
    return baseName.substr( 0, p );
}

inline string getBasePath( const string& path )
{
    string::size_type p = path.find_last_of('.');
    if ( p == string::npos ) return path;
    return path.substr( 0, p );
}

inline std::basic_string<TCHAR> getModuleDir()
{
    TCHAR moduleFileName[ _MAX_PATH ];
    GetModuleFileName( NULL, moduleFileName, _MAX_PATH );
    const std::basic_string<TCHAR> modname = moduleFileName;
    const std::basic_string<TCHAR> moddir = modname.substr(0, modname.rfind('\\'));
    return moddir;
}

inline bool isDirectory( const std::basic_string<TCHAR>& path )
{
    DWORD dwAttr = GetFileAttributes( path.c_str() );
    if ( dwAttr == -1 ) return false;
    if ( dwAttr & FILE_ATTRIBUTE_DIRECTORY ) return true;
    return false;
}

inline DWORD getFileSize( const std::basic_string<TCHAR>& s )
{
    HANDLE h = CreateFile(
        s.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return 0;

    DWORD fileSize = GetFileSize( h, NULL );

    CloseHandle( h );

    return fileSize;
}

inline int CompareFileTime(
    const std::basic_string<TCHAR>& s1,
    const std::basic_string<TCHAR>& s2
)
{
    FILETIME t1;
    FILETIME t2;

    HANDLE h = CreateFile(
        s1.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return -1;

    if ( ! GetFileTime( h, NULL, NULL, &t1 ) ) {
        CloseHandle( h );
        return -1;
    }
    CloseHandle( h );

    h = CreateFile(
        s2.c_str(),
        GENERIC_READ, FILE_SHARE_READ, NULL,
        OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if ( h == INVALID_HANDLE_VALUE ) return -1;

    if ( ! GetFileTime( h, NULL, NULL, &t2 ) ) {
        CloseHandle( h );
        return -1;
    }

    CloseHandle( h );

    return CompareFileTime( &t2, &t1 );
}

inline int CompareFileTime(
    const set<std::basic_string<TCHAR> >& list,
    const std::basic_string<TCHAR>& s2
)
{
    for( set<std::basic_string<TCHAR> >::const_iterator p = list.begin() ;
         p != list.end() ; ++ p )
    {
        if ( CompareFileTime( *p, s2 ) < 0 ) return -1;
    }

    return 1;
}


inline bool outputFile( const string& s, strm::fstream& out )
{
    FILE* fp = fopen( s.c_str(), "rb" );
    if ( fp == NULL ) return false;

    vector<char> buffer( 0x10000 );
    while( size_t read_size = fread( &buffer[0], 1, buffer.size(), fp ) )
    {
        out.write( &buffer[0], read_size );
    }

    fclose( fp );
    return true;
}

class profile
{
  public:
    profile( const string& s ) : str(s) {
        s_time = clock();
    }

    ~profile() {
        clock_t e_time = clock();
        clock_t diff = e_time - s_time;
        cout << str << ":" << diff << endl;
    }

  private:
    clock_t s_time;
    string str;

};

} // namespace util
} // namespace sndlib

#endif // !defined(AFX_UTILLITY_H__1F7DEA24_379A_49A8_A97A_55C456684AE9__INCLUDED_)

