﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sndlib/ExprParser.h>
#include "seqconv.h"
#include "Command.h"

class Parser
{
  public:
    Parser( const CommandLineArg& cmdinfo );
    ~Parser(void);

    void Parse( const std::string& filename, std::istream& in, sndlib::strm::fstream& out );

    typedef sndlib::ExprParser::SymbolTable SymbolTable;
    typedef sndlib::ExprParser::SymbolTable LabelTable;

  private:
    typedef std::map< std::string, Command > CommandTable;
    void ParseBody( sndlib::Lexer& lexer, sndlib::strm::fstream& out );

    CommandLineArg mCmdInfo;

  public:
    Command mNoteCommand;
    Command mNoteRandCommand;
    Command mNoteVarCommand;
    CommandTable mCommandTable;

    SymbolTable mSymbolTable;

    std::string mCurGlobalLabel;
    LabelTable mGlobalLabelTable;
    std::map< std::string, LabelTable > mLocalLabelTable;

  private:
    long mDataBlockOffset;

    bool mFirstFlag;

  public:
    typedef std::vector< std::pair< std::string, int > > SeqLabelTable;
};



template<>
inline void Command::ArgNumber<Command::u24>::Parse(
    sndlib::Lexer& lexer, sndlib::strm::fstream& out, bool testFlag, Parser& parser )
{
    sndlib::ExprParser exprParser( testFlag );
    __int64 x = exprParser.Parse(
        lexer,
        &parser.mSymbolTable,
        &parser.mGlobalLabelTable,
        &parser.mLocalLabelTable[ parser.mCurGlobalLabel ]
    );

    if ( ! testFlag ) {
        lexer.VerifyMinMaxValue( mName, x, mMin, mMax );
    }

    sndlib::strm::binary<u8, TargetEndian> y;
    y = static_cast<u8>( x & 0xff );
    out << y;
    y = static_cast<u8>( ( x >> 8 ) & 0xff );
    out << y;
    y = static_cast<u8>( ( x >> 16 ) & 0xff );
    out << y;
}

template<>
inline void Command::ArgNumber<Command::vmidi>::Parse(
    sndlib::Lexer& lexer, sndlib::strm::fstream& out, bool testFlag, Parser& parser )
{
    sndlib::ExprParser exprParser( testFlag );
    __int64 x = exprParser.Parse( lexer, &parser.mSymbolTable );

    if ( ! testFlag ) {
        lexer.VerifyMinMaxValue( mName, x, mMin, mMax );
    }

    sndlib::strm::binary<u8, TargetEndian> y;
    if ( x < 0x80 ) {
        y = static_cast<u8>( x );
        out << y;
    }
    else if ( x < 0x4000 ) {
        y = static_cast<u8>( 0x80 | ( ( x >> 7 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x00 | ( ( x >> 0 ) & 0x7f ) );
        out << y;
    }
    else if ( x < 0x200000 ) {
        y = static_cast<u8>( 0x80 | ( ( x >> 14 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x80 | ( ( x >> 7 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x00 | ( ( x >> 0 ) & 0x7f ) );
        out << y;
    }
    else {
        y = static_cast<u8>( 0x80 | ( ( x >> 21 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x80 | ( ( x >> 14 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x80 | ( ( x >> 7 ) & 0x7f ) );
        out << y;
        y = static_cast<u8>( 0x00 | ( ( x >> 0 ) & 0x7f ) );
        out << y;
    }
}

template< typename T >
inline void Command::ArgNumber<T>::Parse(
    sndlib::Lexer& lexer, sndlib::strm::fstream& out, bool testFlag, Parser& parser )
{
    sndlib::ExprParser exprParser( testFlag );
    __int64 x = exprParser.Parse(
        lexer,
        &parser.mSymbolTable,
        &parser.mGlobalLabelTable,
        &parser.mLocalLabelTable[ parser.mCurGlobalLabel ]
    );

    if ( ! testFlag ) {
        lexer.VerifyMinMaxValue( mName, x, mMin, mMax );
    }

    sndlib::strm::binary<T, TargetEndian> y = static_cast<T>( x );
    out << y;
}

