﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "WaveDataUtility.h"
#include "../../seq2wav/WaveDumpUtility.h"
#include <nw/snd.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_StreamSoundFileReader.h>
#include <winext/cafe/ax.h>

using namespace nw::internal::winext;

// =============================================================================
// function
// =============================================================================
namespace
{
// =============================================================================
bool IsBfwav(const void* data)
{
    const char* tmp = reinterpret_cast<const char*>(data);
    if (tmp[0] == 'F' &&
        tmp[1] == 'W' &&
        tmp[2] == 'A' &&
        tmp[3] == 'V')
    {
        return true;
    }
    return false;
}

// -----------------------------------------------------------------------------
bool IsBfstm(const void* data)
{
    const char* tmp = reinterpret_cast<const char*>(data);
    if (tmp[0] == 'F' &&
        tmp[1] == 'S' &&
        tmp[2] == 'T' &&
        tmp[3] == 'M')
    {
        return true;
    }
    return false;
}

// -----------------------------------------------------------------------------
int FrameToNibble(int sampleCount)
{
    const int SamplesPerFrame = 14;
    const int NibblesPerBlcok = 16;
    const int NibblesPerFrameHeader = 2;
    int block = (sampleCount / SamplesPerFrame);
    int rest = (sampleCount % SamplesPerFrame);
    return block * NibblesPerBlcok + rest + NibblesPerFrameHeader; // [nibble]
}

// =============================================================================
// Decode 関数 (入力サンプルを「符号付き 16bit LE」にデコードする)
// =============================================================================
// -----------------------------------------------------------------------------
bool DecodePcm8(const void* input, void* output, int sampleLength, size_t outputSize)
{
    if (outputSize < sampleLength * sizeof(int16_t))
    {
        return false;
    }
    const int8_t* in = reinterpret_cast<const int8_t*>(input);
    int16_t* out = reinterpret_cast<int16_t*>(output);
    for (int i = 0; i < sampleLength; i++)
    {
        int16_t s = static_cast<int16_t>(in[i] << 8);
        out[i] = s;
    }
    return true;
}

// -----------------------------------------------------------------------------
bool DecodePcm16BE(const void* input, void* output, int sampleLength, size_t outputSize)
{
    if (outputSize < sampleLength * sizeof(int16_t))
    {
        return false;
    }
    const int16_t* in = reinterpret_cast<const int16_t*>(input);
    int16_t* out = reinterpret_cast<int16_t*>(output);
    for (int i = 0; i < sampleLength; i++)
    {
        uint16_t s = in[i];
        s = (s<<8) | (s>>8);
        out[i] = static_cast<int16_t>(s);
    }
    return true;
}

// -----------------------------------------------------------------------------
bool DecodeDspadpcm(const void* input, void* output, int sampleLength, size_t outputSize, DSPADPCM* adpcm)
{
    if (outputSize < sampleLength * sizeof(int16_t))
    {
        return false;
    }
    const uint8_t* in = reinterpret_cast<const uint8_t*>(input);
    int16_t* out = reinterpret_cast<int16_t*>(output);
    int decodedSize = AXDecodeAdpcmData(in, adpcm, sampleLength, out);
    if (decodedSize != sampleLength)
    {
        return false;
    }
    return true;
}

// -----------------------------------------------------------------------------
void SetupDSPADPCM(DSPADPCM* adpcm, const nw::snd::DspAdpcmParam& param, uint32_t endAddrNibble)
{
    NW_ASSERT_NOT_NULL(adpcm);
    for (int i = 0; i < 16; i++)
    {
        adpcm->coef[i] = param.coef[i/2][i%2];
    }
    adpcm->ps = param.predScale;
    adpcm->yn1 = param.yn1;
    adpcm->yn2 = param.yn2;
    adpcm->num_adpcm_nibbles = endAddrNibble;
    adpcm->sa = 2;
    adpcm->ea = endAddrNibble;
    adpcm->ca = 2;
}


// =============================================================================
class IWaveDataReader
{
public:
    struct LoopInfo
    {
        int32_t begin;
        int32_t end;
    };
    virtual ~IWaveDataReader() {}
    virtual size_t GetRequiredMemorySizeForDecodedSamples() const = 0;
    virtual bool GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size) const = 0;
    virtual bool GetAdpcmContext(int channel, int offsetSample, int* adjustOffsetSample, nn::atkTool::WaveDataUtility::AdpcmContext* context) const = 0;
    virtual int GetChannelCount() const = 0;
    virtual int GetSampleCount() const = 0;
    virtual int GetSampleRate() const = 0;
    virtual bool IsLoop() const = 0;
    virtual bool GetLoopInfo(LoopInfo& info) const = 0;
};

// =============================================================================
class BfwavReader : public IWaveDataReader
{
public:
    BfwavReader(const void* bfwav);
    ~BfwavReader();
    virtual size_t GetRequiredMemorySizeForDecodedSamples() const;
    virtual bool GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size) const;
    virtual bool GetAdpcmContext(int channel, int offsetSample, int* adjustOffsetSample, nn::atkTool::WaveDataUtility::AdpcmContext* context) const;
    virtual int GetChannelCount() const;
    virtual int GetSampleCount() const;
    virtual int GetSampleRate() const;
    virtual bool IsLoop() const { return m_WaveInfo.loopFlag; }
    virtual bool GetLoopInfo(LoopInfo& info) const;
private:
    nw::snd::internal::WaveFileReader m_Reader;
    nw::snd::internal::WaveInfo m_WaveInfo;
};

// =============================================================================
class BfstmReader : public IWaveDataReader
{
public:
    BfstmReader(const void* bfstm);
    ~BfstmReader();
    virtual size_t GetRequiredMemorySizeForDecodedSamples() const;
    virtual bool GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size) const;
    virtual bool GetAdpcmContext(int channel, int offsetSample, int* adjustOffsetSample, nn::atkTool::WaveDataUtility::AdpcmContext* context) const;
    virtual int GetChannelCount() const;
    virtual int GetSampleCount() const;
    virtual int GetSampleRate() const;
    virtual bool IsLoop() const
    {
        return m_StreamInfo.isLoop;
    }
    virtual bool GetLoopInfo(LoopInfo& info) const;
private:
    nw::snd::internal::StreamSoundFileReader m_Reader;
    nw::snd::internal::StreamSoundFile::StreamSoundInfo m_StreamInfo;
    const void* m_pBfstm;
};

// =============================================================================
class DecodeBuffer
{
public:
    DecodeBuffer(int channelCount, int sampleCount)
        : m_pBufferArray(nullptr)
        , m_ChannelCount(channelCount)
        , m_SampleCount(sampleCount)
    {
        m_pBufferArray = new int16_t*[m_ChannelCount];
        for (int ch = 0; ch < m_ChannelCount; ch++)
        {
            m_pBufferArray[ch] = new int16_t[m_SampleCount];
        }
    }
    ~DecodeBuffer()
    {
        for (int ch = 0; ch < m_ChannelCount; ch++)
        {
            delete [] m_pBufferArray[ch];
        }
        delete [] m_pBufferArray;
    }
    int16_t* GetDecodeBuffer(int channel)
    {
        if (m_ChannelCount <= channel)
        {
            return nullptr;
        }
        return m_pBufferArray[channel];
    }
private:
    int16_t** m_pBufferArray;
    int m_ChannelCount;
    int m_SampleCount;
};

// =============================================================================
struct ReaderInstance
{
    // ファイルダンプ用デバッグメンバー
    bool isDump;
    int8_t padding[3];
    const char* filenamePrefix;

    // デコード用メンバー
    IWaveDataReader* reader;
    static const int ReaderSize = sizeof(BfwavReader) > sizeof(BfstmReader) ? sizeof(BfwavReader) : sizeof(BfstmReader);
    char readerImpl[ReaderSize];
    // void* readerImpl;
};


// =============================================================================
BfwavReader::BfwavReader(const void* bfwav)
    : m_Reader(bfwav)
{
    m_Reader.ReadWaveInfo(&m_WaveInfo);
}

// -----------------------------------------------------------------------------
BfwavReader::~BfwavReader() {}

// -----------------------------------------------------------------------------
int BfwavReader::GetChannelCount() const
{
    return m_WaveInfo.channelCount;
}

// -----------------------------------------------------------------------------
int BfwavReader::GetSampleCount() const
{
    return m_WaveInfo.loopEndFrame;
}

// -----------------------------------------------------------------------------
size_t BfwavReader::GetRequiredMemorySizeForDecodedSamples() const
{
    return GetSampleCount() * sizeof(int16_t);
}

// -----------------------------------------------------------------------------
int BfwavReader::GetSampleRate() const
{
    return m_WaveInfo.sampleRate;
}

// -----------------------------------------------------------------------------
bool BfwavReader::GetLoopInfo(LoopInfo& info) const
{
    if (IsLoop() == false)
    {
        return false;
    }
    info.begin = static_cast<int32_t>(m_WaveInfo.loopStartFrame);
    info.end = static_cast<int32_t>(m_WaveInfo.loopEndFrame - 1);
    return true;
}

// -----------------------------------------------------------------------------
bool BfwavReader::GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size) const
{
    int16_t* outputs = reinterpret_cast<int16_t*>(bufferForDecodedSamples);
    const void* dataAddr = m_WaveInfo.channelParam[channel].dataAddress;
    const int SampleCount = GetSampleCount();
    if (SampleCount * sizeof(int16_t) > size)
    {
        return false;
    }

    bool result = false;
    switch (m_WaveInfo.sampleFormat)
    {
    case nw::snd::SAMPLE_FORMAT_DSP_ADPCM:
        {
            DSPADPCM adpcm = {0};
            {
                const nw::snd::DspAdpcmParam& param = m_WaveInfo.channelParam[channel].adpcmParam;
                SetupDSPADPCM(&adpcm, param, FrameToNibble(SampleCount));

                adpcm.num_samples = SampleCount;
            }
            result = DecodeDspadpcm(dataAddr, outputs, SampleCount, size, &adpcm);
        }
        break;
    case nw::snd::SAMPLE_FORMAT_PCM_S16:
        {
            result = DecodePcm16BE(dataAddr, outputs, SampleCount, size);
        }
        break;
    case nw::snd::SAMPLE_FORMAT_PCM_S8:
        {
            result = DecodePcm8(dataAddr, outputs, SampleCount, size);
        }
        break;
    }
    return result;
}

// -----------------------------------------------------------------------------
bool BfwavReader::GetAdpcmContext(int channel, int offsetSample, int* adjustOffsetSample, nn::atkTool::WaveDataUtility::AdpcmContext* context) const
{
    nw::snd::internal::WaveFileReader::OffsetContextInfo offsetContextInfoCache;
    bool result = m_Reader.CalcOffsetContextInfo(offsetSample, &offsetContextInfoCache);
    if (result)
    {
        *adjustOffsetSample = offsetContextInfoCache.offsetSample;
        *context = *reinterpret_cast<nn::atkTool::WaveDataUtility::AdpcmContext*>(&offsetContextInfoCache.contexts[channel]);
    }
    return result;
}




// =============================================================================
BfstmReader::BfstmReader(const void* bfstm)
    : m_Reader()
    , m_pBfstm(bfstm)
{
    m_Reader.Initialize(bfstm);
    m_Reader.ReadStreamSoundInfo(&m_StreamInfo);
}

// -----------------------------------------------------------------------------
BfstmReader::~BfstmReader()
{
    m_Reader.Finalize();
}

// -----------------------------------------------------------------------------
int BfstmReader::GetChannelCount() const
{
    return m_StreamInfo.channelCount;
}

// -----------------------------------------------------------------------------
int BfstmReader::GetSampleCount() const
{
    return m_StreamInfo.frameCount;
}

// -----------------------------------------------------------------------------
size_t BfstmReader::GetRequiredMemorySizeForDecodedSamples() const
{
    return GetSampleCount() * sizeof(int16_t);
}

// -----------------------------------------------------------------------------
int BfstmReader::GetSampleRate() const
{
    return m_StreamInfo.sampleRate;
}

// -----------------------------------------------------------------------------
bool BfstmReader::GetLoopInfo(LoopInfo& info) const
{
    if (IsLoop() == false)
    {
        return false;
    }
    info.begin = static_cast<int32_t>(m_StreamInfo.loopStart);
    info.end = static_cast<int32_t>(m_StreamInfo.frameCount) - 1;
    return true;
}

// -----------------------------------------------------------------------------
bool BfstmReader::GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size) const
{
    int16_t* outputs = reinterpret_cast<int16_t*>(bufferForDecodedSamples);
    nw::snd::SampleFormat format = static_cast<nw::snd::SampleFormat>(m_StreamInfo.encodeMethod);

    // ブロックのデインターリーブに必要なパラメータの定義
    const int BlockBytesPeurChannel = m_StreamInfo.oneBlockBytes;
    const int BlockBytes = m_StreamInfo.channelCount * BlockBytesPeurChannel;
        // ↑ 複数チャンネルを考慮したブロックサイズ (1ch なら 8KB、2ch なら 16KB、4ch なら 32KB)
    const int SampleCount = m_StreamInfo.oneBlockSamples;
    const int BlockCount = m_StreamInfo.blockCount;
    const int DataOffset = m_Reader.GetSampleDataOffset(); // 読み込むデータのオフセット
    uint32_t bufferOffset = 0;  // 書き込むバッファのオフセット

    // ADPCM パラメータの生成
    DSPADPCM adpcm = {0};
    int16_t yn1 = 0;
    int16_t yn2 = 0;
    if (format == nw::snd::SAMPLE_FORMAT_DSP_ADPCM)
    {
        nw::snd::DspAdpcmParam param;
        nw::snd::internal::DspAdpcmLoopParam loopParam;
        bool result = m_Reader.ReadDspAdpcmChannelInfo(&param, &loopParam, channel);
        if (result == false)
        {
            return false;
        }

        SetupDSPADPCM(&adpcm, param, BlockBytesPeurChannel * 2);

        // 最終ブロックでのみ書き換える
        adpcm.num_samples = SampleCount;
    }

    // ブロックのデインターリーブ
    for (int blockIndex = 0; blockIndex < BlockCount; blockIndex++)
    {
        // 1 ブロックのサイズ (オフセット)・サンプル数
        int offsetPerChannel = BlockBytesPeurChannel;
        int sampleCount = SampleCount;
        bool isLastBlock = (blockIndex == (m_StreamInfo.blockCount - 1)) ? true : false;
        if (isLastBlock)
        {
            offsetPerChannel = m_StreamInfo.lastBlockPaddedBytes;
            sampleCount = m_StreamInfo.lastBlockSamples;
        }
        int dataOffset = DataOffset + (BlockBytes * blockIndex) + (offsetPerChannel * channel);

        // デコードすべきデータの先頭アドレス
        const void* dataAddr = nw::ut::AddOffsetToPtr(m_pBfstm, dataOffset);

        // デコード後のデータを詰め込むバッファの先頭アドレスとサイズ
        void* writeAddr = nw::ut::AddOffsetToPtr(outputs, bufferOffset);
        size_t decodeBufferSize = sampleCount * sizeof(int16_t);
        bufferOffset += decodeBufferSize;
        if (bufferOffset > size)
        {
            return false;
        }

        // ブロックごとにデコード
        bool result = false;
        switch (format)
        {
        case nw::snd::SAMPLE_FORMAT_DSP_ADPCM:
            {
                if (isLastBlock)
                {
                    adpcm.num_samples = sampleCount;

                    // SetupDSPADPCM で設定される num_adpcm_nibbles と ea をここで設定する
                    adpcm.num_adpcm_nibbles = m_StreamInfo.lastBlockBytes * 2;
                    adpcm.ea = m_StreamInfo.lastBlockBytes * 2;
                }
                if (blockIndex != 0) // コンテキストを更新 (blockIndex==0のときは、for の前に設定済み)
                {
                    adpcm.ps = reinterpret_cast<const u8*>(dataAddr)[0];
                    adpcm.yn1 = yn1;
                    adpcm.yn2 = yn2;
                    // adpcm.ps は adpcm.ca = 0 とすることで、ADPCM フレームの 0 バイト目から
                    // 読み取ってくれるため設定不要。
                }
                result = DecodeDspadpcm(dataAddr, writeAddr, sampleCount, decodeBufferSize, &adpcm);
                int16_t* decodedSample = reinterpret_cast<int16_t*>(writeAddr);
                yn1 = decodedSample[sampleCount-1];
                yn2 = decodedSample[sampleCount-2];
            }
            break;
        case nw::snd::SAMPLE_FORMAT_PCM_S16:
            {
                result = DecodePcm8(dataAddr, writeAddr, sampleCount, decodeBufferSize);
            }
            break;
        case nw::snd::SAMPLE_FORMAT_PCM_S8:
            {
                result = DecodePcm8(dataAddr, writeAddr, sampleCount, decodeBufferSize);
            }
            break;
        }

        if (result == false)
        {
            return false;
        }
    }
    return true;
}

// -----------------------------------------------------------------------------
bool BfstmReader::GetAdpcmContext(int channel, int offsetSample, int* adjustOffsetSample, nn::atkTool::WaveDataUtility::AdpcmContext* context) const
{
    return false;
    // ユーザーからの要求があるまで実装しない
}
} // namespace


// =============================================================================
namespace nn { namespace atkTool {


// =============================================================================
class WaveDataUtility::WaveDataUtilityImpl : public ReaderInstance
{
public:
    // -------------------------------------------------------------------------
    WaveDataUtilityImpl()
        : m_State(State_Create)
    {}

    bool IsInitialize() const
    {
        return m_State == State_Initialize ? true : false;
    }
    bool IsFinalize() const
    {
        return m_State == State_Finalize ? true : false;
    }
    void SetStateInitialize()
    {
        m_State = State_Initialize;
    }
    void SetStateFinalize()
    {
        m_State = State_Finalize;
    }

private:
    // -------------------------------------------------------------------------
    enum State
    {
        State_Create,
        State_Initialize,
        State_Finalize
    };

    // -------------------------------------------------------------------------
    State m_State;
};

// =============================================================================
WaveDataUtility::WaveDataUtility()
    : m_pImpl(nullptr)
{
}


// =============================================================================
WaveDataUtility::~WaveDataUtility()
{
    Finalize();
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::Initialize(const void* data)
{
    // エラーチェック
    {
        if (m_pImpl && m_pImpl->IsInitialize())
        {
            return Result_Success;
        }
        if (data == nullptr)
        {
            return Result_NullData;
        }
    }

    // インスタンス生成
    m_pImpl = new WaveDataUtilityImpl();
    if (m_pImpl == nullptr)
    {
        return Result_OutOfMemory;
    }

    // ファイルリーダー生成
    if (IsBfwav(data))
    {
        m_pImpl->reader = new (m_pImpl->readerImpl) BfwavReader(data);
    }
    else if (IsBfstm(data))
    {
        m_pImpl->reader = new (m_pImpl->readerImpl) BfstmReader(data);
    }
    else
    {
        return Result_InvalidFormat;
    }

    // ファイルダンプ (デバッグ目的)
    {
        m_pImpl->isDump = false;
        m_pImpl->filenamePrefix = nullptr;

        // NOTE: Initialize 後、ユーザーが true / "hoge" などをセットすると、
        // GetDecodedSamples 時に "hoge.1.wav" のようなファイル名で出力されます。
        // (1 は GetDecodedSamples に渡す channel です)
    }

    m_pImpl->SetStateInitialize();
    return Result_Success;
}



// =============================================================================
WaveDataUtility::Result WaveDataUtility::DumpDecodedSamples(const char* dumpFilePath)
{
    // エラーチェック
    {
        if (m_pImpl == nullptr || m_pImpl->IsInitialize() == false)
        {
            return Result_BeforeInitialize;
        }
        if (dumpFilePath == nullptr)
        {
            return Result_NullDumpFilePath;
        }
    }

    const int ChannelCount = m_pImpl->reader->GetChannelCount();
    const int SampleRate = m_pImpl->reader->GetSampleRate();
    const int SampleCount = m_pImpl->reader->GetSampleCount();

    DecodeBuffer buffer(ChannelCount, SampleCount);

    // デコード
    for (int ch = 0; ch < ChannelCount; ch++)
    {
        Result result = GetDecodedSamples(ch, buffer.GetDecodeBuffer(ch), SampleCount * sizeof(int16_t));
        if (result != Result_Success)
        {
            return result;
        }
    }

    // インターリーブしながらダンプ
    {
        internal::WaveDumpInstance dumpInstance;
        bool ret = internal::WaveDumpInitialize(&dumpInstance, dumpFilePath, ChannelCount, SampleRate);
        if (ret == false)
        {
            return Result_FailedToOpen;
        }

        for (int sample = 0; sample < SampleCount; sample++)
        {
            const int ChannelMax = 128;
            int16_t frame[ChannelMax] = {0};
            for (int ch = 0; ch < ChannelCount; ch++)
            {
                frame[ch] = buffer.GetDecodeBuffer(ch)[sample];
            }
            ret = internal::WaveDumpWrite(&dumpInstance, frame, ChannelCount * sizeof(int16_t));
            NW_ASSERT(ret);
        }

        // ループ情報の埋め込み
        if (m_pImpl->reader->IsLoop())
        {
            IWaveDataReader::LoopInfo info;
            ret = m_pImpl->reader->GetLoopInfo(info);
            NW_ASSERT(ret); // IsLoop が true のときは false が入り得ない

            ret = internal::WaveDumpSetLoop(&dumpInstance, info.begin, info.end);
            NW_ASSERT(ret);
        }

        ret = internal::WaveDumpFinalize(&dumpInstance);
        NW_ASSERT(ret);
    }

    return Result_Success;
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::GetRequiredMemorySizeForDecodedSamples(size_t* size)
{
    // エラーチェック
    {
        if (m_pImpl == nullptr || m_pImpl->IsInitialize() == false)
        {
            return Result_BeforeInitialize;
        }
        if (size == nullptr)
        {
            return Result_NullSize;
        }
    }

    size_t result = m_pImpl->reader->GetRequiredMemorySizeForDecodedSamples();
    *size = result;

    return Result_Success;
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::GetDecodedSamples(int channel, void* bufferForDecodedSamples, size_t size)
{
    // エラーチェック
    {
        if (m_pImpl == nullptr || m_pImpl->IsInitialize() == false)
        {
            return Result_BeforeInitialize;
        }
        if (bufferForDecodedSamples == nullptr)
        {
            return Result_NullBuffer;
        }
        size_t requiredSize = m_pImpl->reader->GetRequiredMemorySizeForDecodedSamples();
        if (requiredSize > size)
        {
            return Result_InvalidSize;
        }
        if (m_pImpl->reader->GetChannelCount() <= channel)
        {
            return Result_InvalidChannel;
        }
    }

    bool result = m_pImpl->reader->GetDecodedSamples(channel, bufferForDecodedSamples, size);
    if (result)
    {
        // デバッグ目的
        if (m_pImpl->isDump && m_pImpl->filenamePrefix)
        {
            char filename[MAX_PATH] = {'\0'};
            _snprintf_s(filename, sizeof(filename) / sizeof(filename[0]),
                    MAX_PATH, "%s.%d.wav", m_pImpl->filenamePrefix, 0);
            std::printf("dump filename(%s)\n", filename);

            internal::WaveDumpInstance dumpInstance;
            bool ret = internal::WaveDumpInitialize(&dumpInstance, filename, 1, m_pImpl->reader->GetSampleRate());
            NW_ASSERT(ret);
            ret = internal::WaveDumpWrite(&dumpInstance, bufferForDecodedSamples, size);
            NW_ASSERT(ret);
            ret = internal::WaveDumpFinalize(&dumpInstance);
            NW_ASSERT(ret);
        }
        return Result_Success;
    }
    else
    {
        return Result_FailureDecode;
    }
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::GetAdpcmContext(
        int channel, int offsetSample, int* adjustOffsetSample,
        WaveDataUtility::AdpcmContext* context)
{
    // エラーチェック
    {
        if (m_pImpl == nullptr || m_pImpl->IsInitialize() == false)
        {
            return Result_BeforeInitialize;
        }
        if (adjustOffsetSample == nullptr)
        {
            return Result_NullAdjustOffsetSample;
        }
        if (context == nullptr)
        {
            return Result_NullAdpcmContext;
        }
        if (m_pImpl->reader->GetChannelCount() <= channel)
        {
            return Result_InvalidChannel;
        }
        if (m_pImpl->reader->GetSampleCount() <= offsetSample)
        {
            return Result_InvalidOffsetSample;
        }
    }

    bool result = m_pImpl->reader->GetAdpcmContext(channel, offsetSample, adjustOffsetSample, context);
    if (result)
    {
        return Result_Success;
    }
    else
    {
        return Result_Unsupported;
    }
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::GetWaveInfo(WaveDataUtility::WaveInfo* info)
{
    // エラーチェック
    {
        if (m_pImpl == nullptr || m_pImpl->IsInitialize() == false)
        {
            return Result_BeforeInitialize;
        }
        if (info == nullptr)
        {
            return Result_NullWaveInfo;
        }
    }

    info->channelCount = m_pImpl->reader->GetChannelCount();
    info->sampleRate = m_pImpl->reader->GetSampleRate();
    info->sampleCount = m_pImpl->reader->GetSampleCount();
    info->loopFlag = m_pImpl->reader->IsLoop();
    if (info->loopFlag)
    {
        IWaveDataReader::LoopInfo loopInfo;
        bool result = m_pImpl->reader->GetLoopInfo(loopInfo);
        NW_ASSERT(result);

        info->loopStart = loopInfo.begin;
        info->loopEnd = loopInfo.end;
    }
    return Result_Success;
}


// =============================================================================
WaveDataUtility::Result WaveDataUtility::Finalize()
{
    if (m_pImpl && m_pImpl->IsInitialize())
    {
        m_pImpl->reader->~IWaveDataReader();
        m_pImpl->SetStateFinalize();
        delete m_pImpl;
        m_pImpl = nullptr;
    }
    return Result_Success;
}

}} // nn::atkTool
