﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.WaveFileFormat
{

    public class WaveFileAiffReader : WaveFileReader, IDisposable
    {
        FileStream strm;
        BinaryReader reader;
        long dataBegin;
        long dataEnd;
        long curSample;
        WaveFileImpl.ToInt16Handler toInt16;
        UInt32 numSampleFrames;
        Int16 numChannels;
        Int16 sampleSize;

        void IDisposable.Dispose()
        {
            strm.Close();
        }

        public override WaveFile Open(string filePath)
        {
            return OpenAiff(filePath);
        }

        public WaveFileAiff OpenAiff(string filePath)
        {
            WaveFileAiff waveFile = new WaveFileAiff();

            strm = File.OpenRead(filePath);
            reader = BinaryReaderBigEndian.CreateInstance(strm, new System.Text.ASCIIEncoding());

            string fileChunk = new string(reader.ReadChars(4));
            if (fileChunk != "FORM")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            Int32 fileSize = reader.ReadInt32();
            string format = new string(reader.ReadChars(4));
            if (format != "AIFF")
            {
                throw new WaveFileReaderException("Invalid format");
            }

            while (strm.Position < strm.Length)
            {
                string chunkType = new string(reader.ReadChars(4));
                UInt32 chunkSize = reader.ReadUInt32();
                long chunkEnd = strm.Position + chunkSize;
                long nextChunkPos = ((chunkEnd + 0x01) & ~0x01); // 2バイトアライン

                if (chunkType == "COMM")
                {
                    WaveFileAiff.AiffCommonChunk commonChunk = waveFile.CommonChunk;
                    commonChunk.Read(reader);

                    numSampleFrames = commonChunk.numSampleFrames;
                    numChannels = commonChunk.numChannels;
                    sampleSize = commonChunk.sampleSize;
                }
                else if (chunkType == "SSND")
                {
                    waveFile.SoundDataChunk.Read(reader);

                    dataBegin = strm.Position;
                    dataEnd = chunkEnd;
                }
                else if (chunkType == "INST")
                {
                    waveFile.InstChunk.Read(reader);
                }
                else if (chunkType == "MARK")
                {
                    waveFile.MarkerChunk.Read(reader);
                }
                else
                {
                    // unknown chunk
                }

                strm.Position = nextChunkPos;
            }

            uint sampleByte = (uint)(sampleSize + 7) / 8;
            switch (sampleByte)
            {
                case 1:
                    toInt16 = new WaveFileImpl.ToInt16Handler(WaveFileImpl.S8ToInt16);
                    break;
                case 2:
                    toInt16 = new WaveFileImpl.ToInt16Handler(WaveFileImpl.BigS16ToInt16);
                    break;
                case 3:
                    toInt16 = new WaveFileImpl.ToInt16Handler(WaveFileImpl.BigS24ToInt16);
                    break;
                case 4:
                    toInt16 = new WaveFileImpl.ToInt16Handler(WaveFileImpl.BigS32ToInt16);
                    break;
                default:
                    throw new WaveFileReaderException("Unsupported sample type");
            }

            curSample = 0;
            strm.Position = dataBegin;

            return waveFile;
        }

        public override void Close()
        {
            strm.Close();
        }

        public override Int16[] ReadInt16(long samples)
        {
            long endSample = Math.Min(curSample + samples, numSampleFrames * numChannels);
            long readSamples = endSample - curSample;
            int sampleByte = (sampleSize + 7) / 8;
            long readSize = readSamples * sampleByte;

            byte[] buffer = reader.ReadBytes(Convert.ToInt32(readSize));
            Int16[] sampleBuffer = toInt16(buffer); ;

            curSample += sampleBuffer.Length;

            return sampleBuffer;
        }
    }
}
