﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Runtime.Serialization;
using System.Diagnostics;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{

    public class SoundException : ApplicationException
    {
        public SoundException() : base() { }
        public SoundException(string str) : base(str) { }
        public SoundException(string str, Exception e) : base(str, e) { }
        protected SoundException(SerializationInfo info, StreamingContext sc) : base(info, sc) { }
    }

    internal class IntConverter
    {
        public static int Parse(string str, KeyValuePair<int, string>[] table)
        {
            foreach (KeyValuePair<int, string> kvp in table)
            {
                if (str == kvp.Value) return kvp.Key;
            }
            throw new SoundException("IntConverter.Parse: invalid string \"" + str + "\"");
        }
        public static string ToString(int value, KeyValuePair<int, string>[] table)
        {
            foreach (KeyValuePair<int, string> kvp in table)
            {
                if (value.Equals(kvp.Key)) return kvp.Value;
            }
            throw new SoundException("IntConverter.ToString: invalid value");
        }
    }

    public class Util
    {
        public static int CompareFileByLastWriteTime(string targetFilePath, params string[] dependFileList)
        {
            try
            {
                int result = -1;
                foreach (string dependFilePath in dependFileList)
                {
                    int comp = DateTime.Compare(
                        File.GetLastWriteTime(targetFilePath),
                        File.GetLastWriteTime(dependFilePath)
                    );
                    if (comp < 0) return -1;
                    result = Math.Max(result, comp);
                }
                return result;
            }
            catch
            {
                return -1;
            }
        }
        public static string ConvertKeyNoToString(int key)
        {
            int oct = key / 12;
            key = (oct >= 0) ?
                  key - oct * 12 :
                  key + 12 - oct * 12;
            --oct;

            string[] keystr = new string[12] {
                "cn", "cs", "dn", "ds", "en", "fn",
                "fs", "gn", "gs", "an", "as", "bn"
            };
            StringBuilder sb = new StringBuilder();
            sb.Append(keystr[key]);
            if (oct >= 0)
            {
                sb.Append(oct);
            }
            else
            {
                sb.Append('m');
                sb.Append(-oct);
            }

            return sb.ToString();
        }
        public static int ConvertStringToKeyNo(string str)
        {
            int pos = 0;
            if (pos >= str.Length)
            {
                throw new SoundException("Invalid key string format \"" + str + "\"");
            }

            int key = 0;
            switch (str[pos])
            {
                case 'c': case 'C': key = 0; break;
                case 'd': case 'D': key = 2; break;
                case 'e': case 'E': key = 4; break;
                case 'f': case 'F': key = 5; break;
                case 'g': case 'G': key = 7; break;
                case 'a': case 'A': key = 9; break;
                case 'b': case 'B': key = 11; break;
                default:
                    throw new SoundException("Invalid key string format \"" + str + "\"");
            }

            pos++;
            if (pos >= str.Length)
            {
                throw new SoundException("Invalid key string format \"" + str + "\"");
            }

            switch (str[pos])
            {
                case 'n': case 'N': pos++; break;
                case 'f': case 'F': case 'b': pos++; key--; break;
                case '#': case 's': case 'S': pos++; key++; break;
                default: break;
            }

            if (pos >= str.Length)
            {
                throw new SoundException("Invalid key string format \"" + str + "\"");
            }

            bool minusFlag = false;
            if (str[pos] == 'm' || str[pos] == 'M' || str[pos] == '-')
            {
                minusFlag = true;
                pos++;
                if (pos >= str.Length)
                {
                    throw new SoundException("Invalid key string format \"" + str + "\"");
                }
            }

            int octave = 0;
            while (pos < str.Length)
            {
                octave *= 10;

                switch (str[pos])
                {
                    case '0': octave += 0; break;
                    case '1': octave += 1; break;
                    case '2': octave += 2; break;
                    case '3': octave += 3; break;
                    case '4': octave += 4; break;
                    case '5': octave += 5; break;
                    case '6': octave += 6; break;
                    case '7': octave += 7; break;
                    case '8': octave += 8; break;
                    case '9': octave += 9; break;
                    default:
                        throw new SoundException("Invalid key string format \"" + str + "\"");
                }

                pos++;
            }

            if (minusFlag) octave = -octave;

            key += (octave + 1) * 12;

            return key;
        }

        public class ProcessHandler
        {
            internal Process process = null;

            internal event EventHandler Killed;

            public void Kill()
            {
                if (process != null)
                {
                    process.Kill();
                }

                if (null != Killed)
                {
                    Killed(this, new EventArgs());
                }
            }
        }
        public delegate void ProcessOutputCallback(string msg);

        public static bool StartProcess(
            string command,
            string arg,
            string workingDir,
            ProcessOutputCallback outputCallback,
            ProcessOutputCallback errorCallback,
            ProcessHandler handler
        )
        {
            ProcessStartInfo startInfo = new ProcessStartInfo(command);
            startInfo.Arguments = arg;
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            startInfo.RedirectStandardOutput = true;
            startInfo.RedirectStandardError = true;
            startInfo.WorkingDirectory = workingDir;

            Process process = Process.Start(startInfo);
            if (process == null) return false;

            if (handler != null) handler.process = process;

            while (true)
            {
                string readLine = process.StandardOutput.ReadLine();
                if (readLine == null)
                {
                    break;
                }
                if (outputCallback != null)
                {
                    outputCallback(readLine);
                }
                else
                {
                    Console.WriteLine(readLine);
                }
            }
            while (true)
            {
                string readLine = process.StandardError.ReadLine();
                if (readLine == null)
                {
                    break;
                }
                if (errorCallback != null)
                {
                    errorCallback(readLine);
                }
                else
                {
                    Console.Error.WriteLine(readLine);
                }
            }

            process.WaitForExit();
            if (handler != null) handler.process = null;

            return process.ExitCode == 0;
        }

        public static bool StartProcess(
            string command,
            string arg,
            string workingDir,
            ProcessHandler handler
        )
        {
            ProcessStartInfo startInfo = new ProcessStartInfo(command);
            startInfo.Arguments = arg;
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = true;
            startInfo.WorkingDirectory = workingDir;

            Process process = Process.Start(startInfo);
            if (process == null) return false;

            if (handler != null) handler.process = process;

            process.WaitForExit();
            if (handler != null) handler.process = null;

            return process.ExitCode == 0;
        }

    }

}

