﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{
    /// <summary>
    /// PatriciaTree アイテムのインターフェイス。
    /// </summary>
    public interface IPatriciaTreeItem
    {
        /// <summary>
        /// アイテムのキーを取得します。
        /// </summary>
        string Key { get; }
    }

    /// <summary>
    /// PatriciaTree を構築するためのクラス。
    /// </summary>
    /// <typeparam name="IPatriciaTreeItem">PatriciaTree で扱うアイテムの種類。IPatriciaTreeData を実装する必要があります。</typeparam>
    public class PatriciaTree
    {
        #region ** 定数

        public const int InvalidIndex = -1;

        #endregion

        #region ** インターフェイス

        /// <summary>
        /// PatriciaTree Node を操作します。
        /// </summary>
        public interface INode
        {
            #region ** プロパティ

            /// <summary>
            /// インデックスを取得します。
            /// </summary>
            int Index { get; }

            /// <summary>
            /// LeftNode を取得します。
            /// </summary>
            INode Left { get; }

            /// <summary>
            /// RightNode を取得します。
            /// </summary>
            INode Right { get; }

            /// <summary>
            /// この Node のキーを取得します。
            /// </summary>
            string Key { get; }

            /// <summary>
            /// この Node の Bit を取得します。
            /// </summary>
            int Bit { get; }

            #endregion
        }

        /// <summary>
        /// PatriciaTree Leaf を操作します。
        /// </summary>
        public interface ILeaf : INode
        {
            #region ** プロパティ

            /// <summary>
            /// Node に関連付けられたアイテムを取得します。
            /// </summary>
            IPatriciaTreeItem Item { get; }

            #endregion
        }

        /// <summary>
        /// PatriciaTree BranchNode を操作します。
        /// </summary>
        public interface IBranchNode : INode { }

        #endregion


        #region ** パラメータ

        private Node _root = null;					// RootNode
        private NodeCollection _nodes = new NodeCollection();	// Node コンテナ
        private ItemCollection _items = new ItemCollection();	// アイテム コンテナ

        #endregion

        #region ** プロパティ

        /// <summary>
        /// PatriciaTree の RootNode を取得します。
        /// </summary>
        public INode Root
        {
            get { return _root; }
        }

        /// <summary>
        /// PatriciaTree に含まれるノードコレクションを取得します。
        /// </summary>
        public INodeCollection Nodes
        {
            get { return _nodes; }
        }

        /// <summary>
        /// PatriciaTree に含まれるアイテムコレクションを取得します。
        /// </summary>
        public IItemCollection Items
        {
            get { return _items; }
        }

        #endregion

        #region ** メソッド

        #region ** アイテムの追加と削除

        /// <summary>
        /// PatriciaTree にアイテムを追加します。
        /// </summary>
        /// <param name="item">アイテム</param>
        public void Add(IPatriciaTreeItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            _items.Add(item);
            AddNode(_root, item);
        }

        /// <summary>
        /// PatriciaTree から指定アイテムを削除します。
        /// </summary>
        /// <param name="item">アイテム</param>
        /// <returns>正常に削除された場合は true。アイテムが見つからなかった場合は false。</returns>
        public bool Remove(IPatriciaTreeItem item)
        {
            return Remove(item.Key);
        }

        /// <summary>
        /// PatriciaTree から指定キーを含むアイテムを削除します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>正常に削除された場合は true。アイテムが見つからなかった場合は false。</returns>
        public bool Remove(string key)
        {
            Node node = FindNode(key);
            if (null == node) { return false; }

            // Leaf 以外は削除できない
            Debug.Assert(node is ILeaf);

            // リンクを切って、コンテナからアイテムを削除する
            CutLink(node);

            _nodes.Remove(node);
            _items.Remove(key);

            return true;
        }

        /// <summary>
        /// PatriciaTree からすべての項目を削除します。
        /// </summary>
        public void Clear()
        {
            _nodes.Clear();
            _items.Clear();
        }

        /// <summary>
        /// PatriciaTree にノードを追加します。
        /// </summary>
        /// <param name="node">対象ノード</param>
        /// <param name="parentNode">node の親ノード</param>
        /// <param name="item">新しく追加するノードアイテム</param>
        private void AddNode(Node node, IPatriciaTreeItem item)
        {
            Debug.Assert(null != item);

            if (null == node)
            {

                Debug.Assert(null == _root);
                Debug.Assert(0 == _nodes.Count);

                _root = new Leaf(_nodes.Count, item);
                _nodes.Add(_root);

                return;

            }

            int curBit = (null == node.Parent) ? 0 : node.Parent.Bit;

            if (node is BranchNode)
            {

                int diffBit = BitHelper.CompareBit(node.Key, item.Key, curBit, node.Bit);

                if (0 <= diffBit)
                {

                    // ブランチして Leaf を追加する
                    Leaf newLeaf = new Leaf(_nodes.Count, item);
                    _nodes.Add(newLeaf);

                    Branch(node, newLeaf, diffBit);

                }
                else
                {
                    AddNode(BitHelper.GetBit(item.Key, node.Bit) ? node.Right : node.Left, item);
                }

            }
            else if (node is Leaf)
            {

                int diffBit = BitHelper.CompareBit(node.Key, item.Key, curBit);

                if (0 < diffBit)
                {

                    // ブランチして Leaf を追加する
                    Leaf newLeaf = new Leaf(_nodes.Count, item);
                    _nodes.Add(newLeaf);

                    Branch(node, newLeaf, diffBit);

                }
                else
                {
                    // キーが同じ Leaf は追加できない
                    throw new ArgumentException("item");
                }

            }
            else
            {
                // Invalid Node
                throw new Exception("invalid node type.");
            }
        }

        /// <summary>
        /// ノードを分岐します。
        /// </summary>
        /// <param name="node1">子ノード1</param>
        /// <param name="node2">子ノード2</param>
        /// <param name="bit">ブランチノードのビット</param>
        private void Branch(Node node, Node newNode, int bit)
        {
            if (BitHelper.GetBit(node.Key, bit) == BitHelper.GetBit(newNode.Key, bit))
            {
                Debug.Fail("Invalid argument.");
                throw new ArgumentException("node or newNode");
            }

            Node branchNode = new BranchNode(_nodes.Count, GetBranchKey(node, newNode, bit), bit);
            _nodes.Add(branchNode);

            // BranchNode をリンクする
            if (null != node.Parent)
            {

                if (node.Parent.Left == node)
                {
                    node.Parent.Left = branchNode;
                }
                else
                {
                    Debug.Assert(node.Parent.Right == node);
                    node.Parent.Right = branchNode;
                }

            }
            else
            {
                _root = branchNode;
            }

            Debug.Assert(node.Bit != bit);

            if (node.Bit > bit && BitHelper.GetBit(node.Key, bit))
            {
                branchNode.Left = newNode;
                branchNode.Right = node;
            }
            else
            {
                branchNode.Left = node;
                branchNode.Right = newNode;
            }
        }

        private string GetBranchKey(Node node1, Node node2, int bit)
        {
            Debug.Assert(0 <= bit);
            Debug.Assert(null != node1);
            Debug.Assert(null != node2);
            Debug.Assert((BitHelper.GetBitCount(node1.Key) > bit) || (BitHelper.GetBitCount(node2.Key) > bit));
            return BitHelper.SubString((BitHelper.GetBitCount(node1.Key) <= bit) ? node2.Key : node1.Key, bit);
        }

        private void CutLink(Node node)
        {
            if (null == node)
            {
                Debug.Fail("Target node must not be null.");
                throw new ArgumentNullException("node");
            }
            if (!(node is Leaf))
            {
                Debug.Fail("Target node must be leaf.");
                throw new Exception();
            }

            if (node == _root)
            {
                _root = null;
                return;
            }

            Debug.Assert(null != node.Parent);

            Node otherNode = (node == node.Parent.Left) ? node.Parent.Right : node.Parent.Left;
            Debug.Assert(node != otherNode);

            if (node.Parent == _root)
            {
                otherNode.Parent = null;
                _root = otherNode;
            }
            else
            {

                if (node.Parent == node.Parent.Parent.Left)
                {
                    node.Parent.Parent.Left = otherNode;
                }
                else
                {
                    Debug.Assert(node.Parent == node.Parent.Parent.Right);
                    node.Parent.Parent.Right = otherNode;
                }

            }
        }

        #endregion

        #region ** アイテムの参照

        /// <summary>
        /// 指定キーを含む子ノードを検索します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>見つかった場合は対象ノード、見つからなかった場合は null。</returns>
        private Node FindNode(string key)
        {
            return FindNode(_root, key);
        }

        /// <summary>
        /// 指定キーを含む子ノードを検索します。
        /// </summary>
        /// <param name="node">検索対象ノード</param>
        /// <param name="key">キー</param>
        /// <returns>見つかった場合は対象ノード、見つからなかった場合は null。</returns>
        private Node FindNode(Node node, string key)
        {
            if (null == node) { return null; }

            if (node is Leaf)
            {
                return (node.Key == key) ? node : null;
            }

            return FindNode(BitHelper.GetBit(key, node.Bit) ? node.Right : node.Left, key);
        }

        #endregion

        #endregion

        #region ** Node クラス

        protected class Node : INode
        {
            #region ** パラメータ

            // PatriciaTree パラメータ
            private int _index = 0;
            private Node _parent = null;
            private Node _left = null;
            private Node _right = null;
            private int _bit = 0;

            // データ
            private string _key = string.Empty;
            private IPatriciaTreeItem _item = null;

            #endregion

            protected Node(int index, string key, int bit) : this(index, key, bit, null) { }
            protected Node(int index, string key, int bit, IPatriciaTreeItem data)
            {
                if (null == key)
                {
                    Debug.Fail("key is null.");
                    throw new ArgumentNullException("key");
                }
                if (0 > bit)
                {
                    Debug.Fail("bit < 0");
                    throw new ArgumentOutOfRangeException("bit");
                }

                _index = index;
                _key = key;
                _bit = bit;
                _item = data;
            }

            #region ** プロパティ

            public int Index
            {
                get { return _index; }
            }

            public Node Parent
            {
                get { return _parent; }
                set
                {
                    if (value == _parent) { return; }

                    if (null != value)
                    {

                        // this.Key は、ParentNode.Key を含むはず
                        if (string.Compare(Key, 0, value.Key, 0, value.Key.Length) != 0) { throw new Exception(); }

                        // this.Key は ParentNode.Key よりも長いはず
                        if (BitHelper.GetBitCount(Key) <= value.Bit) { throw new Exception(); }

                        if (BitHelper.GetBit(Key, value.Bit))
                        {
                            //if( null != value.Right ) { throw new Exception(); }
                            value.Right = this;
                        }
                        else
                        {
                            //if( null != value.Left ) { throw new Exception(); }
                            value.Left = this;
                        }

                    }

                    _parent = value;
                }
            }

            public Node Left
            {
                get { return _left; }
                set
                {
                    if (value == _left) { return; }

                    // LeftNode のリンクを切っておく
                    if (null != _left)
                    {
                        _left._parent = null;
                    }

                    if (null != value)
                    {

                        if (null != value._parent)
                        {
                            Debug.Fail("Target node still have parent.");
                            throw new Exception();
                        }

                        value._parent = this;

                    }

                    _left = value;
                }
            }

            public Node Right
            {
                get { return _right; }
                set
                {
                    if (value == _right) { return; }

                    // RightNode のリンクを切っておく
                    if (null != _right)
                    {
                        _right._parent = null;
                    }

                    if (null != value)
                    {

                        if (null != value._parent)
                        {
                            Debug.Fail("Target node still have parent.");
                            throw new Exception();
                        }

                        value._parent = this;

                    }

                    _right = value;
                }
            }

            public string Key
            {
                get { return _key; }
            }

            public int Bit
            {
                get { return _bit; }
            }

            public IPatriciaTreeItem Item
            {
                get { return _item; }
                set { _item = value; }
            }

            #endregion

            #region ** メソッド

            public Node GetNode(bool bitValue)
            {
                return (bitValue) ? Right : Left;
            }

            #endregion

            #region ** INode の実装

            INode INode.Left { get { return Left; } }
            INode INode.Right { get { return Right; } }

            #endregion
        }

        protected class Leaf : Node, ILeaf
        {
            public Leaf(int index, IPatriciaTreeItem item) : base(index, item.Key, BitHelper.GetBitCount(item.Key), item) { }
        }

        protected class BranchNode : Node, IBranchNode
        {
            public BranchNode(int index, string key, int bit) : base(index, key, bit) { }
        }

        #endregion

        #region ** コレクション

        public interface INodeCollection : IEnumerable<INode>
        {
            #region ** プロパティ

            /// <summary>
            /// コレクションに含まれるアイテム数を取得します。
            /// </summary>
            int Count { get; }

            #endregion

            #region ** メソッド

            /// <summary>
            /// コレクションに指定アイテムが格納されているかどうかを判断します。
            /// </summary>
            /// <param name="item">アイテム</param>
            /// <returns>item が存在する場合は true。それ以外の場合は false。</returns>
            bool Contains(INode item);

            /// <summary>
            /// 指定アイテムのインデックスを取得します。
            /// </summary>
            /// <param name="item">対象アイテムのキー</param>
            /// <returns>インデックス</returns>
            int IndexOf(INode item);

            #endregion

            #region ** インデクサ

            /// <summary>
            /// 指定インデックスのアイテムを取得します。
            /// </summary>
            /// <param name="index">インデックス</param>
            /// <returns>アイテム</returns>
            INode this[int index] { get; }

            #endregion
        }

        public interface IItemCollection : IEnumerable<IPatriciaTreeItem>
        {
            #region ** プロパティ

            /// <summary>
            /// コレクションに含まれるアイテム数を取得します。
            /// </summary>
            int Count { get; }

            #endregion

            #region ** メソッド

            /// <summary>
            /// コレクションに指定アイテムが格納されているかどうかを判断します。
            /// </summary>
            /// <param name="item">アイテム</param>
            /// <returns>item が存在する場合は true。それ以外の場合は false。</returns>
            bool Contains(IPatriciaTreeItem item);

            /// <summary>
            /// コレクションに指定キーが格納されているかどうかを判断します。
            /// </summary>
            /// <param name="key">キー</param>
            /// <returns>key が存在する場合は true。それ以外の場合は false。</returns>
            bool Contains(string key);

            /// <summary>
            /// 指定アイテムのインデックスを取得します。
            /// </summary>
            /// <param name="item">対象アイテムのキー</param>
            /// <returns>インデックス</returns>
            int IndexOf(IPatriciaTreeItem item);

            /// <summary>
            /// 指定キーを持つアイテムのインデックスを取得します。
            /// </summary>
            /// <param name="item">対象アイテムのキー</param>
            /// <returns>インデックス</returns>
            int IndexOf(string key);

            #endregion

            #region ** インデクサ

            /// <summary>
            /// 指定インデックスのアイテムを取得します。
            /// </summary>
            /// <param name="index">インデックス</param>
            /// <returns>アイテム</returns>
            IPatriciaTreeItem this[int index] { get; }

            /// <summary>
            /// 指定キーを含むアイテムを取得します。
            /// </summary>
            /// <param name="key">キー</param>
            /// <returns>アイテム</returns>
            IPatriciaTreeItem this[string key] { get; }

            #endregion
        }

        private class NodeCollection : Collection<INode>, INodeCollection { }

        private class ItemCollection : KeyedCollection<string, IPatriciaTreeItem>, IItemCollection
        {
            #region ** メソッド

            public int IndexOf(string key)
            {
                return IndexOf(this[key]);
            }

            protected override string GetKeyForItem(IPatriciaTreeItem item)
            {
                return item.Key;
            }

            #endregion
        }

        #endregion

        #region ** ヘルパー

        private class BitHelper
        {
            public static bool GetBit(char ch, int bit)
            {
                if (bit < 0 || 7 < bit)
                {
                    Debug.Fail("bit < 0 || 7 < bit");
                    throw new ArgumentOutOfRangeException("bit");
                }
                return (0 != ((ch >> (7 - bit)) & 1));
            }

            public static bool GetBit(string str, int bit)
            {
                if (null == str)
                {
                    Debug.Fail("str is null.");
                    throw new ArgumentNullException("str");
                }
                if (0 > bit)
                {
                    Debug.Fail("bit < 0");
                    throw new ArgumentOutOfRangeException("bit");
                }

                int index = bit / 8;
                char charactor = (index < str.Length) ? str[index] : '\0';

                return GetBit(charactor, bit - (index * 8));
            }

            public static int GetBitCount(string str)
            {
                if (null == str)
                {
                    Debug.Fail("str is null.");
                    throw new ArgumentNullException("str");
                }

                return str.Length * 8;
            }

            public static int CompareBit(string str1, string str2, int start)
            {
                return CompareBit(str1, str2, start, -1);
            }

            public static int CompareBit(string str1, string str2, int start, int end)
            {
                if (null == str1)
                {
                    Debug.Fail("str is null.");
                    throw new ArgumentNullException("str");
                }
                if (null == str2)
                {
                    Debug.Fail("str is null.");
                    throw new ArgumentNullException("str");
                }
                if (0 > start)
                {
                    Debug.Fail("start bit < 0");
                    throw new ArgumentOutOfRangeException("bit");
                }

                int endBit = end;

                if (0 > end)
                {
                    endBit = BitHelper.GetBitCount((str1.Length < str2.Length) ? str2 : str1);
                }

                Debug.Assert(start < endBit);

                for (int current = start; current < endBit; current++)
                {

                    bool bit1 = GetBit(str1, current);
                    bool bit2 = GetBit(str2, current);

                    if (bit1 ^ bit2) { return current; }

                }

                return -1;
            }

            public static string SubString(string str, int bit)
            {
                if (null == str)
                {
                    Debug.Fail("str is null.");
                    throw new ArgumentNullException("str");
                }
                if (0 > bit)
                {
                    Debug.Fail("bit < 0");
                    throw new ArgumentOutOfRangeException("bit");
                }

                int length = bit / 8;
                int error = bit - (length * 8);

                if (0 < error)
                {
                    length++;
                }

                return str.Substring(0, length);
            }
        }

        #endregion
    }
}
