﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    #region ** ストリームサウンドファイル

    internal class Nw4rStreamSoundFile : Nw4rFileRoot
    {
        public Nw4rStreamSoundFile() : base("StreamSoundFileRoot") { }

        #region ** メソッド

        public static Nw4rStreamSoundFile FromFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            Nw4rStreamSoundFile root = new Nw4rStreamSoundFile();
            root.LoadFile(fileName);

            return root;
        }

        protected void LoadFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            using (FileStream stream = File.OpenRead(fileName))
            {
                Read(BinaryReaderBigEndian.CreateInstance(stream));
            }
        }

        protected override void ReadBinaryBlocks(System.IO.BinaryReader reader, Nw4rFileHeader header)
        {
            if (null == header) { throw new ArgumentNullException("header"); }
            if (this != header.Parent) { throw new ArgumentException("header"); }

            // とりあえず HEAD のロードのみ対応
            foreach (Nw4rFileHeader.BlockEntry entry in header.BlockEntries)
            {

                reader.BaseStream.Position = entry.Location.Head;

                Nw4rFileDataBlockHeader blockHeader = new Nw4rFileDataBlockHeader();
                Nw4rFileDataBlock newBlock = null;

                blockHeader.Read(reader);

                switch (blockHeader.Kind)
                {
                    case "HEAD":
                        newBlock = new Nw4rStreamHeadBlock(blockHeader);
                        break;

                    default:
                        continue;
                }

                newBlock.ReadBody(reader);

                entry.Block = newBlock;
                AddNode(newBlock);

            }
        }

        #endregion
    }

    #region ** 各ブロック

    internal class Nw4rStreamHeadBlock : Nw4rFileDataBlockT<Nw4rStreamHeadBlockBody>
    {
        public static readonly string NodeName = "HeadBlock";

        public Nw4rStreamHeadBlock(Nw4rFileDataBlockHeader header) : base(NodeName, header) { }
    }

    internal class Nw4rStreamHeadBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rFileDataOffsetRef _streamInfoRef = new Nw4rFileDataOffsetRef();
        private Nw4rFileDataOffsetRef _trackTableRef = new Nw4rFileDataOffsetRef();
        private Nw4rFileDataOffsetRef _channelTableRef = new Nw4rFileDataOffsetRef();

        private Nw4rStreamInformation _streamInfo = new Nw4rStreamInformation();
        private Nw4rStreamTrackTable _trackTable = new Nw4rStreamTrackTable();
        private Nw4rStreamChannelTable _channelTable = new Nw4rStreamChannelTable();

        public Nw4rStreamHeadBlockBody() { }

        #region ** プロパティ

        public Nw4rFileInnerTable TrackTable
        {
            get { return _trackTable; }
        }

        public Nw4rFileInnerTable ChannelTable
        {
            get { return _channelTable; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _streamInfoRef.Read(reader);
            _trackTableRef.Read(reader);
            _channelTableRef.Read(reader);

            // ストリーム情報
            _streamInfo.Read(reader);

            // トラックテーブル
            _trackTable.Read(reader);

            foreach (Nw4rFileInnerOffsetRefTableItem item in _trackTable.Items)
            {

                reader.BaseStream.Position = Offset + item.Reference.Size;

                Nw4rStreamTrackInfoEx trackInfo = new Nw4rStreamTrackInfoEx();
                trackInfo.Read(reader);

                AddNode(trackInfo);

                item.Data = trackInfo;

            }

            // チャンネルテーブル
            _channelTable.Read(reader);

            foreach (Nw4rFileInnerOffsetRefTableItem item in _channelTable.Items)
            {

                reader.BaseStream.Position = Offset + item.Reference.Size;

                Nw4rStreamChannel channelInfo = new Nw4rStreamChannel(_streamInfo);
                channelInfo.Read(reader);

                AddNode(channelInfo);

                item.Data = channelInfo;

            }

            new Nw4rByteAligner(reader.BaseStream).SeekForward(32);
        }

        #endregion
    }

    internal class Nw4rStreamInformation : Nw4rFileNode
    {
        #region ** 固定値

        public enum FormatType
        {
            PCM8 = 0,
            PCM16 = 1,
            ADPCM = 2,
        }

        #endregion

        #region ** フィールド

        private FormatType _format = 0;		// 1byte : フォーマット
        private bool _loopFlag = false;	// 1byte : ループフラグ
        private int _channels = 0;		// 1byte : チャンネル数
        private int _samplesPerSec = 0;		// 3byte : サンプリングレート
        //													// 2byte : 互換性維持のための領域
        private long _loopBegin = 0;		// 4byte : ループ開始位置
        private long _loopEnd = 0;		// 4byte : ループ終了位置
        private long _dataOffset = 0;		// 4byte : 波形データへのオフセット
        private int _blockCount = 0;		// 4byte : ブロック数
        private long _blockSize = 0;		// 4byte : ブロックサイズ
        private int _samplesPerBlock = 0;		// 4byte : １ブロックに含まれるサンプル数
        private long _lastBlockSize = 0;		// 4byte : 最終ブロックサイズ
        private int _samplesPerLastBlock = 0;		// 4byte : 最終ブロックに含まれるサンプル数
        private long _paddingSize = 0;		// 4byte : パディングサイズ
        private int _adpcmDataInterval = 0;		// 4byte :
        private long _adpcmDataSize = 0;		// 4byte :

        #endregion

        public Nw4rStreamInformation() : base("StreamInfo") { }

        #region ** プロパティ

        public FormatType Format
        {
            get { return _format; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _format = (FormatType)reader.ReadByte();
            _loopFlag = (reader.ReadByte() == 1);
            _channels = reader.ReadByte();
            _samplesPerSec = (int)reader.ReadByte() << 16 | reader.ReadUInt16();

            reader.BaseStream.Seek(sizeof(UInt16), SeekOrigin.Current);

            _loopBegin = reader.ReadUInt32();
            _loopEnd = reader.ReadUInt32();
            _dataOffset = reader.ReadUInt32();
            _blockCount = reader.ReadInt32();
            _blockSize = reader.ReadUInt32();
            _samplesPerBlock = reader.ReadInt32();
            _lastBlockSize = reader.ReadUInt32();
            _samplesPerLastBlock = reader.ReadInt32();
            _paddingSize = reader.ReadUInt32();
            _adpcmDataInterval = reader.ReadInt32();
            _adpcmDataSize = reader.ReadUInt32();
        }

        #endregion
    }

    #region ** トラックテーブル

    internal class Nw4rStreamTrackTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** 固定値

        public enum DataType
        {
            TrackInfo = 0,
            TrackInfoEx = 1,
        }

        #endregion

        private DataType _dataType = 0;

        #region ** プロパティ

        public DataType ItemDataType
        {
            get { return _dataType; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadHeader(BinaryReader reader, out int itemCount)
        {
            itemCount = reader.ReadByte();				// アイテム数
            _dataType = (DataType)reader.ReadByte();	// データの種類

            reader.BaseStream.Seek(sizeof(Byte) * 2, SeekOrigin.Current);
        }

        #endregion
    }

    internal class Nw4rStreamTrackInfoEx : Nw4rFileNode
    {
        #region ** フィールド

        private int _volume = 0;
        private int _pan = 0;

        #endregion

        public Nw4rStreamTrackInfoEx() : base("StreamTrackInfoEx") { }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _volume = reader.ReadByte();
            _pan = reader.ReadByte();

            reader.BaseStream.Seek(sizeof(Byte) * 2, SeekOrigin.Current);	// padding
            reader.BaseStream.Seek(sizeof(UInt32), SeekOrigin.Current);		// Reserved

            int channelCount = reader.ReadByte();	// チャンネル数

            for (int i = 0; i < channelCount; i++)
            {

                Nw4rStreamChannelIndex newChannel = new Nw4rStreamChannelIndex();
                newChannel.Read(reader);

                AddNode(newChannel);

            }

            new Nw4rByteAligner(reader.BaseStream).SeekForward(4);
        }

        #endregion
    }

    internal class Nw4rStreamChannelIndex : Nw4rFileNode
    {
        #region ** フィールド

        private int _channelIndex = 0;

        #endregion

        public Nw4rStreamChannelIndex() : base("StreamTrackInfo") { }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _channelIndex = reader.ReadByte();
        }

        #endregion
    }

    #endregion

    #region ** チャンネルテーブル

    internal class Nw4rStreamChannelTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnReadHeader(BinaryReader reader, out int itemCount)
        {
            itemCount = reader.ReadByte();	// アイテム数

            reader.BaseStream.Seek(sizeof(Byte) * 3, SeekOrigin.Current);
        }

        #endregion
    }

    internal class Nw4rStreamChannel : Nw4rFileNode
    {
        #region ** フィールド

        private Nw4rStreamInformation _streamInfo = null;

        private Nw4rFileDataOffsetRef _adpcmInfoRef = new Nw4rFileDataOffsetRef();
        private AdpcmInfo _adpcmInfo = new AdpcmInfo();

        #endregion

        public Nw4rStreamChannel(Nw4rStreamInformation streamInfo) : base("StreamTrackInfoEx")
        {
            if (null == streamInfo) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("streamInfo")); }

            _streamInfo = streamInfo;
        }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            _adpcmInfoRef.Read(reader);

            if (Nw4rStreamInformation.FormatType.ADPCM == _streamInfo.Format)
            {
                _adpcmInfo.Read(reader);
            }
        }

        #endregion

        private class AdpcmInfo
        {
            #region ** フィールド

            private uint[] _coef = new uint[16];
            private int _gain;
            private int _pred_scale;
            private int _yn1;
            private int _yn2;
            private int _loop_pred_scale;
            private int _loop_yn1;
            private int _loop_yn2;

            #endregion

            #region ** メソッド

            public void Read(BinaryReader reader)
            {
                Debug.Assert(null != reader);

                for (int i = 0; i < _coef.Length; i++)
                {
                    _coef[i] = reader.ReadUInt16();
                }

                _gain = reader.ReadUInt16();
                _pred_scale = reader.ReadUInt16();
                _yn1 = reader.ReadUInt16();
                _yn2 = reader.ReadUInt16();
                _loop_pred_scale = reader.ReadUInt16();
                _loop_yn1 = reader.ReadUInt16();
                _loop_yn2 = reader.ReadUInt16();

                reader.BaseStream.Seek(sizeof(UInt16), SeekOrigin.Current);
            }

            #endregion
        }
    }

    #endregion

    #endregion

    #endregion
}
