﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rSoundSetItemList : IEnumerable
    {
        ArrayList itemList = new ArrayList();

        /******************************************************************************
           interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return itemList.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        public int Count
        {
            get { return itemList.Count; }
        }

        public void CheckDuplicateLabel()
        {
            Hashtable hash = new Hashtable();
            foreach (Nw4rSoundSetItem item in itemList)
            {
                if (hash.ContainsKey(item.Label))
                {
                    throw new Nw4rFileFormatException(
                        "Multiple defined label \"" + item.Label + "\"");
                }
                hash.Add(item.Label, null);
            }
        }

        public bool Contains(string label)
        {
            foreach (Nw4rSoundSetItem item in itemList)
            {
                if (label == item.Label) return true;
            }
            return false;
        }
        public void Clear()
        {
            itemList.Clear();
        }

        /******************************************************************************
           protected internal
         ******************************************************************************/
        protected internal void Add(Nw4rSoundSetItem item)
        {
            itemList.Add(item);
        }
        protected internal void Remove(Nw4rSoundSetItem item)
        {
            itemList.Remove(item);
        }
    }

    public class Nw4rSoundSetSeqList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetSeq seq)
        {
            base.Add(seq);
        }
        public void Remove(Nw4rSoundSetSeq seq)
        {
            base.Remove(seq);
        }
    }

    public class Nw4rSoundSetSeqSoundSetList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetSeqSoundSet seqSoundSet)
        {
            base.Add(seqSoundSet);
        }
        public void Remove(Nw4rSoundSetSeqSoundSet seqSoundSet)
        {
            base.Remove(seqSoundSet);
        }
    }

    public class Nw4rSoundSetWaveSoundSetList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetWaveSoundSet wsd)
        {
            base.Add(wsd);
        }
        public void Remove(Nw4rSoundSetWaveSoundSet wsd)
        {
            base.Remove(wsd);
        }
    }

    public class Nw4rSoundSetBankList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetBank bank)
        {
            base.Add(bank);
        }
        public void Remove(Nw4rSoundSetBank bank)
        {
            base.Remove(bank);
        }
    }

    public class Nw4rSoundSetStrmList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetStrm strm)
        {
            base.Add(strm);
        }
        public void Remove(Nw4rSoundSetStrm strm)
        {
            base.Remove(strm);
        }
    }

    public class Nw4rSoundSetPlayerList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetPlayer player)
        {
            base.Add(player);
        }
        public void Remove(Nw4rSoundSetPlayer player)
        {
            base.Remove(player);
        }
    }

    public class Nw4rSoundSetGroupList : Nw4rSoundSetItemList
    {
        public void Add(Nw4rSoundSetGroup group)
        {
            base.Add(group);
        }
        public void Remove(Nw4rSoundSetGroup group)
        {
            base.Remove(group);
        }
    }

    public abstract class Nw4rSoundSetItem
    {
        string label;
        bool enabled = true;
        int colorIndex = 0;
        string comment;

        public string Label
        {
            set { label = value; }
            get { return label; }
        }
        public bool Enabled
        {
            set { enabled = value; }
            get { return enabled; }
        }
        public int ColorIndex
        {
            set { colorIndex = value; }
            get { return colorIndex; }
        }
        public string Comment
        {
            set { comment = value; }
            get { return comment; }
        }

        public void ParseCommonElement(XmlDocument doc, XmlElement commonElem)
        {
            XmlElement elem;

            label = commonElem.GetAttribute("name");

            if (commonElem.HasAttribute("enabled"))
            {
                enabled = bool.Parse(commonElem.GetAttribute("enabled"));
            }

            if (commonElem.HasAttribute("colorIndex"))
            {
                colorIndex = int.Parse(commonElem.GetAttribute("colorIndex"));
            }

            elem = (XmlElement)commonElem.SelectSingleNode("comment");
            if (elem != null)
            {
                comment = elem.InnerText;
            }
        }
        public void AppendCommonElement(XmlDocument doc, XmlElement commonElem)
        {
            XmlElement elem;

            commonElem.SetAttribute("name", this.label);

            if (!this.enabled)
            {
                commonElem.SetAttribute("enabled", this.enabled.ToString());
            }

            if (0 != this.colorIndex)
            {
                commonElem.SetAttribute("colorIndex", this.colorIndex.ToString());
            }

            if (this.comment != null && this.comment != String.Empty)
            {
                elem = doc.CreateElement("comment");
                elem.InnerText = this.comment;
                commonElem.AppendChild(elem);
            }
        }
    }

    public class Nw4rSoundSetSoundItem : Nw4rSoundSetItem
    {
        Nw4rSound3DParam sound3DParam = new Nw4rSound3DParam();
        int volume = 127;
        int playerPriority = 64;
        int remoteFilter = 0;
        ulong userParam = 0;
        string playerLabel;
        int actorPlayer = 0;

        public Nw4rSound3DParam Sound3DParam
        {
            get { return sound3DParam; }
            set { sound3DParam = value; }
        }
        public int Volume
        {
            set { volume = value; }
            get { return volume; }
        }
        public int PlayerPrio
        {
            set { playerPriority = value; }
            get { return playerPriority; }
        }
        public int RemoteFilter
        {
            set { remoteFilter = value; }
            get { return remoteFilter; }
        }
        public ulong UserParam
        {
            set { userParam = value; }
            get { return userParam; }
        }
        public string Player
        {
            set { playerLabel = value; }
            get { return playerLabel; }
        }
        public int ActorPlayer
        {
            set { actorPlayer = value; }
            get { return actorPlayer; }
        }

        public void ParseSoundElement(XmlDocument doc, XmlElement seqSoundElem)
        {
            XmlElement elem;

            elem = (XmlElement)seqSoundElem.SelectSingleNode("volume");
            if (elem != null)
            {
                volume = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("user_param");
            if (elem != null)
            {
                userParam = UInt32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("player_prio");
            if (elem != null)
            {
                playerPriority = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("remote_filter");
            if (elem != null)
            {
                remoteFilter = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("player");
            if (elem != null)
            {
                playerLabel = elem.InnerText;
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("actor_player");
            if (elem != null)
            {
                actorPlayer = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("sound3d_param");
            if (elem != null)
            {
                sound3DParam = Nw4rSound3DParam.Parse(doc, elem);
            }

            ParseCommonElement(doc, seqSoundElem);
        }

        public void AppendSoundElement(XmlDocument doc, XmlElement seqElem)
        {
            XmlElement elem;

            {
                elem = doc.CreateElement("volume");
                elem.InnerText = this.volume.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("user_param");
                elem.InnerText = this.userParam.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("player_prio");
                elem.InnerText = this.playerPriority.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("remote_filter");
                elem.InnerText = this.remoteFilter.ToString();
                seqElem.AppendChild(elem);
            }
            if (this.Player != null)
            {
                elem = doc.CreateElement("player");
                elem.InnerText = this.playerLabel;
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("actor_player");
                elem.InnerText = this.actorPlayer.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = Sound3DParam.ToXmlElement(doc);
                seqElem.AppendChild(elem);
            }

            AppendCommonElement(doc, seqElem);
        }
    }

    public struct Nw4rSoundSetBankFileType
    {
        public const int Invalid = 0;
        public const int Text = 1;
        public const int Binary = 2;
        public const int Xml = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Text, "Text" ),
            new KeyValuePair<int, string>( Binary, "Binary" ),
            new KeyValuePair<int, string>( Xml, "Xml" ),
        };

        public static implicit operator int(Nw4rSoundSetBankFileType s) { return s.val_; }
        public static implicit operator Nw4rSoundSetBankFileType(int s) { return new Nw4rSoundSetBankFileType(s); }
        public static Nw4rSoundSetBankFileType Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rSoundSetBankFileType(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rSoundSetSeqFileType
    {
        public const int Invalid = 0;
        public const int Text = 1;
        public const int Binary = 2;
        public const int Smf = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Text, "Text" ),
            new KeyValuePair<int, string>( Binary, "Binary" ),
            new KeyValuePair<int, string>( Smf, "Smf" ),
        };

        public static implicit operator int(Nw4rSoundSetSeqFileType s) { return s.val_; }
        public static implicit operator Nw4rSoundSetSeqFileType(int s) { return new Nw4rSoundSetSeqFileType(s); }
        public static Nw4rSoundSetSeqFileType Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rSoundSetSeqFileType(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rSoundSetStrmFileFormat
    {
        public const int Invalid = 0;
        public const int Adpcm = 1;
        public const int Pcm16 = 2;
        public const int Pcm8 = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Adpcm, "Adpcm" ),
            new KeyValuePair<int, string>( Pcm16, "Pcm16" ),
            new KeyValuePair<int, string>( Pcm8, "Pcm8" ),
        };

        public static implicit operator int(Nw4rSoundSetStrmFileFormat s) { return s.val_; }
        public static implicit operator Nw4rSoundSetStrmFileFormat(int s) { return new Nw4rSoundSetStrmFileFormat(s); }
        public static Nw4rSoundSetStrmFileFormat Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rSoundSetStrmFileFormat(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rSoundSetWaveSoundFileFormat
    {
        public const int Invalid = 0;
        public const int Adpcm = 1;
        public const int Pcm16 = 2;
        public const int Pcm8 = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Adpcm, "Adpcm" ),
            new KeyValuePair<int, string>( Pcm16, "Pcm16" ),
            new KeyValuePair<int, string>( Pcm8, "Pcm8" ),
        };

        public static implicit operator int(Nw4rSoundSetWaveSoundFileFormat s) { return s.val_; }
        public static implicit operator Nw4rSoundSetWaveSoundFileFormat(int s) { return new Nw4rSoundSetWaveSoundFileFormat(s); }
        public static Nw4rSoundSetWaveSoundFileFormat Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rSoundSetWaveSoundFileFormat(int e) { val_ = e; }
        private int val_;
    }

    public class Nw4rSoundSetSeq : Nw4rSoundSetSoundItem
    {
        Nw4rSoundSetSeqFileType fileType;
        string filePath;
        string fullPath;
        string bankLabel;
        int channelPriority = 64;
        bool releasePriorityFixFlag = false;
        string startPosition;

        public Nw4rSoundSetSeqFileType FileType
        {
            set { fileType = value; }
            get { return fileType; }
        }
        public string FilePath
        {
            get { return filePath; }
        }
        public void SetFilePath(string filePath, string soundSetFilePath)
        {
            this.filePath = filePath;
            this.fullPath = Path.Combine(Path.GetDirectoryName(soundSetFilePath), filePath);
        }
        public string FullPath
        {
            get { return fullPath; }
        }
        public string Bank
        {
            set { bankLabel = value; }
            get { return bankLabel; }
        }
        public string StartPosition
        {
            set { startPosition = value; }
            get { return startPosition; }
        }
        public int ChannelPrio
        {
            set { channelPriority = value; }
            get { return channelPriority; }
        }
        public bool ReleasePrioFixFlag
        {
            set { releasePriorityFixFlag = value; }
            get { return releasePriorityFixFlag; }
        }

        internal static Nw4rSoundSetSeq Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement seqSoundElem, string soundSetFilePath)
        {
            XmlElement elem; // テンポラリ用

            Nw4rSoundSetSeq seq = new Nw4rSoundSetSeq();

            XmlElement bankArrayElem = (XmlElement)seqSoundElem.SelectSingleNode("bank_array");
            if (bankArrayElem != null)
            {
                elem = (XmlElement)bankArrayElem.SelectSingleNode("bank");
                if (elem != null)
                {
                    if (elem.HasAttribute("name"))
                    {
                        seq.Bank = elem.GetAttribute("name");
                    }
                }
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("file");
            if (elem != null)
            {
                if (elem.HasAttribute("path"))
                {
                    seq.SetFilePath(elem.GetAttribute("path"), soundSetFilePath);
                    seq.FileType = Nw4rSoundSetSeqFileType.Parse(elem.GetAttribute("input"));
                }
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("start_position");
            if (elem != null)
            {
                seq.StartPosition = elem.InnerText;
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("channel_prio");
            if (elem != null)
            {
                seq.ChannelPrio = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("channel_release_prio");
            if (elem != null)
            {
                if (elem.HasAttribute("use_channel_prio"))
                {
                    seq.ReleasePrioFixFlag = Boolean.Parse(elem.GetAttribute("use_channel_prio"));
                }
            }

            seq.ParseSoundElement(doc, seqSoundElem);

            return seq;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement seqElem = doc.CreateElement("seq_sound");
            if (this.bankLabel != null)
            {
                XmlElement bankArrayElem = doc.CreateElement("bank_array");
                elem = doc.CreateElement("bank");
                elem.SetAttribute("name", this.bankLabel);
                elem.SetAttribute("index", "0");
                bankArrayElem.AppendChild(elem);
                seqElem.AppendChild(bankArrayElem);
            }
            if (this.FilePath != null)
            {
                elem = doc.CreateElement("file");
                elem.SetAttribute("path", this.FilePath);
                elem.SetAttribute("input", this.FileType.ToString());
                seqElem.AppendChild(elem);
            }
            if (this.StartPosition != null && this.StartPosition != String.Empty)
            {
                elem = doc.CreateElement("start_position");
                elem.InnerText = this.startPosition;
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_prio");
                elem.InnerText = this.channelPriority.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_release_prio");
                elem.SetAttribute("use_channel_prio", this.releasePriorityFixFlag.ToString());
                seqElem.AppendChild(elem);
            }

            AppendSoundElement(doc, seqElem);

            return seqElem;
        }
    }

    public class Nw4rSoundSetSeqSoundSet : Nw4rSoundSetItem, IEnumerable
    {
        ArrayList itemList = new ArrayList();
        //string filePath;
        //string fullPath;

        /******************************************************************************
           interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return itemList.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        //public string FilePath
        //{
        //    get { return filePath; }
        //}
        //public void SetFilePath( string filePath, string soundSetFilePath )
        //{
        //    this.filePath = filePath;
        //    this.fullPath = Path.Combine( Path.GetDirectoryName( soundSetFilePath ), filePath );
        //}
        //public string FullPath
        //{
        //    get { return fullPath; }
        //}
        public int Count
        {
            get { return itemList.Count; }
        }

        public void Clear()
        {
            itemList.Clear();
        }

        public void Add(Nw4rSoundSetSeq seqSound)
        {
            itemList.Add(seqSound);
        }
        public void Remove(Nw4rSoundSetSeq seqSound)
        {
            itemList.Remove(seqSound);
        }

        internal static Nw4rSoundSetSeqSoundSet Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement wsdElem, string soundSetFilePath)
        {
            Nw4rSoundSetSeqSoundSet wsd = new Nw4rSoundSetSeqSoundSet();

            XmlElement seqSoundArrayElem = (XmlElement)wsdElem.SelectSingleNode("seq_sound_array");
            if (seqSoundArrayElem != null)
            {
                foreach (XmlElement seqSoundElem in seqSoundArrayElem.SelectNodes("seq_sound"))
                {
                    Nw4rSoundSetSeq seqSound = Nw4rSoundSetSeq.Parse(soundSet, doc, seqSoundElem, soundSetFilePath);
                    if (seqSound != null)
                    {
                        wsd.itemList.Add(seqSound);
                    }
                }
            }

            wsd.ParseCommonElement(doc, wsdElem);

            return wsd;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement wsdElem = doc.CreateElement("seq_sound_set");

            if (itemList.Count > 0)
            {
                XmlElement seqSoundArrayElem = doc.CreateElement("seq_sound_array");
                foreach (Nw4rSoundSetSeq seqSound in itemList)
                {
                    XmlElement seqSoundElem = seqSound.ToXmlElement(doc);
                    seqSoundArrayElem.AppendChild(seqSoundElem);
                }
                wsdElem.AppendChild(seqSoundArrayElem);
            }

            AppendCommonElement(doc, wsdElem);

            return wsdElem;
        }
    }

    public class Nw4rSoundSetWaveSoundSet : Nw4rSoundSetItem, IEnumerable
    {
        ArrayList itemList = new ArrayList();
        //string filePath;
        //string fullPath;

        /******************************************************************************
           interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return itemList.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        //public string FilePath
        //{
        //    get { return filePath; }
        //}
        //public void SetFilePath( string filePath, string soundSetFilePath )
        //{
        //    this.filePath = filePath;
        //    this.fullPath = Path.Combine( Path.GetDirectoryName( soundSetFilePath ), filePath );
        //}
        //public string FullPath
        //{
        //    get { return fullPath; }
        //}
        public int Count
        {
            get { return itemList.Count; }
        }

        public void Clear()
        {
            itemList.Clear();
        }

        public void Add(Nw4rSoundSetWaveSound waveSound)
        {
            itemList.Add(waveSound);
        }
        public void Remove(Nw4rSoundSetWaveSound waveSound)
        {
            itemList.Remove(waveSound);
        }

        internal static Nw4rSoundSetWaveSoundSet Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement wsdElem, string soundSetFilePath)
        {
            Nw4rSoundSetWaveSoundSet wsd = new Nw4rSoundSetWaveSoundSet();

            XmlElement waveSoundArrayElem = (XmlElement)wsdElem.SelectSingleNode("wave_sound_array");
            if (waveSoundArrayElem != null)
            {
                foreach (XmlElement waveSoundElem in waveSoundArrayElem.SelectNodes("wave_sound"))
                {
                    Nw4rSoundSetWaveSound waveSound = Nw4rSoundSetWaveSound.Parse(soundSet, wsd, doc, waveSoundElem, soundSetFilePath);
                    if (waveSound != null)
                    {
                        wsd.itemList.Add(waveSound);
                    }
                }
            }

            wsd.ParseCommonElement(doc, wsdElem);

            return wsd;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement wsdElem = doc.CreateElement("wave_sound_set");

            if (itemList.Count > 0)
            {
                XmlElement waveSoundArrayElem = doc.CreateElement("wave_sound_array");
                foreach (Nw4rSoundSetWaveSound waveSound in itemList)
                {
                    XmlElement waveSoundElem = waveSound.ToXmlElement(doc);
                    waveSoundArrayElem.AppendChild(waveSoundElem);
                }
                wsdElem.AppendChild(waveSoundArrayElem);
            }

            AppendCommonElement(doc, wsdElem);

            return wsdElem;
        }
    }

    public class Nw4rSoundSetWaveSound : Nw4rSoundSetSoundItem
    {
        Nw4rSoundSetWaveSoundFileFormat fileFormat;
        string filePath;
        string fullPath;
        int indexOnWaveSoundSet;
        Nw4rAdsrEnvelope envelope = new Nw4rAdsrEnvelope();
        int pan = 64;
        int surroundPan = 0; // 0で最前方
        double pitch = 1.0;
        double fxSendA = 0.0;
        double fxSendB = 0.0;
        double fxSendC = 0.0;
        double mainSend = 1.0;
        int channelPriority = 64;
        bool releasePriorityFixFlag = false;
        Nw4rXmlPanMode panMode = new Nw4rXmlPanMode();
        Nw4rXmlPanCurve panCurve = new Nw4rXmlPanCurve();

        public Nw4rSoundSetWaveSound() { }
        public Nw4rSoundSetWaveSound(Nw4rSoundSetWaveSoundSet wsdSet)
        {
            this.indexOnWaveSoundSet = wsdSet.Count;
        }

        public Nw4rSoundSetWaveSoundFileFormat FileFormat
        {
            set { fileFormat = value; }
            get { return fileFormat; }
        }
        public string FilePath
        {
            get { return filePath; }
        }
        public void SetFilePath(string filePath, string soundSetFilePath)
        {
            this.filePath = filePath;
            this.fullPath = Path.Combine(Path.GetDirectoryName(soundSetFilePath), filePath);
        }
        public string FullPath
        {
            get { return fullPath; }
        }
        public int IndexOnWaveSoundSet
        {
            get { return indexOnWaveSoundSet; }
        }
        public Nw4rAdsrEnvelope AdsrEnvelope
        {
            get { return envelope; }
            set { envelope = value; }
        }
        public int Pan
        {
            set { pan = value; }
            get { return pan; }
        }
        public int SurroundPan
        {
            set { surroundPan = value; }
            get { return surroundPan; }
        }
        public double Pitch
        {
            set { pitch = value; }
            get { return pitch; }
        }
        public int FxSendA
        {
            set { fxSendA = (double)value / 127.0; }
            get { return (int)(fxSendA * 127.0); }
        }
        public int FxSendB
        {
            set { fxSendB = (double)value / 127.0; }
            get { return (int)(fxSendB * 127.0); }
        }
        public int FxSendC
        {
            set { fxSendC = (double)value / 127.0; }
            get { return (int)(fxSendC * 127.0); }
        }
        public int MainSend
        {
            set { mainSend = (double)value / 127.0; }
            get { return (int)(mainSend * 127.0); }
        }
        public int ChannelPrio
        {
            set { channelPriority = value; }
            get { return channelPriority; }
        }
        public bool ReleasePrioFixFlag
        {
            set { releasePriorityFixFlag = value; }
            get { return releasePriorityFixFlag; }
        }
        public Nw4rXmlPanMode PanMode
        {
            set { panMode = value; }
            get { return panMode; }
        }
        public Nw4rXmlPanCurve PanCurve
        {
            set { panCurve = value; }
            get { return panCurve; }
        }

        internal static Nw4rSoundSetWaveSound Parse(Nw4rSoundSet soundSet, Nw4rSoundSetWaveSoundSet wsdSet,
                                                     XmlDocument doc, XmlElement wsdSoundElem, string soundSetFilePath)
        {
            XmlElement elem; // テンポラリ用

            Nw4rSoundSetWaveSound wsd = new Nw4rSoundSetWaveSound(wsdSet);

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("file");
            if (elem != null)
            {
                if (elem.HasAttribute("path"))
                {
                    wsd.SetFilePath(elem.GetAttribute("path"), soundSetFilePath);
                    wsd.FileFormat = Nw4rSoundSetWaveSoundFileFormat.Parse(elem.GetAttribute("encode"));
                }
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("adsr_envelope");
            if (elem != null)
            {
                wsd.AdsrEnvelope = Nw4rAdsrEnvelope.Parse(doc, elem);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("pan");
            if (elem != null)
            {
                wsd.Pan = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("surround_pan");
            if (elem != null)
            {
                wsd.SurroundPan = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("pitch");
            if (elem != null)
            {
                wsd.Pitch = Double.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("fx_send_a");
            if (elem != null)
            {
                wsd.fxSendA = Double.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("fx_send_b");
            if (elem != null)
            {
                wsd.fxSendB = Double.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("fx_send_c");
            if (elem != null)
            {
                wsd.fxSendC = Double.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("main_send");
            if (elem != null)
            {
                wsd.mainSend = Double.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("channel_prio");
            if (elem != null)
            {
                wsd.ChannelPrio = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("channel_release_prio");
            if (elem != null)
            {
                if (elem.HasAttribute("use_channel_prio"))
                {
                    wsd.ReleasePrioFixFlag = Boolean.Parse(elem.GetAttribute("use_channel_prio"));
                }
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("pan_mode");
            if (elem != null)
            {
                wsd.PanMode = Nw4rXmlPanMode.Parse(elem.InnerText);
            }

            elem = (XmlElement)wsdSoundElem.SelectSingleNode("pan_curve");
            if (elem != null)
            {
                wsd.PanCurve = Nw4rXmlPanCurve.Parse(elem.InnerText);
            }

            wsd.ParseSoundElement(doc, wsdSoundElem);

            return wsd;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement wsdElem = doc.CreateElement("wave_sound");

            if (this.FilePath != null)
            {
                elem = doc.CreateElement("file");
                elem.SetAttribute("path", this.FilePath);
                elem.SetAttribute("encode", this.FileFormat.ToString());
                wsdElem.AppendChild(elem);
            }
            if (envelope != null)
            {
                XmlElement adsrEnvElem = AdsrEnvelope.ToXmlElement(doc);
                wsdElem.AppendChild(adsrEnvElem);
            }
            {
                elem = doc.CreateElement("pan");
                elem.InnerText = this.pan.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("surround_pan");
                elem.InnerText = this.surroundPan.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pitch");
                elem.InnerText = this.pitch.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("fx_send_a");
                elem.InnerText = this.fxSendA.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("fx_send_b");
                elem.InnerText = this.fxSendB.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("fx_send_c");
                elem.InnerText = this.fxSendC.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("main_send");
                elem.InnerText = this.mainSend.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_prio");
                elem.InnerText = this.channelPriority.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_release_prio");
                elem.SetAttribute("use_channel_prio", this.releasePriorityFixFlag.ToString());
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_mode");
                elem.InnerText = this.PanMode.ToString();
                wsdElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_curve");
                elem.InnerText = this.PanCurve.ToString();
                wsdElem.AppendChild(elem);
            }

            AppendSoundElement(doc, wsdElem);

            return wsdElem;
        }

        internal XmlElement ToXmlElementForWsd(XmlDocument doc, string waveSoundSetFilePath)
        {
            XmlElement elem; // テンポラリ用

            XmlElement wsdElem = doc.CreateElement("wave_sound");

            if (this.FilePath != null)
            {
                XmlElement fileArrayElem = doc.CreateElement("file_array");

                if (envelope != null)
                {
                    XmlElement adsrEnvElem = AdsrEnvelope.ToXmlElement(doc);
                    wsdElem.AppendChild(adsrEnvElem);
                }
                {
                    elem = doc.CreateElement("pan");
                    elem.InnerText = this.pan.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("surround_pan");
                    elem.InnerText = this.surroundPan.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("pitch");
                    elem.InnerText = this.pitch.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("fx_send_a");
                    elem.InnerText = this.fxSendA.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("fx_send_b");
                    elem.InnerText = this.fxSendB.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("fx_send_c");
                    elem.InnerText = this.fxSendC.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("main_send");
                    elem.InnerText = this.mainSend.ToString();
                    wsdElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("file");
                    elem.SetAttribute(
                        "path",
                        NPath.MakeRelative(
                            this.FullPath,
                            Path.GetDirectoryName(waveSoundSetFilePath)
                        ));
                    elem.SetAttribute("encode", this.FileFormat.ToString());
                    fileArrayElem.AppendChild(elem);
                    wsdElem.AppendChild(fileArrayElem);
                }
            }

            return wsdElem;
        }
    }

    public class Nw4rSoundSetBank : Nw4rSoundSetItem
    {
        Nw4rSoundSetBankFileType fileType;
        string filePath;
        string fullPath;

        public Nw4rSoundSetBankFileType FileType
        {
            set { fileType = value; }
            get { return fileType; }
        }
        public string FilePath
        {
            get { return filePath; }
        }
        public void SetFilePath(string filePath, string soundSetFilePath)
        {
            this.filePath = filePath;
            this.fullPath = Path.Combine(Path.GetDirectoryName(soundSetFilePath), filePath);
        }
        public string FullPath
        {
            get { return fullPath; }
        }

        internal static Nw4rSoundSetBank Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement bankElem, string soundSetFilePath)
        {
            XmlElement elem; // テンポラリ用

            Nw4rSoundSetBank bank = new Nw4rSoundSetBank();

            elem = (XmlElement)bankElem.SelectSingleNode("file");
            if (elem != null)
            {
                if (elem.HasAttribute("path"))
                {
                    bank.SetFilePath(elem.GetAttribute("path"), soundSetFilePath);

                    if (elem.HasAttribute("input"))
                    {
                        bank.FileType = Nw4rSoundSetBankFileType.Parse(elem.GetAttribute("input"));
                    }
                }
            }

            bank.ParseCommonElement(doc, bankElem);

            return bank;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement bankElem = doc.CreateElement("bank");
            if (this.FilePath != null)
            {
                elem = doc.CreateElement("file");
                elem.SetAttribute("path", this.FilePath);
                if (this.FileType != Nw4rSoundSetBankFileType.Invalid)
                {
                    elem.SetAttribute("input", this.FileType.ToString());
                }
                bankElem.AppendChild(elem);
            }

            AppendCommonElement(doc, bankElem);

            return bankElem;
        }
    }

    public class Nw4rSoundSetStrm : Nw4rSoundSetSoundItem
    {
        public class Track
        {
            int index = -1;
            string filePath;
            string fullPath;
            Nw4rSoundSetStrmFileFormat format;
            int volume = 127;
            int pan = 64;

            public Nw4rSoundSetStrmFileFormat FileFormat
            {
                set { format = value; }
                get { return format; }
            }
            public string FilePath
            {
                get { return filePath; }
            }
            public void SetFilePath(string filePath, string soundSetFilePath)
            {
                this.filePath = filePath;
                this.fullPath = Path.Combine(Path.GetDirectoryName(soundSetFilePath), filePath);
            }
            public string FullPath
            {
                get { return fullPath; }
            }
            public int Index
            {
                set { index = value; }
                get { return index; }
            }
            public int Volume
            {
                set { volume = value; }
                get { return volume; }
            }
            public int Pan
            {
                set { pan = value; }
                get { return pan; }
            }
        }

        List<Track> trackList = new List<Track>();
        Nw4rXmlPanMode panMode = new Nw4rXmlPanMode();
        Nw4rXmlPanCurve panCurve = new Nw4rXmlPanCurve();

        public List<Track> TrackList
        {
            get { return trackList; }
        }
        public Nw4rXmlPanMode PanMode
        {
            set { panMode = value; }
            get { return panMode; }
        }
        public Nw4rXmlPanCurve PanCurve
        {
            set { panCurve = value; }
            get { return panCurve; }
        }
        internal static Nw4rSoundSetStrm Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement strmSoundElem, string soundSetFilePath)
        {
            XmlElement elem; // テンポラリ用

            Nw4rSoundSetStrm strm = new Nw4rSoundSetStrm();

            XmlElement trackArrayElem = (XmlElement)strmSoundElem.SelectSingleNode("strm_track_array");
            if (trackArrayElem != null)
            {
                foreach (XmlElement strmTrackElem in trackArrayElem.SelectNodes("strm_track"))
                {
                    Nw4rSoundSetStrm.Track track = new Nw4rSoundSetStrm.Track();
                    if (strmTrackElem.HasAttribute("index"))
                    {
                        track.Index = Int32.Parse(strmTrackElem.GetAttribute("index"));
                    }

                    elem = (XmlElement)strmTrackElem.SelectSingleNode("file");
                    if (elem != null)
                    {
                        if (elem.HasAttribute("path"))
                        {
                            track.SetFilePath(elem.GetAttribute("path"), soundSetFilePath);
                            track.FileFormat = Nw4rSoundSetStrmFileFormat.Parse(elem.GetAttribute("encode"));
                        }
                    }
                    elem = (XmlElement)strmTrackElem.SelectSingleNode("volume");
                    if (elem != null)
                    {
                        track.Volume = Int32.Parse(elem.InnerText);
                    }
                    elem = (XmlElement)strmTrackElem.SelectSingleNode("pan");
                    if (elem != null)
                    {
                        track.Pan = Int32.Parse(elem.InnerText);
                    }
                    strm.TrackList.Add(track);
                }
            }
            else
            {
                elem = (XmlElement)strmSoundElem.SelectSingleNode("file");
                if (elem != null)
                {
                    if (elem.HasAttribute("path"))
                    {
                        Nw4rSoundSetStrm.Track track = new Nw4rSoundSetStrm.Track();
                        track.SetFilePath(elem.GetAttribute("path"), soundSetFilePath);
                        track.FileFormat = Nw4rSoundSetStrmFileFormat.Parse(elem.GetAttribute("encode"));
                        strm.TrackList.Add(track);
                    }
                }
            }

            elem = (XmlElement)strmSoundElem.SelectSingleNode("pan_mode");
            if (elem != null)
            {
                strm.PanMode = Nw4rXmlPanMode.Parse(elem.InnerText);
            }

            elem = (XmlElement)strmSoundElem.SelectSingleNode("pan_curve");
            if (elem != null)
            {
                strm.PanCurve = Nw4rXmlPanCurve.Parse(elem.InnerText);
            }

            strm.ParseSoundElement(doc, strmSoundElem);

            return strm;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement strmElem = doc.CreateElement("strm_sound");

            XmlElement trackArrayElem = doc.CreateElement("strm_track_array");
            foreach (Nw4rSoundSetStrm.Track track in trackList)
            {
                XmlElement trackElem = doc.CreateElement("strm_track");
                if (track.Index >= 0)
                {
                    trackElem.SetAttribute("index", track.Index.ToString());
                }

                if (track.FilePath != null)
                {
                    elem = doc.CreateElement("file");
                    elem.SetAttribute("path", track.FilePath);
                    elem.SetAttribute("encode", track.FileFormat.ToString());
                    trackElem.AppendChild(elem);

                    elem = doc.CreateElement("volume");
                    elem.InnerText = track.Volume.ToString();
                    trackElem.AppendChild(elem);

                    elem = doc.CreateElement("pan");
                    elem.InnerText = track.Pan.ToString();
                    trackElem.AppendChild(elem);
                }

                trackArrayElem.AppendChild(trackElem);
            }
            strmElem.AppendChild(trackArrayElem);

            {
                elem = doc.CreateElement("pan_mode");
                elem.InnerText = this.PanMode.ToString();
                strmElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_curve");
                elem.InnerText = this.PanCurve.ToString();
                strmElem.AppendChild(elem);
            }

            AppendSoundElement(doc, strmElem);

            return strmElem;
        }
    }

    public class Nw4rSoundSetPlayer : Nw4rSoundSetItem
    {
        int soundMax = 1;
        int heapSize = 0;

        public int SoundLimit
        {
            set { soundMax = value; }
            get { return soundMax; }
        }
        public int HeapSize
        {
            set { heapSize = value; }
            get { return heapSize; }
        }
        internal static Nw4rSoundSetPlayer Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement playerElem)
        {
            XmlElement elem; // テンポラリ用

            Nw4rSoundSetPlayer player = new Nw4rSoundSetPlayer();

            elem = (XmlElement)playerElem.SelectSingleNode("sound_limit");
            if (elem != null)
            {
                player.SoundLimit = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)playerElem.SelectSingleNode("heap_size");
            if (elem != null)
            {
                player.HeapSize = Int32.Parse(elem.InnerText);
            }

            player.ParseCommonElement(doc, playerElem);

            return player;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement playerElem = doc.CreateElement("player");

            {
                elem = doc.CreateElement("sound_limit");
                elem.InnerText = this.SoundLimit.ToString();
                playerElem.AppendChild(elem);
            }
            if (this.heapSize > 0)
            {
                elem = doc.CreateElement("heap_size");
                elem.InnerText = this.HeapSize.ToString();
                playerElem.AppendChild(elem);
            }

            AppendCommonElement(doc, playerElem);

            return playerElem;
        }
    }

    public class Nw4rSoundSetGroup : Nw4rSoundSetItem, IEnumerable
    {
        ArrayList items = new ArrayList();
        bool _outputFlag = true;
        bool _waveArchiveShare = false;

        public bool OutputFlag
        {
            set { _outputFlag = value; }
            get { return _outputFlag; }
        }

        public bool WaveArchiveShare
        {
            get { return _waveArchiveShare; }
            set { _waveArchiveShare = value; }
        }

        public int Count
        {
            get { return items.Count; }
        }
        public int Add(Nw4rSoundSetGroupItem item)
        {
            return items.Add(item);
        }
        public void Remove(Nw4rSoundSetGroupItem item)
        {
            items.Remove(item);
        }
        public void Clear()
        {
            items.Clear();
        }
        IEnumerator IEnumerable.GetEnumerator()
        {
            return items.GetEnumerator();
        }

        internal static Nw4rSoundSetGroup Parse(Nw4rSoundSet soundSet, XmlDocument doc, XmlElement groupElem)
        {
            Nw4rSoundSetGroup group = new Nw4rSoundSetGroup();

            if (groupElem.HasAttribute("output_flag"))
            {
                group.OutputFlag = bool.Parse(groupElem.GetAttribute("output_flag"));
            }
            else
            {
                group.OutputFlag = true;
            }

            if (groupElem.HasAttribute("wave_archive_share"))
            {
                group.WaveArchiveShare = bool.Parse(groupElem.GetAttribute("wave_archive_share"));
            }
            else
            {
                group.WaveArchiveShare = false;
            }

            XmlElement groupItemArrayElem = (XmlElement)groupElem.SelectSingleNode("group_item_array");
            if (groupItemArrayElem != null)
            {
                foreach (XmlElement groupItemElem in groupItemArrayElem.SelectNodes("group_item"))
                {
                    if (groupItemElem.HasAttribute("name"))
                    {
                        string label = groupItemElem.GetAttribute("name");
                        Nw4rSoundSetGroupItem groupItem = new Nw4rSoundSetGroupItem(label);
                        group.Add(groupItem);
                    }
                }
            }

            group.ParseCommonElement(doc, groupElem);

            return group;
        }

        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement groupElem = doc.CreateElement("group");

            groupElem.SetAttribute("output_flag", _outputFlag.ToString());

            groupElem.SetAttribute("wave_archive_share", _waveArchiveShare.ToString());

            if (items.Count > 0)
            {
                XmlElement groupItemArrayElem = doc.CreateElement("group_item_array");
                foreach (Nw4rSoundSetGroupItem groupItem in items)
                {
                    elem = doc.CreateElement("group_item");
                    elem.SetAttribute("name", groupItem.Label);
                    groupItemArrayElem.AppendChild(elem);
                }
                groupElem.AppendChild(groupItemArrayElem);
            }

            AppendCommonElement(doc, groupElem);

            return groupElem;
        }
    }

    public class Nw4rSoundSetGroupItem
    {
        string label;

        public Nw4rSoundSetGroupItem(string label)
        {
            this.label = label;
        }
        public string Label
        {
            set { label = value; }
            get { return label; }
        }
    }

    public class Nw4rSoundSet : Nw4rXmlFile
    {
        string name = string.Empty;
        Nw4rSoundSetBankList bankList = new Nw4rSoundSetBankList();
        Nw4rSoundSetSeqList seqList = new Nw4rSoundSetSeqList();
        Nw4rSoundSetStrmList strmList = new Nw4rSoundSetStrmList();
        Nw4rSoundSetSeqSoundSetList seqSetList = new Nw4rSoundSetSeqSoundSetList();
        Nw4rSoundSetWaveSoundSetList wsdList = new Nw4rSoundSetWaveSoundSetList();
        Nw4rSoundSetPlayerList playerList = new Nw4rSoundSetPlayerList();
        Nw4rSoundSetGroupList groupList = new Nw4rSoundSetGroupList();

        public Nw4rSoundSet() { }
        public Nw4rSoundSet(string name)
        {
            this.name = name;
        }

        public string Name
        {
            get { return name; }
        }
        public Nw4rSoundSetSeqList SeqList
        {
            get { return seqList; }
        }
        public Nw4rSoundSetStrmList StrmList
        {
            get { return strmList; }
        }
        public Nw4rSoundSetSeqSoundSetList SeqSoundSetList
        {
            get { return seqSetList; }
        }
        public Nw4rSoundSetWaveSoundSetList WaveSoundSetList
        {
            get { return wsdList; }
        }
        public Nw4rSoundSetBankList BankList
        {
            get { return bankList; }
        }
        public Nw4rSoundSetPlayerList PlayerList
        {
            get { return playerList; }
        }
        public Nw4rSoundSetGroupList GroupList
        {
            get { return groupList; }
        }

        public void Clear()
        {
            bankList.Clear();
            seqList.Clear();
            strmList.Clear();
            wsdList.Clear();
            playerList.Clear();
            groupList.Clear();
        }

        protected override string FileTitle { get { return "NintendoWare Sound Set"; } }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            Clear();

            XmlElement soundSetElem = (XmlElement)bodyElem.SelectSingleNode("sound_set");
            if (soundSetElem == null)
            {
                throw new Nw4rFileFormatException("Invalid SoundSet file format.");
            }

            XmlElement seqSoundArrayElem = (XmlElement)soundSetElem.SelectSingleNode("seq_sound_array");
            if (seqSoundArrayElem != null)
            {
                foreach (XmlElement seqSoundElem in seqSoundArrayElem.SelectNodes("seq_sound"))
                {
                    Nw4rSoundSetSeq seq = Nw4rSoundSetSeq.Parse(this, doc, seqSoundElem, this.FilePath);
                    if (seq != null)
                    {
                        seqList.Add(seq);
                    }
                }
            }

            XmlElement strmSoundArrayElem = (XmlElement)soundSetElem.SelectSingleNode("strm_sound_array");
            if (strmSoundArrayElem != null)
            {
                foreach (XmlElement strmSoundElem in strmSoundArrayElem.SelectNodes("strm_sound"))
                {
                    Nw4rSoundSetStrm strm = Nw4rSoundSetStrm.Parse(this, doc, strmSoundElem, this.FilePath);
                    if (strm != null)
                    {
                        strmList.Add(strm);
                    }
                }
            }

            XmlElement seqSoundSetArrayElem = (XmlElement)soundSetElem.SelectSingleNode("seq_sound_set_array");
            if (seqSoundSetArrayElem != null)
            {
                foreach (XmlElement seqSoundSetElem in seqSoundSetArrayElem.SelectNodes("seq_sound_set"))
                {
                    Nw4rSoundSetSeqSoundSet seqSoundSet = Nw4rSoundSetSeqSoundSet.Parse(this, doc, seqSoundSetElem, this.FilePath);
                    if (seqSoundSet != null)
                    {
                        seqSetList.Add(seqSoundSet);
                    }
                }
            }

            XmlElement waveSoundSetArrayElem = (XmlElement)soundSetElem.SelectSingleNode("wave_sound_set_array");
            if (waveSoundSetArrayElem != null)
            {
                foreach (XmlElement waveSoundSetElem in waveSoundSetArrayElem.SelectNodes("wave_sound_set"))
                {
                    Nw4rSoundSetWaveSoundSet wsd = Nw4rSoundSetWaveSoundSet.Parse(this, doc, waveSoundSetElem, this.FilePath);
                    if (wsd != null)
                    {
                        wsdList.Add(wsd);
                    }
                }
            }

            XmlElement bankArrayElem = (XmlElement)soundSetElem.SelectSingleNode("bank_array");
            if (bankArrayElem != null)
            {
                foreach (XmlElement bankElem in bankArrayElem.SelectNodes("bank"))
                {
                    Nw4rSoundSetBank bank = Nw4rSoundSetBank.Parse(this, doc, bankElem, this.FilePath);
                    if (bank != null)
                    {
                        bankList.Add(bank);
                    }
                }
            }

            XmlElement playerArrayElem = (XmlElement)soundSetElem.SelectSingleNode("player_array");
            if (playerArrayElem != null)
            {
                foreach (XmlElement playerElem in playerArrayElem.SelectNodes("player"))
                {
                    Nw4rSoundSetPlayer player = Nw4rSoundSetPlayer.Parse(this, doc, playerElem);
                    if (player != null)
                    {
                        playerList.Add(player);
                    }
                }
            }

            XmlElement groupArrayElem = (XmlElement)soundSetElem.SelectSingleNode("group_array");
            if (groupArrayElem != null)
            {
                foreach (XmlElement groupElem in groupArrayElem.SelectNodes("group"))
                {
                    Nw4rSoundSetGroup group = Nw4rSoundSetGroup.Parse(this, doc, groupElem);
                    if (group != null)
                    {
                        groupList.Add(group);
                    }
                }
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement soundSetElem = doc.CreateElement("sound_set");
            bodyElem.AppendChild(soundSetElem);

            if (seqList.Count > 0)
            {
                XmlElement seqSoundArrayElem = doc.CreateElement("seq_sound_array");
                foreach (Nw4rSoundSetSeq seq in seqList)
                {
                    XmlElement seqElem = seq.ToXmlElement(doc);
                    seqSoundArrayElem.AppendChild(seqElem);
                }
                soundSetElem.AppendChild(seqSoundArrayElem);
            }
            if (strmList.Count > 0)
            {
                XmlElement strmSoundArrayElem = doc.CreateElement("strm_sound_array");
                foreach (Nw4rSoundSetStrm strm in strmList)
                {
                    XmlElement strmElem = strm.ToXmlElement(doc);
                    strmSoundArrayElem.AppendChild(strmElem);
                }
                soundSetElem.AppendChild(strmSoundArrayElem);
            }
            if (seqSetList.Count > 0)
            {
                XmlElement seqSoundSetArrayElem = doc.CreateElement("seq_sound_set_array");
                foreach (Nw4rSoundSetSeqSoundSet seqSoundSet in seqSetList)
                {
                    XmlElement seqSoundSetElem = seqSoundSet.ToXmlElement(doc);
                    seqSoundSetArrayElem.AppendChild(seqSoundSetElem);
                }
                soundSetElem.AppendChild(seqSoundSetArrayElem);
            }
            if (wsdList.Count > 0)
            {
                XmlElement waveSoundSetArrayElem = doc.CreateElement("wave_sound_set_array");
                foreach (Nw4rSoundSetWaveSoundSet wsd in wsdList)
                {
                    XmlElement wsdElem = wsd.ToXmlElement(doc);
                    waveSoundSetArrayElem.AppendChild(wsdElem);
                }
                soundSetElem.AppendChild(waveSoundSetArrayElem);
            }
            if (bankList.Count > 0)
            {
                XmlElement bankArrayElem = doc.CreateElement("bank_array");
                foreach (Nw4rSoundSetBank bank in bankList)
                {
                    XmlElement bankElem = bank.ToXmlElement(doc);
                    bankArrayElem.AppendChild(bankElem);
                }
                soundSetElem.AppendChild(bankArrayElem);
            }
            if (playerList.Count > 0)
            {
                XmlElement playerArrayElem = doc.CreateElement("player_array");
                foreach (Nw4rSoundSetPlayer player in playerList)
                {
                    XmlElement playerElem = player.ToXmlElement(doc);
                    playerArrayElem.AppendChild(playerElem);
                }
                soundSetElem.AppendChild(playerArrayElem);
            }
            if (groupList.Count > 0)
            {
                XmlElement groupArrayElem = doc.CreateElement("group_array");
                foreach (Nw4rSoundSetGroup group in groupList)
                {
                    XmlElement groupElem = group.ToXmlElement(doc);
                    groupArrayElem.AppendChild(groupElem);
                }
                soundSetElem.AppendChild(groupArrayElem);
            }
        }


    }

}
