﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.Text;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rSoundList
    {
        public class OutputSetting
        {
            public string SettingName;
            public string OutputFilePath;
            public string Title;
            public bool CombineSoundSet = true;

            public TListOutputInfo ListOutputInfo;
            public TSoundListColumnInfo SoundListColumnInfo;
            public TGroupListColumnInfo GroupListColumnInfo;
            public TPlayerListColumnInfo PlayerListColumnInfo;
            public TBankListColumnInfo BankListColumnInfo;

            public struct TListInfo
            {
                public bool OutputFlag;
                public int Order;

                public TListInfo(bool outputFlag, int order)
                {
                    OutputFlag = outputFlag;
                    Order = order;
                }
            }

            public struct TColumnInfo
            {
                public bool OutputFlag;
                public int Order;

                public TColumnInfo(bool outoutFlag, int order)
                {
                    OutputFlag = outoutFlag;
                    Order = order;
                }
            }

            public struct TListOutputInfo
            {
                public TListInfo SoundList;
                public TListInfo GroupList;
                public TListInfo PlayerList;
                public TListInfo BankList;
            }

            public struct TSoundListColumnInfo
            {
                public TColumnInfo Index;
                public TColumnInfo Name;
                public TColumnInfo Type;
                public TColumnInfo Volume;
                public TColumnInfo Player;
                public TColumnInfo ActorPlayer;
                public TColumnInfo PlayerPrio;
                public TColumnInfo Comment;
            }

            public struct TGroupListColumnInfo
            {
                public TColumnInfo Index;
                public TColumnInfo Name;
                public TColumnInfo Comment;
            }

            public struct TPlayerListColumnInfo
            {
                public TColumnInfo Index;
                public TColumnInfo Name;
                public TColumnInfo SoundLimit;
                public TColumnInfo Comment;
            }

            public struct TBankListColumnInfo
            {
                public TColumnInfo Index;
                public TColumnInfo Name;
                public TColumnInfo Comment;
            }

            public OutputSetting(string settingName, string outputFilePath)
            {
                SettingName = settingName;
                Title = Path.GetFileNameWithoutExtension(outputFilePath) + " Project Sound List - " + settingName;
                OutputFilePath = outputFilePath;

                int order = 0;
                ListOutputInfo.SoundList = new TListInfo(true, order++);
                ListOutputInfo.GroupList = new TListInfo(true, order++);
                ListOutputInfo.PlayerList = new TListInfo(true, order++);
                ListOutputInfo.BankList = new TListInfo(false, order++);

                order = 0;
                SoundListColumnInfo.Index = new TColumnInfo(true, order++);
                SoundListColumnInfo.Name = new TColumnInfo(true, order++);
                SoundListColumnInfo.Type = new TColumnInfo(true, order++);
                SoundListColumnInfo.Player = new TColumnInfo(true, order++);
                SoundListColumnInfo.PlayerPrio = new TColumnInfo(true, order++);
                SoundListColumnInfo.Comment = new TColumnInfo(true, order++);
                SoundListColumnInfo.ActorPlayer = new TColumnInfo(false, order++);
                SoundListColumnInfo.Volume = new TColumnInfo(false, order++);

                order = 0;
                GroupListColumnInfo.Index = new TColumnInfo(true, order++);
                GroupListColumnInfo.Name = new TColumnInfo(true, order++);
                GroupListColumnInfo.Comment = new TColumnInfo(true, order++);

                order = 0;
                PlayerListColumnInfo.Index = new TColumnInfo(true, order++);
                PlayerListColumnInfo.Name = new TColumnInfo(true, order++);
                PlayerListColumnInfo.SoundLimit = new TColumnInfo(true, order++);
                PlayerListColumnInfo.Comment = new TColumnInfo(true, order++);

                order = 0;
                BankListColumnInfo.Index = new TColumnInfo(true, order++);
                BankListColumnInfo.Name = new TColumnInfo(true, order++);
                BankListColumnInfo.Comment = new TColumnInfo(true, order++);
            }

            public OutputSetting Clone()
            {
                return (OutputSetting)MemberwiseClone();
            }
        }



        class ListItem
        {
            private string _class = string.Empty;
            private List<string> _texts = new List<string>();

            public string Class
            {
                get { return _class; }
                set { _class = value; }
            }

            public List<string> Texts
            {
                get { return _texts; }
            }
        }

        public enum EItemType
        {
            Index,
            Name,
            Type,
            Volume,
            Player,
            ActorPlayer,
            PlayerPrio,
            SoundLimit,
            Comment
        }

        public struct ColumnInfo
        {
            public string Name;
            public string SortType;
            public EItemType ItemType;
            public int Order;
        }

        class ItemTable
        {
            public LinkedList<ListItem> ItemList = new LinkedList<ListItem>();
            public string Title;
            public List<ColumnInfo> Column;
            public int Order;

            public ItemTable(string _title, List<ColumnInfo> _column)
            {
                Title = _title;
                Column = _column;
            }

            public void ImportStrmSound(Nw4rSoundSetStrm sound, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!sound.Enabled) ? "Disabled" : string.Empty;

                if (0 < sound.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + sound.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(sound.Label);
                            break;
                        case EItemType.Type:
                            item.Texts.Add("Stream");
                            break;
                        case EItemType.Volume:
                            item.Texts.Add(sound.Volume.ToString());
                            break;
                        case EItemType.Player:
                            item.Texts.Add(sound.Player);
                            break;
                        case EItemType.ActorPlayer:
                            item.Texts.Add(sound.ActorPlayer.ToString());
                            break;
                        case EItemType.PlayerPrio:
                            item.Texts.Add(sound.PlayerPrio.ToString());
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(sound.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }

            public void ImportSeqSound(Nw4rSoundSetSeq sound, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!sound.Enabled) ? "Disabled" : string.Empty;

                if (0 < sound.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + sound.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(sound.Label);
                            break;
                        case EItemType.Type:
                            item.Texts.Add("Sequence");
                            break;
                        case EItemType.Volume:
                            item.Texts.Add(sound.Volume.ToString());
                            break;
                        case EItemType.Player:
                            item.Texts.Add(sound.Player);
                            break;
                        case EItemType.ActorPlayer:
                            item.Texts.Add(sound.ActorPlayer.ToString());
                            break;
                        case EItemType.PlayerPrio:
                            item.Texts.Add(sound.PlayerPrio.ToString());
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(sound.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }

            public void ImportWaveSound(Nw4rSoundSetWaveSound sound, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!sound.Enabled) ? "Disabled" : string.Empty;

                if (0 < sound.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + sound.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(sound.Label);
                            break;
                        case EItemType.Type:
                            item.Texts.Add("Wave");
                            break;
                        case EItemType.Volume:
                            item.Texts.Add(sound.Volume.ToString());
                            break;
                        case EItemType.Player:
                            item.Texts.Add(sound.Player);
                            break;
                        case EItemType.ActorPlayer:
                            item.Texts.Add(sound.ActorPlayer.ToString());
                            break;
                        case EItemType.PlayerPrio:
                            item.Texts.Add(sound.PlayerPrio.ToString());
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(sound.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }

            public void ImportBank(Nw4rSoundSetBank bank, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!bank.Enabled) ? "Disabled" : string.Empty;

                if (0 < bank.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + bank.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(bank.Label);
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(bank.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }

            public void ImportPlayer(Nw4rSoundSetPlayer player, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!player.Enabled) ? "Disabled" : string.Empty;

                if (0 < player.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + player.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(player.Label);
                            break;
                        case EItemType.SoundLimit:
                            item.Texts.Add(player.SoundLimit.ToString());
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(player.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }

            public void ImportGroup(Nw4rSoundSetGroup group, List<ColumnInfo> outputInfo)
            {
                ListItem item = new ListItem();

                item.Class = (!group.Enabled) ? "Disabled" : string.Empty;

                if (0 < group.ColorIndex)
                {

                    if (0 <= item.Class.Length)
                    {
                        item.Class += " ";
                    }

                    item.Class += "ColorType" + group.ColorIndex.ToString();

                }

                foreach (ColumnInfo info in outputInfo)
                {
                    switch (info.ItemType)
                    {
                        case EItemType.Index:
                            item.Texts.Add(ItemList.Count.ToString());
                            break;
                        case EItemType.Name:
                            item.Texts.Add(group.Label);
                            break;
                        case EItemType.Comment:
                            item.Texts.Add(group.Comment);
                            break;
                    }
                }
                ItemList.AddLast(item);
            }
        }

        class ItemTableSet
        {
            public string Name;

            public ItemTable SoundListTable = null;
            public ItemTable GroupListTable = null;
            public ItemTable PlayerListTable = null;
            public ItemTable BankListTable = null;

            public List<ItemTable> ItemTableList = new List<ItemTable>();
        }

        List<ItemTableSet> itemTableSetList = new List<ItemTableSet>();

        string ProjectFilePath;
        string title;
        Nw4rSoundMakerProject Project = new Nw4rSoundMakerProject();
        int listIndex = -1;
        bool combineSoundSet;

        List<ColumnInfo> soundItemInfo = null;
        List<ColumnInfo> groupItemInfo = null;
        List<ColumnInfo> playerItemInfo = null;
        List<ColumnInfo> bankItemInfo = null;

        public static Nw4rSoundList CreateFromSoundProject(string projectFilePath, OutputSetting setting)
        {
            Nw4rSoundList soundList = new Nw4rSoundList();

            soundList.ProjectFilePath = projectFilePath;
            soundList.title = (setting.Title != String.Empty) ? setting.Title : "Sound Set List";
            soundList.combineSoundSet = setting.CombineSoundSet;

            soundList.SetupOutputItemList(setting);

            ItemTableSet itemTableSet = null;
            if (setting.CombineSoundSet)
            {
                itemTableSet = soundList.CreateItemTableSet(setting);
                soundList.itemTableSetList.Add(itemTableSet);
            }

            soundList.Project.Load(projectFilePath);

            foreach (Nw4rSoundMakerProject.SoundSet s in soundList.Project.SoundSets)
            {
                Nw4rSoundSet soundSet = new Nw4rSoundSet();
                string fullPath = Path.Combine(Path.GetDirectoryName(projectFilePath), s.FilePath);
                soundSet.Load(fullPath);

                if (!setting.CombineSoundSet)
                {
                    itemTableSet = soundList.CreateItemTableSet(setting);
                    itemTableSet.Name = s.Name;
                    soundList.itemTableSetList.Add(itemTableSet);
                }
                soundList.ImportSoundSet(soundSet, itemTableSet);
            }

            return soundList;
        }

        private static int CompareItemTableOrder(ItemTable x, ItemTable y)
        {
            return x.Order.CompareTo(y.Order);
        }

        private static int CompareColumnInfoOrder(ColumnInfo x, ColumnInfo y)
        {
            return x.Order.CompareTo(y.Order);
        }

        void AppendOutputItemInfoList(
            List<ColumnInfo> list,
            string name,
            string sortType,
            EItemType itemType,
            OutputSetting.TColumnInfo itemInfo
        )
        {
            if (itemInfo.OutputFlag)
            {
                ColumnInfo columnInfo = new ColumnInfo();
                columnInfo.Name = name;
                columnInfo.SortType = sortType;
                columnInfo.ItemType = itemType;
                columnInfo.Order = itemInfo.Order;
                list.Add(columnInfo);
            }
        }

        void SetupOutputItemList(OutputSetting setting)
        {
            soundItemInfo = new List<ColumnInfo>();
            AppendOutputItemInfoList(soundItemInfo, "index", "number", EItemType.Index, setting.SoundListColumnInfo.Index);
            AppendOutputItemInfoList(soundItemInfo, "name", "string", EItemType.Name, setting.SoundListColumnInfo.Name);
            AppendOutputItemInfoList(soundItemInfo, "type", "string", EItemType.Type, setting.SoundListColumnInfo.Type);
            AppendOutputItemInfoList(soundItemInfo, "volume", "number", EItemType.Volume, setting.SoundListColumnInfo.Volume);
            AppendOutputItemInfoList(soundItemInfo, "player", "string", EItemType.Player, setting.SoundListColumnInfo.Player);
            AppendOutputItemInfoList(soundItemInfo, "actor player", "number", EItemType.ActorPlayer, setting.SoundListColumnInfo.ActorPlayer);
            AppendOutputItemInfoList(soundItemInfo, "prio", "number", EItemType.PlayerPrio, setting.SoundListColumnInfo.PlayerPrio);
            AppendOutputItemInfoList(soundItemInfo, "comment", "string", EItemType.Comment, setting.SoundListColumnInfo.Comment);
            soundItemInfo.Sort(CompareColumnInfoOrder);

            groupItemInfo = new List<ColumnInfo>();
            AppendOutputItemInfoList(groupItemInfo, "index", "number", EItemType.Index, setting.GroupListColumnInfo.Index);
            AppendOutputItemInfoList(groupItemInfo, "name", "string", EItemType.Name, setting.GroupListColumnInfo.Name);
            AppendOutputItemInfoList(groupItemInfo, "comment", "string", EItemType.Comment, setting.GroupListColumnInfo.Comment);
            groupItemInfo.Sort(CompareColumnInfoOrder);

            playerItemInfo = new List<ColumnInfo>();
            AppendOutputItemInfoList(playerItemInfo, "index", "number", EItemType.Index, setting.PlayerListColumnInfo.Index);
            AppendOutputItemInfoList(playerItemInfo, "name", "string", EItemType.Name, setting.PlayerListColumnInfo.Name);
            AppendOutputItemInfoList(playerItemInfo, "sound limit", "number", EItemType.SoundLimit, setting.PlayerListColumnInfo.SoundLimit);
            AppendOutputItemInfoList(playerItemInfo, "comment", "string", EItemType.Comment, setting.PlayerListColumnInfo.Comment);
            playerItemInfo.Sort(CompareColumnInfoOrder);

            bankItemInfo = new List<ColumnInfo>();
            AppendOutputItemInfoList(bankItemInfo, "index", "number", EItemType.Index, setting.BankListColumnInfo.Index);
            AppendOutputItemInfoList(bankItemInfo, "name", "string", EItemType.Name, setting.BankListColumnInfo.Name);
            AppendOutputItemInfoList(bankItemInfo, "comment", "string", EItemType.Comment, setting.BankListColumnInfo.Comment);
            bankItemInfo.Sort(CompareColumnInfoOrder);
        }

        ItemTableSet CreateItemTableSet(OutputSetting setting)
        {
            ItemTableSet itemTableSet = new ItemTableSet();
            itemTableSet.SoundListTable = new ItemTable("Sound List", soundItemInfo);
            itemTableSet.GroupListTable = new ItemTable("Group List", groupItemInfo);
            itemTableSet.PlayerListTable = new ItemTable("Player List", playerItemInfo);
            itemTableSet.BankListTable = new ItemTable("Bank List", bankItemInfo);

            itemTableSet.SoundListTable.Order = setting.ListOutputInfo.SoundList.Order;
            itemTableSet.GroupListTable.Order = setting.ListOutputInfo.GroupList.Order;
            itemTableSet.PlayerListTable.Order = setting.ListOutputInfo.PlayerList.Order;
            itemTableSet.BankListTable.Order = setting.ListOutputInfo.BankList.Order;

            if (setting.ListOutputInfo.SoundList.OutputFlag) itemTableSet.ItemTableList.Add(itemTableSet.SoundListTable);
            if (setting.ListOutputInfo.GroupList.OutputFlag) itemTableSet.ItemTableList.Add(itemTableSet.GroupListTable);
            if (setting.ListOutputInfo.PlayerList.OutputFlag) itemTableSet.ItemTableList.Add(itemTableSet.PlayerListTable);
            if (setting.ListOutputInfo.BankList.OutputFlag) itemTableSet.ItemTableList.Add(itemTableSet.BankListTable);

            itemTableSet.ItemTableList.Sort(CompareItemTableOrder);

            return itemTableSet;
        }

        void ImportSoundSet(Nw4rSoundSet soundSet, ItemTableSet itemTableSet)
        {
            foreach (Nw4rSoundSetStrm sound in soundSet.StrmList)
            {
                itemTableSet.SoundListTable.ImportStrmSound(sound, soundItemInfo);
            }

            foreach (Nw4rSoundSetWaveSoundSet wsdList in soundSet.WaveSoundSetList)
            {
                foreach (Nw4rSoundSetWaveSound sound in wsdList)
                {
                    itemTableSet.SoundListTable.ImportWaveSound(sound, soundItemInfo);
                }
            }

            foreach (Nw4rSoundSetSeq sound in soundSet.SeqList)
            {
                itemTableSet.SoundListTable.ImportSeqSound(sound, soundItemInfo);
            }

            foreach (Nw4rSoundSetSeqSoundSet seqSoundSet in soundSet.SeqSoundSetList)
            {
                foreach (Nw4rSoundSetSeq sound in seqSoundSet)
                {
                    itemTableSet.SoundListTable.ImportSeqSound(sound, soundItemInfo);
                }
            }

            foreach (Nw4rSoundSetBank bank in soundSet.BankList)
            {
                itemTableSet.BankListTable.ImportBank(bank, bankItemInfo);
            }
            foreach (Nw4rSoundSetPlayer player in soundSet.PlayerList)
            {
                itemTableSet.PlayerListTable.ImportPlayer(player, playerItemInfo);
            }
            foreach (Nw4rSoundSetGroup group in soundSet.GroupList)
            {
                itemTableSet.GroupListTable.ImportGroup(group, groupItemInfo);
            }
        }

        void ExportHtmlBody(XmlTextWriter w)
        {
            // タイトル
            w.WriteStartElement("h1");
            w.WriteString(title);
            w.WriteEndElement();

            // テーブル
            if (combineSoundSet)
            {
                if (itemTableSetList.Count > 0)
                {
                    ItemTableSet itemTableSet = itemTableSetList[0];

                    w.WriteStartElement("div");
                    w.WriteAttributeString("class", "tableset");

                    foreach (ItemTable table in itemTableSet.ItemTableList)
                    {
                        ExportHtmlItemList(w, table);
                    }
                    w.WriteEndElement(); // div
                }
            }
            else
            {
                w.WriteStartElement("div");

                w.WriteStartElement("p");
                w.WriteString("SoundSet : ");
                int index = 0;
                foreach (ItemTableSet itemTableSet in itemTableSetList)
                {
                    w.WriteStartElement("a");
                    w.WriteAttributeString("href", "#");
                    w.WriteAttributeString("onclick", "ShowSoundSet('soundset" + index + "')");
                    w.WriteString(itemTableSet.Name);
                    w.WriteEndElement(); // a

                    w.WriteString(" | ");

                    index++;
                }

                w.WriteStartElement("a");
                w.WriteAttributeString("href", "#");
                w.WriteAttributeString("onclick", "ShowAllSoundSet()");
                w.WriteString("[Show All]");
                w.WriteEndElement(); // a

                w.WriteEndElement(); // p

                w.WriteStartElement("div");
                w.WriteAttributeString("id", "soundsetlist");

                index = 0;
                foreach (ItemTableSet itemTableSet in itemTableSetList)
                {
                    w.WriteStartElement("div");
                    w.WriteAttributeString("id", "soundset" + index);

                    w.WriteStartElement("h2");
                    w.WriteString(itemTableSet.Name);
                    w.WriteEndElement(); // h2

                    w.WriteStartElement("div");
                    w.WriteAttributeString("class", "tableset");

                    foreach (ItemTable table in itemTableSet.ItemTableList)
                    {
                        ExportHtmlItemList(w, table);
                    }
                    w.WriteEndElement(); // div class="tableset"

                    w.WriteEndElement(); // div

                    index++;
                }
                w.WriteEndElement(); // div id="soundsetlist"

                w.WriteEndElement(); // div
            }

            // フッタ
            w.WriteStartElement("hr");
            w.WriteEndElement(); // hr

            w.WriteStartElement("dl");

            w.WriteStartElement("dt");
            w.WriteString("Project:");
            w.WriteEndElement(); // dt
            w.WriteStartElement("dd");
            w.WriteString(ProjectFilePath);
            w.WriteEndElement(); // dd

            w.WriteStartElement("dt");
            w.WriteString("DateTime:");
            w.WriteEndElement(); // dt
            w.WriteStartElement("dd");
            w.WriteString(DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss"));
            w.WriteEndElement(); // dd

            w.WriteEndElement(); // dl
        }

        void ExportHtmlItemList(XmlTextWriter w, ItemTable table)
        {
            if (table.ItemList.Count == 0)
            {
                // 空の表は出力しない
                return;
            }

            listIndex++;
            string displayId = "display" + listIndex.ToString();

            w.WriteStartElement("h3");
            w.WriteStartElement("input");
            w.WriteAttributeString("type", "button");
            w.WriteAttributeString("class", "tableopenclose");
            w.WriteAttributeString("onclick", "SwitchDisplay('" + displayId + "', this)");
            w.WriteAttributeString("title", "close");
            w.WriteAttributeString("style", "font-size:10px; width:16px; height:16px; cursor:pointer");
            w.WriteAttributeString("value", "-");
            w.WriteEndElement(); // input
            w.WriteString(" ");
            w.WriteStartElement("a");
            w.WriteAttributeString("name", listIndex.ToString());
            w.WriteString(table.Title);
            w.WriteEndElement(); // a
            w.WriteEndElement(); // h3

            w.WriteStartElement("div");
            w.WriteAttributeString("id", displayId);

            w.WriteStartElement("table");
            w.WriteAttributeString("width", "100%");
            w.WriteStartElement("thead");

            w.WriteStartElement("tr");

            int index = 0;
            foreach (ColumnInfo itemInfo in table.Column)
            {
                w.WriteStartElement("th");
                if (index == 0) w.WriteAttributeString("class", "SortKey");
                w.WriteAttributeString("onclick", "SortTable('" + itemInfo.SortType + "', this)");
                w.WriteAttributeString("title", "sort");
                w.WriteAttributeString("style", "cursor:pointer;");
                w.WriteString(itemInfo.Name);
                w.WriteEndElement(); // th

                index++;
            }

            w.WriteEndElement(); // tr
            w.WriteEndElement(); // thead

            w.WriteStartElement("tbody");

            foreach (ListItem soundItem in table.ItemList)
            {
                w.WriteStartElement("tr");

                if (0 < soundItem.Class.Length)
                {
                    w.WriteAttributeString("class", soundItem.Class);
                }

                foreach (string text in soundItem.Texts)
                {
                    w.WriteStartElement("td");

                    if (text != null)
                    {
                        w.WriteString(text);
                    }
                    else
                    {
                        w.WriteEntityRef("nbsp");
                    }
                    w.WriteEndElement();
                }

                w.WriteEndElement(); // tr
            }

            w.WriteEndElement(); // tbody
            w.WriteEndElement(); // table

            w.WriteEndElement(); // div
        }

        public void ExportHtml(string filePath)
        {
            using (XmlTextWriter w = new XmlTextWriter(filePath, Encoding.UTF8))
            {
                w.Formatting = Formatting.Indented;

                w.WriteStartDocument();

                w.WriteComment(" saved from url=(0013)about:internet ");

                w.WriteStartElement("html");

                w.WriteStartElement("head");

                w.WriteStartElement("meta");
                w.WriteAttributeString("name", "Author");
                w.WriteAttributeString("content", Project.GeneratorName + " " + Project.GeneratorVersion);
                w.WriteEndElement();

                w.WriteStartElement("title");
                w.WriteString(title);
                w.WriteEndElement();

                w.WriteStartElement("style");
                w.WriteAttributeString("type", "text/css");
                w.WriteComment(@"
body {
    margin: 12px;
}

h1 {
    font-size: 16pt;
    text-indent: 1em;
    border-left: solid #7886B6 20px;
    border-bottom: solid #7886B6 2px;
}

h2 {
    font-size: 14pt;
    text-indent: 1em;
    border-left: solid #7886B6 10px;
    border-bottom: solid #7886B6 1px;
}

h3 {
    font-size: 14pt;
}

div {
    margin-bottom: 20px;
}

div.tableset {
    margin: 0em 1em 1em 1em;
}

table {
    border-collapse: collapse;
    border-top: #a1a5a9 1px solid;
    border-bottom: #a1a5a9 1px solid;
}

th {
    padding-left: 0.5em;
    padding-right: 0.5em;
    text-align: left;
    background: #efefef;
}

th.SortKey {
    background: #e4e4ff;
}

tr.Disabled {
    color: #c0c0c0;
}

tr.ColorType1 {
    background-color: #ffaaaa;
}

tr.ColorType2 {
    background-color: #ffe080;
}

tr.ColorType3 {
    background-color: #ffffaa;
}

tr.ColorType4 {
    background-color: #d0ffc0;
}

tr.ColorType5 {
    background-color: #c0ecff;
}

tr.ColorType6 {
    background-color: #b4ddff;
}

tr.ColorType7 {
    background-color: #d8d8ff;
}

tr.ColorType8 {
    background-color: #ffddff;
}

td {
    padding-left: 0.5em;
    padding-right: 0.5em;
    border-top: #a1a5a9 1px dotted;
}

dl {
    font-size: 9pt;
}
");
                w.WriteEndElement();

                w.WriteStartElement("script");
                w.WriteAttributeString("type", "text/javascript");
                w.WriteComment(@"
var gTrElems;
var gColId = new Array();
var gColIndexTmp;
var gReverse = new Array();

//==================================================================
// type - ""number"" or ""string""
function SortTable( type, obj )
{
    var cellArray = new Array();
    var rowArray = new Array();
    var trClassArray = new Array();
    var tdClassArray = new Array();
    var row;
    var col;

    if ( !document.getElementById ) { return; }

    var tableElem = obj.parentNode.parentNode.parentNode;
    var thElems = obj.parentNode.getElementsByTagName(""th"");

    for ( col = 0; col < thElems.length; col++ )
    {
        if ( thElems[col] == obj ) gColIndexTmp = col;
    }

    // init
    if ( typeof(gColId[tableElem]) == ""undefined"" ) {
        gReverse[tableElem] = 0;
        gColId[tableElem] = thElems[0];
    }

    var reverseFlag = 0;
    if ( obj == gColId[tableElem] ) {
        gReverse[tableElem] = - ( gReverse[tableElem] - 1 );
        if ( ! gReverse[tableElem] ) { reverseFlag = 1; }
    }
    else { gReverse[tableElem] = 0; }

    gTrElems = tableElem.getElementsByTagName(""tbody"")[0].getElementsByTagName(""tr"");

    //  read table
    if ( reverseFlag ) {
        for ( row = 0; row < gTrElems.length; row++ ) { rowArray[row] = gTrElems.length-row-1; }
    }
    else {
        for ( row = 0; row < gTrElems.length; row++ ) { rowArray[row] = row; }
    }
    for ( row = 0; row < gTrElems.length; row++ )
    {
        var tdElems = gTrElems[row].getElementsByTagName(""td"");
        trClassArray[row] = gTrElems[row].className;
        cellArray[row]  = new Array();
        tdClassArray[row] = new Array();
        for ( col = 0; col < tdElems.length; col++ )
        {
            cellArray[row][col]  = tdElems[col].innerHTML;
            tdClassArray[row][col] = tdElems[col].className;
        }
    }

    //  sort
    if ( type == ""number"" ) { rowArray.sort(CompareByNumber); }
    else if ( type == ""string"" ) { rowArray.sort(CompareByString); }

    if ( gReverse[tableElem] ) { rowArray.reverse(); }

    //  write table
    for ( row = 0; row < gTrElems.length; row++ )
    {
        gTrElems[row].className = trClassArray[rowArray[row]];

        var tdElems = gTrElems[row].getElementsByTagName(""td"");
        for (col = 0; col < tdElems.length; col++)
        {
            tdElems[col].innerHTML = cellArray[rowArray[row]][col];
            tdElems[col].className = tdClassArray[rowArray[row]][col];
        }
    }

    gColId[tableElem] = obj;

    //  table header
    for ( col = 0; col < thElems.length; col++ ) { thElems[col].className = """"; }
    obj.className = ""SortKey"";
}

//==================================================================
function CompareByNumber(a, b)
{
    var sA = gTrElems[a].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;
    var sB = gTrElems[b].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;

    sA = Number( sA );
    sB = Number( sB );

    if ( sA == sB ) { return 0; }
    if ( sA > sB ) { return 1; }
    return -1;
}

//==================================================================
function CompareByString(a, b)
{
    var sA = gTrElems[a].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;
    var sB = gTrElems[b].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;

    if ( sA == sB ) { return 0; }

    if ( sA == ""&nbsp;"" ) { return 1; }
    if ( sB == ""&nbsp;"" ) { return -1; }

    if ( sA > sB ) { return 1; }
    return -1;
}

//==================================================================
function ResetAllTableOpenCloseButton()
{
    var inputElems = document.getElementsByTagName(""input"");
    for ( index = 0; index < inputElems.length; index++ )
    {
        if ( inputElems[index].className == ""tableopenclose"" )
        {
            inputElems[index].setAttribute( ""title"", ""close"" );
            inputElems[index].value = ""-"";
        }
    }

}

//==================================================================
function ShowAllSoundSet()
{
    var soundsetListElem = document.getElementById( ""soundsetlist"" );
    var soundsetElems = soundsetListElem.getElementsByTagName(""div"");

    for ( soundsetIndex = 0; soundsetIndex < soundsetElems.length; soundsetIndex++ )
    {
        soundsetElems[soundsetIndex].style.display = ""block"";
    }

    ResetAllTableOpenCloseButton();
}

//==================================================================
function ShowSoundSet( soundsetId )
{
    var soundsetListElem = document.getElementById( ""soundsetlist"" );
    var soundsetElems = soundsetListElem.getElementsByTagName(""div"");

    for ( soundsetIndex = 0; soundsetIndex < soundsetElems.length; soundsetIndex++ )
    {
        soundsetElems[soundsetIndex].style.display = ""none"";
    }

    var targetElem = document.getElementById( soundsetId );
    targetElem.style.display = ""block"";

    var datalistElems = targetElem.getElementsByTagName(""div"");
    for ( index = 0; index < datalistElems.length; index++ )
    {
        datalistElems[index].style.display = ""block"";
    }

    ResetAllTableOpenCloseButton();
}

//==================================================================
function SwitchDisplay( taragetId, obj )
{
    var target = document.getElementById( taragetId );
    if ( target.style.display == ""none"" )
    {
        target.style.display = ""block"";
        obj.setAttribute( ""title"", ""close"" );
        obj.value = ""-"";
    }
    else
    {
        target.style.display = ""none"";
        obj.setAttribute( ""title"", ""open"" );
        obj.value = ""+"";
    }
}
");

                w.WriteEndElement(); // head


                w.WriteStartElement("body");
                ExportHtmlBody(w);
                w.WriteEndElement(); // body

                w.WriteEndElement(); // html
                w.WriteEndDocument();
            }
        }

    }

}
