﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    /// <summary>
    /// サウンドアーカイブ マップファイル クラス
    /// </summary>
    internal class Nw4rSoundArchiveMapFile
    {
        #region ** 固定値

        private const int LabelLength = 25;
        private const int ParameterLength = 3;
        private const int IDLength = 6;
        private const int SizeLength = 8;

        #endregion

        #region ** フィールド

        private Nw4rSoundArchiveFile _archiveFile = null;			// サウンドアーカイブ
        private string _baseDirectoryPath = string.Empty;	// 相対ファイルパス出力時のベースパス

        #endregion

        public Nw4rSoundArchiveMapFile(Nw4rSoundArchiveFile archiveFile)
        {
            if (null == archiveFile) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archiveFile")); }
            _archiveFile = archiveFile;
        }

        #region ** プロパティ

        private Nw4rSoundArchive SoundArchive
        {
            get { return _archiveFile.SoundArchive; }
        }

        private Nw4rFileHeader FileHeader
        {
            get { return _archiveFile.Header; }
        }

        private Nw4rSarcSymbolBlock SymbolBlock
        {
            get { return _archiveFile.ChildNodes[Nw4rSarcSymbolBlock.NodeName] as Nw4rSarcSymbolBlock; }
        }

        private Nw4rSarcInformationBlock InformationBlock
        {
            get
            {
                return _archiveFile.ChildNodes[Nw4rSarcInformationBlock.NodeName] as Nw4rSarcInformationBlock;
            }
        }

        private Nw4rSarcFileDataBlock FileDataBlock
        {
            get
            {
                return _archiveFile.ChildNodes[Nw4rSarcFileDataBlock.NodeName] as Nw4rSarcFileDataBlock;
            }
        }

        #endregion

        #region ** メソッド

        public void Write(TextWriter writer, string baseDirectoryPath)
        {
            if (null == writer) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("writer")); }
            if (null == baseDirectoryPath) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("baseDirectoryPath")); }

            _baseDirectoryPath = baseDirectoryPath;

            WriteSoundItems(writer);
            WriteBankItems(writer);
            WriteGroupItems(writer);
            WriteFileItems(writer);
            WriteFileMap(writer);
        }

        #endregion

        #region ** サウンド情報

        private void WriteSoundItems(TextWriter writer)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != SoundArchive);

            // Tag
            writer.WriteLine("# SOUND:");

            // Header
            writer.Write("# " + CreateText("label", -LabelLength));
            writer.Write(" " + CreateText("sndID", IDLength));
            writer.Write(" type");
            writer.Write(" " + CreateText("fileID", IDLength));
            writer.Write(" " + CreateText("vol", ParameterLength));
            writer.Write(" " + CreateText("cpr", ParameterLength));
            writer.Write(" " + CreateText("ppr", ParameterLength));
            writer.Write(" player");
            writer.Write(writer.NewLine);

            // Items
            foreach (Nw4rSound sound in SoundArchive.AllSounds)
            {
                WriteSoundItem(writer, sound);
            }

            writer.Write(writer.NewLine);
        }

        private void WriteSoundItem(TextWriter writer, Nw4rSound sound)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != sound);

            writer.Write("  " + CreateText(sound.Label, -LabelLength));
            writer.Write(" " + CreateText(SoundArchive.AllSounds.IndexOf(sound.Key), IDLength));
            writer.Write(" " + CreateTypeText(sound));
            writer.Write(" " + CreateText(CreateFileIDText(sound), IDLength));
            writer.Write(" " + CreateText(sound.XmlData.Volume, ParameterLength));

            if (sound is Nw4rWaveSound)
            {
                writer.Write(" " + CreateText((sound as Nw4rWaveSound).XmlData.ChannelPrio, ParameterLength));
            }
            else if (sound is Nw4rSequenceSound)
            {
                writer.Write(" " + CreateText((sound as Nw4rSequenceSound).XmlData.ChannelPrio, ParameterLength));
            }
            else
            {
                writer.Write(" " + CreateText("-", ParameterLength));
            }

            writer.Write(" " + CreateText(sound.XmlData.PlayerPrio, ParameterLength));
            writer.Write(" " + sound.XmlData.Player);
            writer.Write(writer.NewLine);
        }

        private string CreateFileIDText(Nw4rSound sound)
        {
            Debug.Assert(null != sound);

            Nw4rSoundBinaryFile binaryFile = null;

            if (sound is Nw4rWaveSound)
            {
                binaryFile = (sound.Parent as Model.Nw4rSoundSetItem).BinaryFile;
            }
            else
            {
                binaryFile = sound.BinaryFile;
            }

            return SoundArchive.BinaryFiles.IndexOf(binaryFile.Key).ToString();
        }

        private string CreateTypeText(Nw4rSound sound)
        {
            Debug.Assert(null != sound);

            if (sound is Nw4rStreamSound)
            {
                return "STM ";
            }
            else if (sound is Nw4rWaveSound)
            {
                return "WSD ";
            }
            else if (sound is Nw4rSequenceSound)
            {
                return "SEQ ";
            }

            return "-   ";
        }

        #endregion

        #region ** バンク情報

        private void WriteBankItems(TextWriter writer)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != SoundArchive);

            // Tag
            writer.WriteLine("# BANK:");

            // Header
            writer.Write("# " + CreateText("label", -LabelLength));
            writer.Write(" " + CreateText("bankID", IDLength));
            writer.Write(" fileID");
            writer.Write(writer.NewLine);

            // Items
            foreach (Nw4rBank bank in SoundArchive.AllBanks)
            {
                WriteBankItem(writer, bank);
            }

            writer.Write(writer.NewLine);
        }

        private void WriteBankItem(TextWriter writer, Nw4rBank bank)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != bank);

            writer.Write("  " + CreateText(bank.Label, -LabelLength));
            writer.Write(" " + CreateText(SoundArchive.AllBanks.IndexOf(bank.Key), IDLength));
            writer.Write(" " + CreateText(SoundArchive.BinaryFiles.IndexOf(bank.BinaryFile.Key), IDLength));
            writer.Write(writer.NewLine);
        }

        #endregion

        #region ** グループ情報

        private void WriteGroupItems(TextWriter writer)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != SoundArchive);

            // Tag
            writer.WriteLine("# GROUP:");

            // Header
            writer.Write("# " + CreateText("label", -LabelLength));
            writer.Write(" " + CreateText("grpID", IDLength));
            writer.Write(" " + CreateText("msize", SizeLength));
            writer.Write(" " + CreateText("asize", SizeLength));
            writer.Write(" path");
            writer.Write(writer.NewLine);

            // Items
            foreach (Nw4rGroup group in SoundArchive.AllGroups)
            {
                if (0 == group.Label.Length) { continue; }
                WriteGroupItem(writer, group);
            }

            writer.Write(writer.NewLine);
        }

        private void WriteGroupItem(TextWriter writer, Nw4rGroup group)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != group);

            Nw4rSarcGroupImageBlock groupImage = GetGroupImage(group);

            writer.Write("  " + CreateText(group.Label, -LabelLength));
            writer.Write(" " + CreateText(SoundArchive.AllGroups.IndexOf(group.Key), IDLength));
            writer.Write(" " + CreateText(groupImage.MainImages.Size, SizeLength));
            writer.Write(" " + CreateText(groupImage.ARamImages.Size, SizeLength));
            writer.Write(writer.NewLine);

            // グループに含まれるファイル情報
            for (int i = 0; i < groupImage.MainImages.ChildNodes.Count; i++)
            {

                Nw4rSarcGroupItemRamImage itemMainImage = groupImage.MainImages.ChildNodes[i] as Nw4rSarcGroupItemRamImage;
                Nw4rSarcGroupItemRamImage itemARamImage = groupImage.ARamImages.ChildNodes[i] as Nw4rSarcGroupItemRamImage;

                writer.Write("    " + CreateText(itemMainImage.BinaryFile.Label, -(LabelLength - 2)));
                writer.Write(" " + CreateText(string.Empty, IDLength));
                writer.Write(" " + CreateText(itemMainImage.Size, SizeLength));
                writer.Write(" " + CreateText(itemARamImage.Size, SizeLength));

                string filePath = itemMainImage.BinaryFile.FilePath;
                if (null == filePath || 0 == filePath.Length)
                {
                    filePath = itemARamImage.BinaryFile.ARamFilePath;
                }

                writer.Write(" " + ((null == filePath) ? string.Empty : Path.GetFileName(filePath)));
                writer.Write(writer.NewLine);

            }
        }

        private Nw4rSarcGroupImageBlock GetGroupImage(Nw4rGroup group)
        {
            Debug.Assert(null != group);

            int index = SoundArchive.AllGroups.IndexOf(group.Key);

            if (0 > index)
            {
                Debug.Assert(false);
                throw new Nw4rFileFormatInternalException();
            }

            return FileDataBlock.Body.ChildNodes[index] as Nw4rSarcGroupImageBlock;
        }

        #endregion

        #region ** ファイル情報

        private void WriteFileItems(TextWriter writer)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != SoundArchive);

            // Tag
            writer.WriteLine("# FILE:");

            // Header
            writer.Write("# " + CreateText("label", -LabelLength));
            writer.Write(" " + CreateText("fileID", IDLength));
            writer.Write(" " + CreateText("msize", SizeLength));
            writer.Write(" " + CreateText("asize", SizeLength));
            writer.Write(" path");
            writer.Write(writer.NewLine);

            // Items
            foreach (Nw4rSoundBinaryFile file in SoundArchive.BinaryFiles)
            {

                if (0 == file.ARamFilePath.Length
                    && 0 == file.ExtensionFileRelativePath.Length
                    && 0 == file.Groups.Count) { continue; }

                WriteFileItem(writer, file);

            }

            writer.Write(writer.NewLine);
        }

        private void WriteFileItem(TextWriter writer, Nw4rSoundBinaryFile file)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != file);

            writer.Write("  " + CreateText(file.Label, -LabelLength));
            writer.Write(" " + CreateText(SoundArchive.BinaryFiles.IndexOf(file.Key), IDLength));
            writer.Write(" " + CreateText(GetFileSize(file.FilePath), SizeLength));
            writer.Write(" " + CreateText(GetFileSize(file.ARamFilePath), SizeLength));

            string filePath = file.FilePath;
            if (null == filePath || 0 == filePath.Length)
            {
                filePath = file.ARamFilePath;
            }

            writer.Write(" " + Path.GetFileName(filePath));
            writer.Write(writer.NewLine);
        }

        private long GetFileSize(string filePath)
        {
            Debug.Assert(null != filePath);

            if (0 == filePath.Length) { return 0; }
            if (!File.Exists(filePath)) { return 0; }

            return new FileInfo(filePath).Length;
        }

        #endregion

        #region ** ファイルマップ情報

        private void WriteFileMap(TextWriter writer)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != SoundArchive);

            // Tag
            writer.WriteLine("# MAP:");

            // Header
            writer.Write("# " + CreateText("offset", SizeLength));
            writer.Write(" " + CreateText("size", SizeLength));
            writer.Write(" contents");
            writer.Write(writer.NewLine);

            // Items
            WriteFileMapNode(writer, FileHeader, "file header");
            WriteFileMapNode(writer, SymbolBlock, "symbol table");
            WriteFileMapNode(writer, InformationBlock, "sound archive header");
            WriteFileDataBlockItems(writer);

            // End
            WriteFileMapItem(writer, _archiveFile.Size, 0, "END", string.Empty);
        }

        private void WriteFileDataBlockItems(TextWriter writer)
        {
            Debug.Assert(null != writer);

            WriteFileMapNode(writer, FileDataBlock, "sound data");

            // グループ
            foreach (Nw4rGroup group in SoundArchive.AllGroups)
            {

                if (0 == group.Label.Length) { continue; }

                Nw4rSarcGroupImageBlock groupImage = GetGroupImage(group);
                WriteFileMapItem(writer, groupImage.MainImages.Offset,
                                  groupImage.MainImages.Size + groupImage.ARamImages.Size, group.Label, "  ");

            }

            // 匿名グループ
            if (0 == SoundArchive.AnonymousGroup.Files.Count) { return; }

            Nw4rSarcGroupImageBlock anonymousGroupImage = GetGroupImage(SoundArchive.AnonymousGroup);

            for (int i = 0; i < SoundArchive.AnonymousGroup.Files.Count; i++)
            {

                Nw4rSarcGroupItemRamImage itemMainImage = anonymousGroupImage.MainImages.ChildNodes[i] as Nw4rSarcGroupItemRamImage;
                Nw4rSarcGroupItemRamImage itemARamImage = anonymousGroupImage.ARamImages.ChildNodes[i] as Nw4rSarcGroupItemRamImage;

                WriteFileMapItem(writer, itemMainImage.Offset, itemMainImage.Size + itemARamImage.Size,
                                  itemMainImage.BinaryFile.Label, "  ");

            }
        }

        private void WriteFileMapNode(TextWriter writer, Nw4rFileNode fileNode, string text)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != text);

            WriteFileMapItem(writer, fileNode.Offset, fileNode.Size, text, string.Empty);
        }

        private void WriteFileMapSubNode(TextWriter writer, Nw4rFileNode fileNode, string text, string indent)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != text);
            Debug.Assert(null != indent);

            WriteFileMapItem(writer, fileNode.Offset, fileNode.Size, text, indent);
        }

        private void WriteFileMapItem(TextWriter writer, long offset, long size, string text, string indent)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != text);
            Debug.Assert(null != indent);

            writer.Write("  " + indent + CreateHexText(offset, SizeLength));
            writer.Write(" " + CreateHexText(size, SizeLength));
            writer.Write(" " + text);
            writer.Write(writer.NewLine);
        }

        #endregion

        #region ** テキスト出力ヘルパー

        private string CreateText(object value, int alignment)
        {
            Debug.Assert(null != value);
            return string.Format(string.Format("{{0,{0}}}", alignment), value);
        }

        private string CreateHexText(long value, int alignment)
        {
            return string.Format(string.Format("{{0,{0}:x8}}", alignment), value);
        }

        #endregion

    }
}
