﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;

using XmlData = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model
{
    #region ** コレクションの定義

    public class Nw4rItemCollection : Nw4rComponentCollectionAdapter<Nw4rItem>
    {
        public Nw4rItemCollection() { }
        public Nw4rItemCollection(Nw4rComponentCollection components) : base(components) { }
    }
    public class Nw4rSoundArchiveItemCollection : Nw4rComponentCollectionAdapter<Nw4rSoundArchiveItem> { }
    public class Nw4rSoundSetCollection : Nw4rComponentCollectionAdapter<Model.Nw4rSoundSet> { }
    public class Nw4rSoundSetItemCollection : Nw4rComponentCollectionAdapter<Model.Nw4rSoundSetItem> { }
    public class Nw4rSoundCollection : Nw4rComponentCollectionAdapter<Nw4rSound>
    {
        public Nw4rSoundCollection() { }
        public Nw4rSoundCollection(Nw4rComponentCollection components) : base(components) { }
    }
    public class Nw4rBankCollection : Nw4rComponentCollectionAdapter<Nw4rBank> { }
    public class Nw4rInstrumentCollection : Nw4rComponentCollectionAdapter<Nw4rInstrument> { }
    public class Nw4rInstrumentKeyRegionCollection : Nw4rComponentCollectionAdapter<Nw4rInstrumentKeyRegion> { }
    public class Nw4rInstrumentVelocityRegionCollection : Nw4rComponentCollectionAdapter<Nw4rInstrumentVelocityRegion> { }
    public class Nw4rPlayerCollection : Nw4rComponentCollectionAdapter<Nw4rPlayer> { }
    public class Nw4rGroupCollection : Nw4rComponentCollectionAdapter<Nw4rGroup> { }
    public class Nw4rGroupItemCollection : Nw4rComponentCollectionAdapter<Nw4rGroupItem> { }
    public class Nw4rSymbolCollection : Nw4rComponentCollectionAdapter<Nw4rSymbol> { }
    public class Nw4rBinaryFileCollection : Nw4rComponentCollectionAdapter<Nw4rSoundBinaryFile> { }

    #endregion

    public class Nw4rSoundArchive : Nw4rSoundArchiveItem
    {
        #region ** 固定値

        private static readonly string BinaryFileLabelFormat = "FILE_{0:00000}";

        #endregion

        #region ** フィールド

        private readonly string DvdDataDirectoryName = "dvddata";

        private string _projectFilePath = string.Empty;

        private Nw4rSoundMakerProject _xmlProject = null;

        private Nw4rSoundArchiveItemCollection _archiveItems = new Nw4rSoundArchiveItemCollection();
        private Nw4rSoundSetCollection _soundSets = new Nw4rSoundSetCollection();
        private Nw4rSoundCollection _sounds = new Nw4rSoundCollection();
        private Nw4rSoundCollection _streamSounds = new Nw4rSoundCollection();
        private Nw4rSoundCollection _waveSounds = new Nw4rSoundCollection();
        private Nw4rSoundSetItemCollection _waveSoundSets = new Nw4rSoundSetItemCollection();
        private Nw4rSoundCollection _sequenceSounds = new Nw4rSoundCollection();
        private Nw4rSoundSetItemCollection _sequenceSoundSets = new Nw4rSoundSetItemCollection();
        private Nw4rBankCollection _banks = new Nw4rBankCollection();
        private Nw4rPlayerCollection _players = new Nw4rPlayerCollection();
        private Nw4rGroupCollection _groups = new Nw4rGroupCollection();
        private Nw4rGroupCollection _innerGroups = new Nw4rGroupCollection();
        private Nw4rSoundArchiveItemCollection _disabledItems = new Nw4rSoundArchiveItemCollection();

        private Nw4rBankCollection _referencedBanks = new Nw4rBankCollection();
        private Nw4rPlayerCollection _referencedPlayers = new Nw4rPlayerCollection();
        private Nw4rGroupCollection _referencedInnerGroups = new Nw4rGroupCollection();

        private Nw4rSymbolCollection _symbols = new Nw4rSymbolCollection();
        private Nw4rBinaryFileCollection _binaryFiles = new Nw4rBinaryFileCollection();
        private Nw4rGroup _anonymousGroup = null;

        #endregion

        #region ** プロパティ

        public override string Label
        {
            get { return "@Root"; }
        }

        public override string Key
        {
            get { return Label; }
        }

        public string ProjectName
        {
            get { return Path.GetFileNameWithoutExtension(_xmlProject.FilePath); }
        }

        public string ProjectFilePath
        {
            get { return _projectFilePath; }
        }

        public string OutputDirectoryPath
        {
            get
            {
                return Path.Combine(Path.GetDirectoryName(_projectFilePath), _xmlProject.OutputDir);
            }
        }

        public string OutputBinarySoundArchiveFilePath
        {
            get { return Path.Combine(OutputDirectoryPath, Path.GetFileNameWithoutExtension(_xmlProject.FilePath) + ".brsar"); }
        }

        public string DvdDataPath
        {
            get { return Path.Combine(OutputDirectoryPath, DvdDataDirectoryName); }
        }

        public string ExtensionFileDirectoryPath
        {
            get { return _xmlProject.ExtFileDir; }
        }

        public Nw4rSoundArchiveConvertSettings ConvertSettings
        {
            get { return _xmlProject.SoundArchiveConvertSettings; }
        }

        public string IntermediateDirectoryPath
        {
            get { return _xmlProject.IntermediateOutputDir; }
        }

        public bool AggregateIntermediateOutput
        {
            get { return _xmlProject.AggregateIntermediateOutput; }
        }

        public bool KeepIntermediateRseq
        {
            get { return _xmlProject.KeepIntermediateRseq; }
        }

        public int SmfConvertTimeBase
        {
            get
            {
                if (null == _xmlProject) { throw new Nw4rFileFormatInternalException(); }
                return _xmlProject.SmfConvertTimebase;
            }
        }

        public Nw4rSoundArchivePlayerSettings SoundArchivePlayerSettings
        {
            get { return _xmlProject.SoundArchivePlayerSettings; }
        }

        #region ** コンポーネント コレクション

        public INw4rComponentCollection<Nw4rSoundArchiveItem> AllItems
        {
            get { return _archiveItems; }
        }

        public INw4rComponentCollection<Model.Nw4rSoundSet> AllSoundSets
        {
            get { return _soundSets; }
        }

        public INw4rComponentCollection<Nw4rSound> AllSounds
        {
            get { return _sounds; }
        }

        public INw4rComponentCollection<Nw4rSound> AllStreamSounds
        {
            get { return _streamSounds; }
        }

        public INw4rComponentCollection<Nw4rSound> AllWaveSounds
        {
            get { return _waveSounds; }
        }

        public INw4rComponentCollection<Model.Nw4rSoundSetItem> AllWaveSoundSets
        {
            get { return _waveSoundSets; }
        }

        public INw4rComponentCollection<Nw4rSound> AllSequenceSounds
        {
            get { return _sequenceSounds; }
        }

        public INw4rComponentCollection<Model.Nw4rSoundSetItem> AllSequenceSoundSets
        {
            get { return _sequenceSoundSets; }
        }

        public INw4rComponentCollection<Nw4rBank> AllBanks
        {
            get { return _banks; }
        }

        public INw4rComponentCollection<Nw4rPlayer> AllPlayers
        {
            get { return _players; }
        }

        public INw4rComponentCollection<Nw4rGroup> AllGroups
        {
            get { return _groups; }
        }

        public INw4rComponentCollection<Nw4rGroup> AllInnerGroups
        {
            get { return _innerGroups; }
        }

        public INw4rComponentCollection<Nw4rSoundArchiveItem> DisabledItems
        {
            get { return _disabledItems; }
        }

        public INw4rComponentCollection<Nw4rBank> ReferencedBanks
        {
            get { return _referencedBanks; }
        }

        public INw4rComponentCollection<Nw4rPlayer> ReferencedPlayers
        {
            get { return _referencedPlayers; }
        }

        public INw4rComponentCollection<Nw4rGroup> ReferencedInnerGroups
        {
            get { return _referencedInnerGroups; }
        }

        public INw4rComponentCollection<Nw4rSymbol> Symbols
        {
            get { return _symbols; }
        }

        public INw4rComponentCollection<Nw4rSoundBinaryFile> BinaryFiles
        {
            get { return _binaryFiles; }
        }

        public Nw4rGroup AnonymousGroup
        {
            get { return _anonymousGroup; }
        }

        #endregion

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareConvert()
        {
            // グループ以外を先に処理する
            foreach (Model.Nw4rSoundSet soundSet in Components)
            {

                foreach (Model.Nw4rSoundSetItem soundSetItem in soundSet.Components)
                {
                    if (soundSetItem is Nw4rGroup) { continue; }
                    soundSetItem.PrepareConvert();
                }

            }

            foreach (Model.Nw4rSoundSet soundSet in Components)
            {

                foreach (Nw4rGroup group in soundSet.Groups)
                {
                    group.PrepareConvert();
                }

            }
        }

        protected override void OnPrepareLink()
        {
            base.OnPrepareLink();

            // 匿名グループの作成
            CreateAnonymousGroup();
        }

        #endregion

        #region ** メソッド

        public void LoadProject(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            _xmlProject = new Nw4rSoundMakerProject();
            _xmlProject.Load(filePath);
            _projectFilePath = filePath;

            // サウンドセットをロードする
            foreach (Nw4rSoundMakerProject.SoundSet soundsetInfo in _xmlProject.SoundSets)
            {

                XmlData.Nw4rSoundSet xmlSoundSet = new XmlData.Nw4rSoundSet(soundsetInfo.Name);
                xmlSoundSet.Load(Path.Combine(Path.GetDirectoryName(filePath), soundsetInfo.FilePath));

                Model.Nw4rSoundSet soundSet = new Model.Nw4rSoundSet(xmlSoundSet);

                Add(soundSet);
                _soundSets.Add(soundSet);

            }

            // コンポーネントの作成
            CreateComponents();

            // グループの作成
            CreateGroupComponents();

            // シンボルの作成
            CreateSymbols();
        }

        #region ** サウンドコンポーネントの作成

        private void CreateComponents()
        {
            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                CreateStreamSoundComponent(soundSet);
                CreateWaveSoundSetComponent(soundSet);
                CreateSequenceSoundComponent(soundSet);
                CreateSequenceSoundSetComponent(soundSet);
                CreateBankComponent(soundSet);
                CreatePlayerComponent(soundSet);
            }
        }

        private void CreateGroupComponents()
        {
            Dictionary<string, Nw4rGroup> groups = new Dictionary<string, Nw4rGroup>();

            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                CreateGroupComponent(soundSet);
            }
        }

        private void CreateStreamSoundComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetStrm xmlStreamSound in soundSet.XmlData.StrmList)
            {

                // サウンドの生成
                Nw4rStreamSound sound = new Nw4rStreamSound(xmlStreamSound);

                if (!xmlStreamSound.Enabled)
                {
                    _disabledItems.Add(sound);
                    continue;
                }

                AddItem(sound);
                soundSet.AddItem(sound);
                soundSet.Add(sound);

            }
        }

        private void CreateWaveSoundSetComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            string wsdOutputDirectory = Path.Combine(Path.GetDirectoryName(soundSet.XmlData.FilePath), @"wsd");

            foreach (Nw4rSoundSetWaveSoundSet xmlWsdSet in soundSet.XmlData.WaveSoundSetList)
            {

                // ウェーブサウンドセットの生成
                Nw4rWaveSoundSet wsdSet = new Nw4rWaveSoundSet(xmlWsdSet);

                // WaveSound コンポーネントの生成
                foreach (Nw4rSoundSetWaveSound xmlWsd in xmlWsdSet)
                {

                    // ウェーブサウンドの生成
                    Nw4rWaveSound wsd = new Nw4rWaveSound(xmlWsd);

                    if (!xmlWsdSet.Enabled || !xmlWsd.Enabled)
                    {
                        _disabledItems.Add(wsd);
                        continue;
                    }

                    AddItem(wsd);
                    soundSet.AddItem(wsd);
                    wsdSet.Add(wsd);

                }

                if (!xmlWsdSet.Enabled)
                {
                    _disabledItems.Add(wsdSet);
                    continue;
                }

                AddItem(wsdSet);
                soundSet.AddItem(wsdSet);
                soundSet.Add(wsdSet);

            }
        }

        private void CreateSequenceSoundComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetSeq xmlSeqSound in soundSet.XmlData.SeqList)
            {

                // シーケンスサウンドの生成
                Nw4rSequenceSound sound = new Nw4rSequenceSound(xmlSeqSound);

                if (!xmlSeqSound.Enabled)
                {
                    _disabledItems.Add(sound);
                    continue;
                }

                AddItem(sound);
                soundSet.AddItem(sound);
                soundSet.Add(sound);

            }
        }

        private void CreateSequenceSoundSetComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetSeqSoundSet xmlSeqSoundSet in soundSet.XmlData.SeqSoundSetList)
            {

                // シーケンスサウンドセットの生成
                Nw4rSequenceSoundSet seqSoundSet = new Nw4rSequenceSoundSet(xmlSeqSoundSet);

                foreach (Nw4rSoundSetSeq xmlSeqSound in xmlSeqSoundSet)
                {

                    // シーケンスサウンドの生成
                    Nw4rSequenceSound seqSound = new Nw4rSequenceSound(xmlSeqSound);

                    if (!xmlSeqSoundSet.Enabled || !xmlSeqSound.Enabled)
                    {
                        _disabledItems.Add(seqSound);
                        continue;
                    }

                    AddItem(seqSound);
                    soundSet.AddItem(seqSound);
                    seqSoundSet.Add(seqSound);

                }

                if (!xmlSeqSoundSet.Enabled)
                {
                    _disabledItems.Add(seqSoundSet);
                    continue;
                }

                AddItem(seqSoundSet);
                soundSet.AddItem(seqSoundSet);
                soundSet.Add(seqSoundSet);

            }
        }

        private void CreateBankComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetBank xmlBank in soundSet.XmlData.BankList)
            {

                // バンクの生成
                Nw4rBank bank = new Nw4rBank(xmlBank);

                if (!xmlBank.Enabled)
                {
                    _disabledItems.Add(bank);
                    continue;
                }

                AddItem(bank);
                soundSet.AddItem(bank);
                soundSet.Add(bank);

            }
        }

        private void CreatePlayerComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetPlayer xmlPlayer in soundSet.XmlData.PlayerList)
            {

                // プレイヤーの生成
                Nw4rPlayer player = new Nw4rPlayer(xmlPlayer);

                if (!xmlPlayer.Enabled)
                {
                    _disabledItems.Add(player);
                    continue;
                }

                AddItem(player);
                soundSet.AddItem(player);
                soundSet.Add(player);

            }
        }

        private void CreateGroupComponent(Model.Nw4rSoundSet soundSet)
        {
            if (null == soundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("soundSet")); }

            foreach (Nw4rSoundSetGroup xmlGroup in soundSet.XmlData.GroupList)
            {

                Nw4rGroup newGroup = new Nw4rGroup(xmlGroup);

                if (!xmlGroup.Enabled)
                {
                    _disabledItems.Add(newGroup);
                    continue;
                }

                AddItem(newGroup);
                soundSet.AddItem(newGroup);
                soundSet.Add(newGroup);

            }
        }

        private void AddItem(Nw4rStreamSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _streamSounds.Add(sound);
        }

        private void AddItem(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { throw new ArgumentNullException("waveSoundSet"); }

            AddItem(waveSoundSet as Nw4rSoundArchiveItem);
            _waveSoundSets.Add(waveSoundSet);
        }

        private void AddItem(Nw4rWaveSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _waveSounds.Add(sound);
        }

        private void AddItem(Nw4rSequenceSoundSet sequenceSoundSet)
        {
            if (null == sequenceSoundSet) { throw new ArgumentNullException("sequenceSoundSet"); }

            AddItem(sequenceSoundSet as Nw4rSoundArchiveItem);
            _sequenceSoundSets.Add(sequenceSoundSet);
        }

        private void AddItem(Nw4rSequenceSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _sequenceSounds.Add(sound);
        }

        private void AddItem(Nw4rSound sound)
        {
            Debug.Assert(null != sound);
            AddItem(sound as Nw4rSoundArchiveItem);
            _sounds.Add(sound);
        }

        private void AddItem(Nw4rBank bank)
        {
            if (null == bank) { throw new ArgumentNullException("bank"); }

            AddItem(bank as Nw4rSoundArchiveItem);
            _banks.Add(bank);
        }

        private void AddItem(Nw4rPlayer player)
        {
            if (null == player) { throw new ArgumentNullException("player"); }

            AddItem(player as Nw4rSoundArchiveItem);
            _players.Add(player);
        }

        private void AddItem(Nw4rGroup group)
        {
            AddItem(group, true);
        }

        private void AddItem(Nw4rGroup group, bool checkLabel)
        {
            if (null == group) { throw new ArgumentNullException("group"); }

            AddItem(group as Nw4rSoundArchiveItem, checkLabel);

            if (group.XmlData.OutputFlag)
            {
                _groups.Add(group);
            }
            else
            {
                _innerGroups.Add(group);
            }
        }

        private void AddItem(Nw4rSoundArchiveItem item)
        {
            AddItem(item, true);
        }

        private void AddItem(Nw4rSoundArchiveItem item, bool checkLabel)
        {
            Debug.Assert(null != item);

            if (checkLabel)
            {
                if (0 == item.Label.Length || Regex.Match(item.Label, "\\W").Success)
                {
                    string ownerLabel = (null != item.Parent) ? (item.Parent as Nw4rSoundArchiveItem).Label : string.Empty;
                    throw new Nw4rInvalidLabelException(item.Label, ownerLabel);
                }
            }

            if (_archiveItems.Contains(item.Label)) { throw new Nw4rMultipleLabelException(item.Label); }
            _archiveItems.Add(item);
        }

        #endregion

        #region ** シンボルの作成

        private void CreateSymbols()
        {
            // プレイヤー
            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                foreach (Model.Nw4rSoundSetItem item in soundSet.Players)
                {
                    if (0 == item.Label.Length) { throw new Nw4rInvalidLabelException(string.Empty, string.Empty); }
                    _symbols.Add(new Nw4rSymbol(item));
                }
            }

            // グループ
            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                foreach (Model.Nw4rSoundSetItem item in soundSet.Groups)
                {
                    if (0 == item.Label.Length)
                    {
                        if (item != _anonymousGroup) { throw new Nw4rInvalidLabelException(string.Empty, string.Empty); }
                        continue;
                    }
                    _symbols.Add(new Nw4rSymbol(item));
                }
            }

            // バンク
            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                foreach (Model.Nw4rSoundSetItem item in soundSet.Banks)
                {
                    if (0 == item.Label.Length) { throw new Nw4rInvalidLabelException(string.Empty, string.Empty); }
                    _symbols.Add(new Nw4rSymbol(item));
                }
            }

            // サウンド
            foreach (Model.Nw4rSoundSet soundSet in _soundSets)
            {
                foreach (Model.Nw4rSoundSetItem item in soundSet.Sounds)
                {
                    if (0 == item.Label.Length) { throw new Nw4rInvalidLabelException(string.Empty, string.Empty); }
                    _symbols.Add(new Nw4rSymbol(item));
                }
            }
        }

        #endregion

        #region ** サウンドファイルの追加

        public void AddBinaryFile(Nw4rSoundBinaryFile file)
        {
            if (null == file) { throw new Nw4rFileFormatInternalException("binaryFile is null"); }

            if (_binaryFiles.Contains(file.Key)) { return; }

            file.Label = CreateBinaryFileLabel();
            _binaryFiles.Add(file);
        }

        private string CreateBinaryFileLabel()
        {
            return string.Format(BinaryFileLabelFormat, _binaryFiles.Count);
        }

        #endregion

        #region ** サウンドグループの作成

        private void CreateAnonymousGroup()
        {
            Nw4rSoundSetGroup xmlGroup = new Nw4rSoundSetGroup();
            xmlGroup.Label = string.Empty;

            _anonymousGroup = new Nw4rGroup(xmlGroup);
            AddItem(_anonymousGroup, false);
            Add(_anonymousGroup);

            // サウンドアーカイブに含まれるファイルで、どのグループにも含まれないものを
            // 匿名グループに含める
            foreach (Nw4rSoundBinaryFile file in BinaryFiles)
            {

                if (file is Nw4rGroupBinaryFile) { continue; }
                if (0 < file.Groups.Count) { continue; }
                if (0 < file.InnerGroups.Count) { continue; }
                if (0 < file.ExtensionFileRelativePath.Length) { continue; }

                _anonymousGroup.AddFile(file);

            }
        }

        #endregion

        #endregion
    }

    /// <summary>
    /// サウンドアーカイブコンバート警告フラグ
    /// </summary>
    [Flags]
    public enum Nw4rSoundArchiveConvertWarningFlags : uint
    {
        NoWarning = 0x00000000,
        WarnUnreferencedItem = 0x00000001,
        WarnIndependentItem = 0x00000002,
        WarnAll = 0xffffffff
    }
}
