﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model
{
    public class Nw4rSoundFile : Nw4rComponent
    {
        #region ** フィールド

        private string _label = string.Empty;
        private string _filePath = string.Empty;
        private string _key = null;

        #endregion

        public Nw4rSoundFile(string filePath) : this(filePath, null) { }

        public Nw4rSoundFile(string filePath, string key)
        {
            if (null == filePath) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("filePath")); }
            _filePath = filePath;
            _key = key;
        }

        #region ** プロパティ

        public string Label
        {
            get { return _label; }
            set
            {
                if (null == value) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("value")); }
                _label = value;
            }
        }

        public string FilePath
        {
            get { return _filePath; }
        }

        public override string Key
        {
            get { return (null == _key) ? FilePath : _key; }
        }

        #endregion
    }

    public class Nw4rSoundIntermediateFile : Nw4rSoundFile
    {
        public Nw4rSoundIntermediateFile(string filePath) : base(filePath) { }
    }

    public class Nw4rSoundBinaryFile : Nw4rSoundFile
    {
        #region ** フィールド

        private string _aramFilePath = string.Empty;
        private string _externalFilePath = string.Empty;

        private Nw4rGroupCollection _groups = new Nw4rGroupCollection();
        private Nw4rGroupCollection _innerGroups = new Nw4rGroupCollection();

        #endregion

        public Nw4rSoundBinaryFile(string filePath) : base(filePath) { }
        public Nw4rSoundBinaryFile(string filePath, string key) : base(filePath, key) { }

        #region ** プロパティ

        public string ARamFilePath
        {
            get { return _aramFilePath; }
            set
            {
                if (null == value) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("value")); }
                _aramFilePath = value;
            }
        }

        public string ExtensionFileRelativePath
        {
            get { return _externalFilePath; }
            set
            {
                if (null == value) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("value")); }
                _externalFilePath = value;
            }
        }

        public Nw4rGroupCollection Groups
        {
            get { return _groups; }
        }

        public Nw4rGroupCollection InnerGroups
        {
            get { return _innerGroups; }
        }

        #endregion
    }

    public class Nw4rStreamSoundBinaryFile : Nw4rSoundBinaryFile
    {
        #region ** フィールド

        private Nw4rStreamSoundFile _file = null;

        #endregion

        public Nw4rStreamSoundBinaryFile(string filePath, string externalFileRelativePath) : base(filePath)
        {
            ExtensionFileRelativePath = (null == externalFileRelativePath) ? string.Empty : externalFileRelativePath;
        }

        #region ** メソッド

        /// <summary>
        /// バイナリ情報を取得します。
        /// </summary>
        /// <param name="sound">対象サウンド</param>
        /// <param name="allocTracks">アロケートトラックフラグ</param>
        /// <param name="channels">チャンネル数</param>
        public void GetBinaryInfo(Nw4rStreamSound sound, out int allocTracks, out int channels)
        {
            if (null == sound) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("sound")); }

            try
            {

                if (null == _file)
                {
                    _file = Nw4rStreamSoundFile.FromFile(FilePath);
                }

                // バイナリファイルから情報を抽出する
                allocTracks = GetAllocTracksFrag(_file);
                channels = GetChannels(_file);

            }
            catch (Nw4rFileFormatException exception)
            {
                throw exception;
            }
            catch (Exception exception)
            {
                throw new Nw4rFileFormatInternalException(exception);
            }
        }

        private int GetAllocTracksFrag(Nw4rStreamSoundFile file)
        {
            Debug.Assert(null != file);

            Nw4rStreamHeadBlockBody body = file.ChildNodes[Nw4rStreamHeadBlock.NodeName].ChildNodes[Nw4rStreamHeadBlockBody.NodeName] as Nw4rStreamHeadBlockBody;
            Debug.Assert(null != body);

            int workAllocTracks = 0;
            int trackIndex = 0;

            foreach (Nw4rFileInnerOffsetRefTableItem item in body.TrackTable.Items)
            {

                if (item.Reference != Nw4rFileDataOffsetRef.InvalidValue)
                {
                    workAllocTracks |= 1 << trackIndex;
                }

                trackIndex++;

            }

            return workAllocTracks;
        }

        private int GetChannels(Nw4rStreamSoundFile file)
        {
            Debug.Assert(null != file);

            Nw4rStreamHeadBlockBody body = file.ChildNodes[Nw4rStreamHeadBlock.NodeName].ChildNodes[Nw4rStreamHeadBlockBody.NodeName] as Nw4rStreamHeadBlockBody;
            Debug.Assert(null != body);

            return body.ChannelTable.Items.Count;
        }

        #endregion
    }

    public class Nw4rWaveSoundSetBinaryFile : Nw4rSoundBinaryFile
    {
        public Nw4rWaveSoundSetBinaryFile(string filePath, string aramFilePath) : this(filePath, aramFilePath, null) { }
        public Nw4rWaveSoundSetBinaryFile(string filePath, string aramFilePath, string key) : base(filePath, key)
        {
            ARamFilePath = aramFilePath;
        }
    }

    public class Nw4rSequenceSoundBinaryFile : Nw4rSoundBinaryFile
    {
        #region ** 固定値

        private const Byte MML_ALLOC_TRACK = 0xfe;

        #endregion

        #region ** フィールド

        private Nw4rSequenceSoundFile _file = null;

        #endregion

        public Nw4rSequenceSoundBinaryFile(string filePath) : base(filePath) { }

        #region ** メソッド

        /// <summary>
        /// バイナリ情報を取得します。
        /// </summary>
        /// <param name="sound">対象サウンド</param>
        /// <param name="startOffset">再生開始オフセット</param>
        /// <param name="allocTracks">アロケートトラックフラグ</param>
        public void GetBinaryInfo(Nw4rSequenceSound sound, out long startOffset, out int allocTracks)
        {
            if (null == sound) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("sound")); }

            if (null == _file)
            {
                _file = Nw4rSequenceSoundFile.FromFile(FilePath);
            }

            // 再生開始ラベルから再生開始オフセットとアロケートトラックフラグを取得する
            long workStartOffset = GetStartOffset(_file, sound);
            int workAllocTracks = GetAllocTracksFrag(_file, workStartOffset);

            // AllocTrack コマンドがある場合は、コマンド分だけオフセットを進めておく
            if (1 != workAllocTracks)
            {
                workStartOffset += 3;
            }

            startOffset = workStartOffset;
            allocTracks = workAllocTracks;
        }

        private long GetStartOffset(Nw4rSequenceSoundFile file, Nw4rSequenceSound sound)
        {
            Debug.Assert(null != file);
            Debug.Assert(null != sound);

            // 再生開始ラベルがない場合は、オフセットは 0
            if (null == sound.XmlData.StartPosition || 0 == sound.XmlData.StartPosition.Length) { return 0; }

            // 再生開始ラベルがある場合は、再生開始ラベルを検索してオフセットを返す
            Nw4rSequenceLabelBlock labelBlock = file.ChildNodes[Nw4rSequenceLabelBlock.NodeName] as Nw4rSequenceLabelBlock;
            if (null == labelBlock) { throw new Nw4rFileFormatInternalException(); }

            Nw4rSequenceLabelData targetLabel = labelBlock.FindLabelData(sound.XmlData.StartPosition);

            if (null == targetLabel)
            {
                throw new Nw4rSequenceStartLabelNotFoundException(sound.XmlData.StartPosition, sound.XmlData.Label);
            }

            return targetLabel.DataOffset;
        }

        private int GetAllocTracksFrag(Nw4rSequenceSoundFile file, long startOffset)
        {
            Debug.Assert(null != file);
            Debug.Assert(0 <= startOffset);

            Nw4rSequenceDataBlock dataBlock = file.ChildNodes[Nw4rSequenceDataBlock.NodeName] as Nw4rSequenceDataBlock;
            if (null == dataBlock) { throw new Nw4rFileFormatInternalException(); }

            BinaryReader dataReader = dataBlock.CreateDataReader();


            // データブロックのオフセットを読み込んで、該当位置までシークする
            long offset = dataReader.ReadUInt32();
            dataReader.BaseStream.Seek(offset - Nw4rFileDataBlockHeader.DataSize + startOffset, SeekOrigin.Begin);


            // 該当位置のコマンドを読み込む
            Byte[] command = dataReader.ReadBytes(3);

            // AllocTrack コマンドでない場合は、0番トラックのみ
            // AllocTrack コマンドの場合は、0番トラック + ビット演算結果を返す
            if (MML_ALLOC_TRACK != command[0]) { return 1; }
            return 1 | command[1] << 8 | command[2];
        }

        #endregion
    }

    public class Nw4rBankBinaryFile : Nw4rSoundBinaryFile
    {
        private string _includeFilePath = string.Empty;

        public Nw4rBankBinaryFile(string filePath, string aramFilePath, string includeFilePath)
            : this(filePath, aramFilePath, includeFilePath, null) { }
        public Nw4rBankBinaryFile(string filePath, string aramFilePath, string includeFilePath, string key) : base(filePath, key)
        {
            ARamFilePath = aramFilePath;
            _includeFilePath = (null == includeFilePath) ? string.Empty : includeFilePath;
        }

        #region ** プロパティ

        public string IncludeFilePath
        {
            get { return _includeFilePath; }
        }

        #endregion
    }

    public class Nw4rGroupBinaryFile : Nw4rSoundBinaryFile
    {
        public Nw4rGroupBinaryFile(string aramFilePath) : base(string.Empty)
        {
            ARamFilePath = aramFilePath;
        }

        public override string Key
        {
            get { return ARamFilePath; }
        }
    }

    public class Nw4rSoundBinaryFileForGroup : Nw4rSoundBinaryFile
    {
        Nw4rSoundBinaryFile _originalFile = null;

        public Nw4rSoundBinaryFileForGroup(string filePath, Nw4rSoundBinaryFile originalFile) : base(filePath)
        {
            if (null == originalFile) { throw new Nw4rFileFormatInternalException("binary file for group has no original file"); }
            _originalFile = originalFile;
        }

        #region ** プロパティ

        public Nw4rSoundBinaryFile OriginalFile
        {
            get { return _originalFile; }
        }

        public override string Key
        {
            get { return _originalFile.Key; }
        }

        #endregion
    }
}
