﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;

using XmlData = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model
{
    #region ** 基本コンポーネント

    public abstract class Nw4rComponent : IPatriciaTreeItem
    {
        #region ** フィールド

        private Nw4rComponent _parent = null;
        private Nw4rComponentCollection _components = new Nw4rComponentCollection();

        private string _label = string.Empty;

        #endregion

        #region ** プロパティ

        public Nw4rComponent Root
        {
            get
            {
                if (null == _parent) { return this; }
                return _parent.Root;
            }
        }

        public Nw4rComponent Parent
        {
            get { return _parent; }
            set { SetParent(value); }
        }

        public Nw4rComponentCollection Components
        {
            get { return _components; }
        }

        public abstract string Key { get; }

        #endregion

        #region ** メソッド

        internal void Add(Nw4rComponent component)
        {
            if (null == component) { throw new ArgumentNullException("component"); }
            if (null != component.Parent) { throw new ArgumentException(); }

            component._parent = this;
            _components.Add(component);
        }

        protected void SetParent(Nw4rComponent parent)
        {
            if (null == parent) { throw new ArgumentNullException(); }
            parent.Add(this);
        }

        #endregion
    }

    #region ** コレクション

    public interface INw4rComponentCollection : INw4rCollection
    {
        #region ** メソッド

        /// <summary>
        /// 指定キーをもつアイテムがコレクション内に含まれているかどうかを調べます。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクションに含まれている場合はtrue、含まれていない場合はfalse。</returns>
        bool Contains(string key);

        /// <summary>
        /// 指定アイテムのインデックスを調べます。
        /// </summary>
        /// <param name="item">アイテム</param>
        /// <returns>指定アイテムのインデックス。</returns>
        int IndexOf(string key);

        #endregion
    }

    public interface INw4rComponentCollection<_ItemType> : INw4rCollection<_ItemType>, INw4rComponentCollection
    {
        #region ** インデクサ

        /// <summary>
        /// 指定したキーをもつ最初のアイテムを取得します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクション内のアイテム。該当ノードが存在しない場合、null を返します。</returns>
        _ItemType this[string key] { get; }

        #endregion
    }

    public class Nw4rComponentCollection : Nw4rIndexedCollection<string, Nw4rComponent>, INw4rComponentCollection
    {
        protected override string GetKeyForItem(Nw4rComponent item)
        {
            return item.Key;
        }
    }

    public class Nw4rComponentCollectionAdapter<_ItemType> : INw4rComponentCollection<_ItemType>
        where _ItemType : Nw4rComponent
    {
        private Nw4rComponentCollection _components = null;

        public Nw4rComponentCollectionAdapter() : this(new Nw4rComponentCollection()) { }

        public Nw4rComponentCollectionAdapter(Nw4rComponentCollection components)
        {
            Debug.Assert(null != components);
            _components = components;
        }

        #region ** プロパティ

        /// <summary>
        /// コレクションに格納されているアイテム数を取得します。
        /// </summary>
        public int Count
        {
            get { return _components.Count; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 指定キーをもつアイテムがコレクション内に含まれているかどうかを調べます。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクションに含まれている場合はtrue、含まれていない場合はfalse。</returns>
        public bool Contains(string key)
        {
            return _components.Contains(key);
        }

        /// <summary>
        /// 指定キーをもつアイテムのインデックスを調べます。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>指定キーをもつアイテムのインデックス。</returns>
        public int IndexOf(string key)
        {
            return _components.IndexOf(key);
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>列挙子</returns>
        public IEnumerator<_ItemType> GetEnumerator()
        {
            return new Enumerater(_components.GetEnumerator());
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>列挙子</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return _components.GetEnumerator();
        }

        /// <summary>
        /// アイテムを追加します。
        /// </summary>
        /// <param name="item">新しいアイテム</param>
        public void Add(_ItemType item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            _components.Add(item);
        }

        public void Clear()
        {
            _components.Clear();
        }

        #endregion

        #region ** インデクサ

        /// <summary>
        /// 指定したインデックス位置にあるアイテムを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のアイテム。インデックスがノード数以上の場合、null を返します。</returns>
        public _ItemType this[int index]
        {
            get { return _components[index] as _ItemType; }
        }

        /// <summary>
        /// 指定したキーをもつ最初のアイテムを取得します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクション内のアイテム。該当ノードが存在しない場合、null を返します。</returns>
        public _ItemType this[string key]
        {
            get { return _components[key] as _ItemType; }
        }

        #endregion

        #region ** 列挙クラス

        private class Enumerater : IEnumerator<_ItemType>
        {
            private IEnumerator<Nw4rComponent> _enumerator = null;

            public Enumerater(IEnumerator<Nw4rComponent> enumerator)
            {
                Debug.Assert(null != enumerator);
                _enumerator = enumerator;
            }

            public _ItemType Current
            {
                get { return _enumerator.Current as _ItemType; }
            }

            object IEnumerator.Current
            {
                get { return _enumerator.Current; }
            }

            public bool MoveNext()
            {
                return _enumerator.MoveNext();
            }

            public void Reset()
            {
                _enumerator.Reset();
            }

            public void Dispose()
            {
                _enumerator.Dispose();
            }
        }

        #endregion
    }

    #endregion

    #endregion

    #region ** サウンド コンポーネント

    #region ** 基本クラス

    /// <summary>
    /// サウンドコンテナ インターフェイス
    /// </summary>
    public interface INw4rSoundContainer
    {
        INw4rComponentCollection<Nw4rSound> Sounds { get; }
    }

    public abstract class Nw4rItem : Nw4rComponent
    {
        #region ** プロパティ

        public abstract string Label { get; }

        public virtual string AbsoluteKey
        {
            get
            {
                if (null == Parent) { return Label; }
                return (Parent as Nw4rItem).AbsoluteKey + "." + Label;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnResolveName(Nw4rFileFormatExceptionList results)
        {
            foreach (Nw4rComponent component in Components)
            {

                Nw4rItem item = component as Nw4rItem;
                if (null == item) { continue; }

                item.OnResolveName(results);

            }
        }

        protected virtual void OnPrepareConvert()
        {
            foreach (Nw4rComponent component in Components)
            {

                Nw4rItem item = component as Nw4rItem;
                if (null == item) { continue; }

                item.OnPrepareConvert();

            }
        }

        protected virtual void OnPrepareLink()
        {
            foreach (Nw4rComponent component in Components)
            {

                Nw4rItem item = component as Nw4rItem;
                if (null == item) { continue; }

                item.OnPrepareLink();

            }
        }

        #endregion

        #region ** メソッド

        public void ResolveName(Nw4rFileFormatExceptionList results)
        {
            Debug.Assert(null != Root);
            OnResolveName(results);
        }

        public void PrepareConvert()
        {
            Debug.Assert(null != Root);
            OnPrepareConvert();
        }

        public void PrepareLink()
        {
            Debug.Assert(null != Root);
            OnPrepareLink();
        }

        #endregion
    }

    public abstract class Nw4rSoundArchiveItem : Nw4rItem
    {
        #region ** プロパティ

        public new Nw4rSoundArchive Root
        {
            get { return base.Root as Nw4rSoundArchive; }
        }

        public override string AbsoluteKey
        {
            // ラベルは一意であることが保障されるので、ラベルでOK
            get { return Label; }
        }

        #endregion
    }

    public abstract class Nw4rBankItem : Nw4rItem
    {
        #region ** プロパティ

        public Nw4rBank Bank
        {
            get
            {
                if (null == Parent) { return null; }

                Nw4rBankItem parent = (Parent as Nw4rBankItem);

                if (null == parent) { return Parent as Nw4rBank; }
                return parent.Bank;
            }
        }

        #endregion
    }

    public abstract class Nw4rInstrumentItem : Nw4rBankItem
    {
        #region ** プロパティ

        public Nw4rInstrument Instrument
        {
            get
            {
                if (null == Parent) { return null; }

                Nw4rInstrumentItem parent = (Parent as Nw4rInstrumentItem);

                if (null == parent) { return Parent as Nw4rInstrument; }
                return parent.Instrument;
            }
        }

        #endregion
    }

    public class Nw4rSoundSetItem : Nw4rSoundArchiveItem
    {
        #region ** 固定値

        private const string PropertyName_BinaryFileForGroup = "BinaryFileForGroup";

        #endregion

        #region ** フィールド

        private XmlData.Nw4rSoundSetItem _xmlData = null;
        private Nw4rSoundBinaryFile _binaryFile = null;

        private AttachedPropertyDictionary _attachedProperties = new AttachedPropertyDictionary();

        #endregion

        protected Nw4rSoundSetItem(XmlData.Nw4rSoundSetItem xmlData)
        {
            if (null == xmlData) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("xmlData")); }
            _xmlData = xmlData;
        }

        #region ** プロパティ

        public XmlData.Nw4rSoundSetItem XmlData
        {
            get { return _xmlData; }
        }

        public override string Label
        {
            get
            {
                if (null == _xmlData) { return string.Empty; }

                return _xmlData.Label;
            }
        }

        public override string Key
        {
            get { return Label; }
        }

        public virtual Nw4rSoundBinaryFile BinaryFile
        {
            get { return _binaryFile; }
            set { _binaryFile = value; }
        }

        #endregion

        #region ** 添付プロパティ

        public Nw4rSoundBinaryFile GetBinaryFileForGroup(Nw4rGroup group)
        {
            return GetAttachedProperty(new AttachedPropertyID(group, PropertyName_BinaryFileForGroup)) as Nw4rSoundBinaryFile;
        }

        public void SetBinaryFileForGroup(Nw4rGroup group, Nw4rSoundBinaryFile binaryFile)
        {
            SetAttachedProperty(new AttachedPropertyID(group, PropertyName_BinaryFileForGroup), binaryFile);
        }

        public object GetAttachedProperty(AttachedPropertyID id)
        {
            if (!_attachedProperties.ContainsKey(id)) { return null; }
            return _attachedProperties[id];
        }

        public void SetAttachedProperty(AttachedPropertyID id, object value)
        {
            Debug.Assert(null != value);
            _attachedProperties.Add(id, value);
        }

        #endregion
    }

    public abstract class Nw4rSound : Nw4rSoundSetItem
    {
        private Nw4rPlayer _player = null;

        protected Nw4rSound(Nw4rSoundSetSoundItem soundSetItem) : base(soundSetItem) { }

        #region ** プロパティ

        public new Nw4rSoundSetSoundItem XmlData
        {
            get { return base.XmlData as Nw4rSoundSetSoundItem; }
        }

        public Nw4rPlayer Player
        {
            get { return _player; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnResolveName(Nw4rFileFormatExceptionList results)
        {
            base.OnResolveName(results);
            Debug.Assert(null != Root);

            // プレイヤーラベルの存在確認
            if (!Root.Symbols.Contains(XmlData.Player))
            {
                results.Add(new Nw4rPlayerLabelNotFoundException(XmlData.Player, XmlData.Label));
                return;
            }

            // プレイヤーコンポーネントの存在確認
            if (!Root.AllPlayers.Contains(XmlData.Player))
            {
                results.Add(new Nw4rPlayerLabelNotFoundException(XmlData.Player, XmlData.Label));
                return;
            }

            _player = Root.AllPlayers[XmlData.Player];

            // 参照リストに追加する
            if (!Root.ReferencedPlayers.Contains(_player.Key))
            {
                (Root.ReferencedPlayers as Nw4rPlayerCollection).Add(_player);
            }
        }

        protected override void OnPrepareConvert()
        {
            base.OnPrepareConvert();

            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException("sound binary file not found");
            }

            if (!Root.BinaryFiles.Contains(BinaryFile.Key))
            {
                throw new Nw4rFileFormatInternalException(
                            string.Format("sound binary file not registered \"{0}\"", BinaryFile.Key));
            }
        }

        protected override void OnPrepareLink()
        {
            base.OnPrepareLink();

            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException("sound binary file is null");
            }
            if (!File.Exists(BinaryFile.FilePath))
            {
                throw new Nw4rFileNotFoundException(BinaryFile.FilePath, Label);
            }
        }

        #endregion
    }

    #endregion

    #region ** サウンドセット

    public class Nw4rSoundSet : Nw4rSoundArchiveItem
    {
        #region ** フィールド

        private XmlData.Nw4rSoundSet _xmlData;

        private Nw4rSoundCollection _sounds = new Nw4rSoundCollection();
        private Nw4rSoundCollection _streamSounds = new Nw4rSoundCollection();
        private Nw4rSoundCollection _waveSounds = new Nw4rSoundCollection();
        private Nw4rSoundSetItemCollection _waveSoundSets = new Nw4rSoundSetItemCollection();
        private Nw4rSoundCollection _sequenceSounds = new Nw4rSoundCollection();
        private Nw4rBankCollection _banks = new Nw4rBankCollection();
        private Nw4rPlayerCollection _players = new Nw4rPlayerCollection();
        private Nw4rGroupCollection _groups = new Nw4rGroupCollection();

        #endregion

        public Nw4rSoundSet(XmlData.Nw4rSoundSet xmlSoundSet)
        {
            Debug.Assert(null != xmlSoundSet);
            _xmlData = xmlSoundSet;
        }

        #region ** プロパティ

        public XmlData.Nw4rSoundSet XmlData
        {
            get { return _xmlData; }
        }

        public override string Label
        {
            get
            {
                if (null == XmlData) { return string.Empty; }
                return XmlData.Name;
            }
        }

        public override string Key
        {
            get { return Label; }
        }

        #endregion

        #region ** コンポーネント コレクション

        public INw4rComponentCollection<Nw4rSound> Sounds
        {
            get { return _sounds; }
        }

        public INw4rComponentCollection<Nw4rSound> StreamSounds
        {
            get { return _streamSounds; }
        }

        public INw4rComponentCollection<Nw4rSound> WaveSounds
        {
            get { return _waveSounds; }
        }

        public INw4rComponentCollection<Nw4rSoundSetItem> WaveSoundSets
        {
            get { return _waveSoundSets; }
        }

        public INw4rComponentCollection<Nw4rSound> SequenceSounds
        {
            get { return _sequenceSounds; }
        }

        public INw4rComponentCollection<Nw4rBank> Banks
        {
            get { return _banks; }
        }

        public INw4rComponentCollection<Nw4rPlayer> Players
        {
            get { return _players; }
        }

        public INw4rComponentCollection<Nw4rGroup> Groups
        {
            get { return _groups; }
        }

        #endregion

        #region ** メソッド

        public void AddItem(Nw4rStreamSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _streamSounds.Add(sound);
        }

        public void AddItem(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { throw new ArgumentNullException("waveSoundSet"); }

            AddItem(waveSoundSet as Nw4rSoundArchiveItem);
            _waveSoundSets.Add(waveSoundSet);
        }

        public void AddItem(Nw4rWaveSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _waveSounds.Add(sound);
        }

        public void AddItem(Nw4rSequenceSoundSet sequenceSoundSet)
        {
            if (null == sequenceSoundSet) { throw new ArgumentNullException("sequenceSoundSet"); }

            AddItem(sequenceSoundSet as Nw4rSoundArchiveItem);
        }

        public void AddItem(Nw4rSequenceSound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            AddItem(sound as Nw4rSound);
            _sequenceSounds.Add(sound);
        }

        public void AddItem(Nw4rBank bank)
        {
            if (null == bank) { throw new ArgumentNullException("bank"); }

            AddItem(bank as Nw4rSoundArchiveItem);
            _banks.Add(bank);
        }

        public void AddItem(Nw4rPlayer player)
        {
            if (null == player) { throw new ArgumentNullException("player"); }

            AddItem(player as Nw4rSoundArchiveItem);
            _players.Add(player);
        }

        public void AddItem(Nw4rGroup group)
        {
            if (null == group) { throw new ArgumentNullException("group"); }

            AddItem(group as Nw4rSoundArchiveItem);
            _groups.Add(group);
        }

        private void AddItem(Nw4rSound sound)
        {
            Debug.Assert(null != sound);
            AddItem(sound as Nw4rSoundArchiveItem);
            _sounds.Add(sound);
        }

        private void AddItem(Nw4rSoundArchiveItem item)
        {
            Debug.Assert(null != item);
            // 今は未実装
        }

        #endregion
    }

    #endregion

    #region ** サウンド

    public class Nw4rStreamSound : Nw4rSound
    {
        #region ** フィールド

        private Nw4rSoundIntermediateFile _xmlIntermediateFile = null;

        private int _allocTracks = 0;
        private int _channels = 0;

        #endregion

        public Nw4rStreamSound(Nw4rSoundSetStrm soundSetItem) : base(soundSetItem) { }

        #region ** プロパティ

        public new Nw4rSoundSetStrm XmlData
        {
            get { return base.XmlData as Nw4rSoundSetStrm; }
        }

        public new Nw4rStreamSoundBinaryFile BinaryFile
        {
            get { return base.BinaryFile as Nw4rStreamSoundBinaryFile; }
            set { base.BinaryFile = value; }
        }

        public Nw4rSoundIntermediateFile XmlIntermediateFile
        {
            get { return _xmlIntermediateFile; }
            set { _xmlIntermediateFile = value; }
        }

        public int AllocTracks
        {
            get { return _allocTracks; }
        }

        public int Channels
        {
            get { return _channels; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareLink()
        {
            if (null == BinaryFile) { throw new Nw4rFileFormatInternalException(); }
            (BinaryFile as Nw4rStreamSoundBinaryFile).GetBinaryInfo(this, out _allocTracks, out _channels);
        }

        #endregion
    }

    public class Nw4rWaveSoundSet : Nw4rSoundSetItem
    {
        public Nw4rWaveSoundSet(Nw4rSoundSetWaveSoundSet xmlData) : base(xmlData) { }

        #region ** プロパティ

        public new Nw4rSoundSetWaveSoundSet XmlData
        {
            get { return base.XmlData as Nw4rSoundSetWaveSoundSet; }
        }

        public new Nw4rWaveSoundSetBinaryFile BinaryFile
        {
            get { return base.BinaryFile as Nw4rWaveSoundSetBinaryFile; }
            set { base.BinaryFile = value; }
        }

        #endregion

        #region ** メソッド

        public void Add(Nw4rWaveSound waveSound)
        {
            base.Add(waveSound);
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareConvert()
        {
            base.OnPrepareConvert();

            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException();
            }

            foreach (Nw4rWaveSound waveSound in Components)
            {
                if (BinaryFile.Components.Contains(waveSound.BinaryFile.Key)) { continue; }
                BinaryFile.Components.Add(waveSound.BinaryFile);
            }
        }

        #endregion
    }

    public class Nw4rWaveSound : Nw4rSound
    {
        public Nw4rWaveSound(Nw4rSoundSetWaveSound xmlData) : base(xmlData) { }

        #region ** プロパティ

        public new Nw4rSoundSetWaveSound XmlData
        {
            get { return base.XmlData as Nw4rSoundSetWaveSound; }
        }

        public new Nw4rWaveSoundSet Parent
        {
            get { return base.Parent as Nw4rWaveSoundSet; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareConvert()
        {
            // WSD 単体ではバイナリが存在しないため、base.OnPrepareConvert を実施しない
        }

        protected override void OnPrepareLink()
        {
            // WSD 単体ではバイナリが存在しないため、base.OnPrepareLink を実施しない
        }

        #endregion
    }

    public class Nw4rSequenceSoundSet : Nw4rSoundSetItem, INw4rSoundContainer
    {
        public Nw4rSequenceSoundSet(Nw4rSoundSetSeqSoundSet xmlData) : base(xmlData) { }

        #region ** プロパティ

        public new Nw4rSoundSetSeqSoundSet XmlData
        {
            get { return base.XmlData as Nw4rSoundSetSeqSoundSet; }
        }

        INw4rComponentCollection<Nw4rSound> INw4rSoundContainer.Sounds
        {
            get { return new Nw4rSoundCollection(Components); }
        }

        #endregion

        #region ** メソッド

        public void Add(Nw4rSequenceSound seqenceSound)
        {
            base.Add(seqenceSound);
        }

        #endregion
    }

    public class Nw4rSequenceSound : Nw4rSound
    {
        #region ** フィールド

        private Nw4rSoundIntermediateFile _xmlIntermediateFile = null;

        private Nw4rBank _bank = null;
        private long _startOffset = 0;
        private int _allocTracks = 0;

        #endregion

        public Nw4rSequenceSound(Nw4rSoundSetSeq xmlData) : base(xmlData) { }

        #region ** プロパティ

        public new Nw4rSoundSetSeq XmlData
        {
            get { return base.XmlData as Nw4rSoundSetSeq; }
        }

        public new Nw4rSequenceSoundBinaryFile BinaryFile
        {
            get { return base.BinaryFile as Nw4rSequenceSoundBinaryFile; }
            set { base.BinaryFile = value; }
        }

        public Nw4rSoundIntermediateFile XmlIntermediateFile
        {
            get { return _xmlIntermediateFile; }
            set { _xmlIntermediateFile = value; }
        }

        public Nw4rBank Bank
        {
            get { return _bank; }
        }

        public long StartOffset
        {
            get { return _startOffset; }
        }

        public int AllocTracks
        {
            get { return _allocTracks; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnResolveName(Nw4rFileFormatExceptionList results)
        {
            base.OnResolveName(results);
            Debug.Assert(null != Root);

            // バンクラベルの存在確認
            if (!Root.Symbols.Contains(XmlData.Bank))
            {
                results.Add(new Nw4rBankLabelNotFoundException(XmlData.Bank, XmlData.Label));
                return;
            }

            // バンクコンポーネントの存在確認
            if (!Root.AllBanks.Contains(XmlData.Bank))
            {
                results.Add(new Nw4rBankLabelNotFoundException(XmlData.Bank, XmlData.Label));
                return;
            }

            _bank = Root.AllBanks[XmlData.Bank];

            // 参照リストに追加する
            if (!Root.ReferencedBanks.Contains(_bank.Key))
            {
                (Root.ReferencedBanks as Nw4rBankCollection).Add(_bank);
            }
        }

        protected override void OnPrepareLink()
        {
            if (null == BinaryFile) { throw new Nw4rFileFormatInternalException(); }
            (BinaryFile as Nw4rSequenceSoundBinaryFile).GetBinaryInfo(this, out _startOffset, out _allocTracks);
        }

        #endregion
    }

    #endregion

    #region ** バンク

    public class Nw4rBank : Nw4rSoundSetItem
    {
        #region ** フィールド

        private Nw4rXmlBank _xmlBank = new Nw4rXmlBank();	// XMLデータ
        private InstrumentDictionary _instrumentMap = null;					// インストルメントマップ

        #endregion

        public Nw4rBank(Nw4rSoundSetBank soundSetItem) : base(soundSetItem) { }

        #region ** プロパティ

        /// <summary>
        /// XMLデータを取得します。
        /// </summary>
        public new Nw4rSoundSetBank XmlData
        {
            get { return base.XmlData as Nw4rSoundSetBank; }
        }

        /// <summary>
        /// XMLBankファイルデータを取得します。
        /// </summary>
        public Nw4rXmlBank XmlBankFileData
        {
            get { return _xmlBank; }
        }

        /// <summary>
        /// バンクのバイナリファイルを取得または設定します。
        /// </summary>
        public new Nw4rBankBinaryFile BinaryFile
        {
            get { return base.BinaryFile as Nw4rBankBinaryFile; }
            set { base.BinaryFile = value; }
        }

        /// <summary>
        /// インストルメントマップを取得します。
        /// </summary>
        public InstrumentDictionary InstrumentMap
        {
            get { return _instrumentMap; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareConvert()
        {
            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException();
            }

            // 依存ファイルコレクションを構築する
            AggregateDependFiles();

            // インストルメント以下のコンバート準備
            base.OnPrepareConvert();
        }

        protected override void OnPrepareLink()
        {
            base.OnPrepareLink();

            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException();
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// バンクファイルを読み込みます。
        /// </summary>
        public void Load()
        {
            Debug.Assert(null != XmlData);

            if (!File.Exists(XmlData.FullPath)) { throw new Nw4rFileNotFoundException(XmlData.FullPath, Label); }

            try
            {
                _xmlBank.Load(XmlData.FullPath);
            }
            catch (System.Xml.XmlException)
            {
                throw new Nw4rInvalidFileException(XmlData.FullPath, Label);
            }

            Build();
        }

        /// <summary>
        /// バンクを構築します。
        /// </summary>
        private void Build()
        {
            Debug.Assert(null != _xmlBank);

            int programCount = 0;

            // インストルメントの作成
            foreach (Nw4rXmlBankInst xmlInstrument in _xmlBank.InstList)
            {

                if (!xmlInstrument.Enabled) { continue; }

                Add(new Nw4rInstrument(xmlInstrument));

                if (programCount <= xmlInstrument.PrgNo)
                {
                    programCount = xmlInstrument.PrgNo + 1;
                }

            }

            CreateInstrumentMap(programCount);
        }

        /// <summary>
        /// インストルメントマップを作成します。
        /// </summary>
        /// <param name="programCount">プログラム数</param>
        private void CreateInstrumentMap(int programCount)
        {
            Debug.Assert(0 <= programCount);

            _instrumentMap = new InstrumentDictionary(programCount);

            foreach (Nw4rInstrument instrument in Components)
            {

                if (null != _instrumentMap[instrument.XmlData.PrgNo])
                {
                    throw new Nw4rFileFormatInternalException(string.Format("program no {0} is already existed",
                                                                            instrument.XmlData.PrgNo));
                }

                _instrumentMap[instrument.XmlData.PrgNo] = instrument;

            }
        }

        /// <summary>
        /// 依存ファイルコレクションを構築します。
        /// </summary>
        private void AggregateDependFiles()
        {
            Debug.Assert(null != BinaryFile);

            foreach (Nw4rInstrument instrument in Components)
            {
                foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
                {
                    foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
                    {

                        if (BinaryFile.Components.Contains(velocityRegion.BinaryFile.Key)) { continue; }
                        BinaryFile.Components.Add(velocityRegion.BinaryFile);

                    }
                }
            }
        }

        #endregion

        #region ** インストルメントマップ

        /// <summary>
        /// インストルメントマップクラス
        /// </summary>
        public class InstrumentDictionary : Dictionary<int, Nw4rInstrument>
        {
            public InstrumentDictionary(int count) : base(count)
            {
                for (int i = 0; i < count; i++)
                {
                    Add(i, null);
                }
            }
        }

        #endregion
    }

    public class Nw4rInstrument : Nw4rBankItem
    {
        public const int KeyMin = 0;
        public const int KeyMax = 127;

        private Nw4rXmlBankInst _xmlInst = null;	// XMLデータ

        public Nw4rInstrument(Nw4rXmlBankInst xmlData)
        {
            if (null == xmlData) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("xmlData")); }
            _xmlInst = xmlData;

            Build();
        }

        #region ** プロパティ

        public Nw4rXmlBankInst XmlData
        {
            get { return _xmlInst; }
        }

        public override string Label
        {
            get { return _xmlInst.Label; }
        }

        public override string Key
        {
            get { return Bank.Label + "." + Label; }
        }

        #endregion

        #region ** メソッド

        public void ResolveName(Nw4rBank bank)
        {
            // コンポーネントの存在確認
            if (!bank.Components.Contains(Label))
            {
                throw new Nw4rLabelNotFoundException(_xmlInst.Label, Bank.Label);
            }
        }

        /// <summary>
        /// インストルメントを構築します。
        /// </summary>
        private void Build()
        {
            Debug.Assert(null != _xmlInst);

            // キー順にソートする
            SortedList<int, Nw4rXmlBankInstKeyRegion> sortedXmlKeyRegions = new SortedList<int, Nw4rXmlBankInstKeyRegion>();

            foreach (Nw4rXmlBankInstKeyRegion xmlKeyRegion in _xmlInst)
            {
                sortedXmlKeyRegions.Add(xmlKeyRegion.RangeMin, xmlKeyRegion);
            }

            // キーリージョンの作成
            foreach (Nw4rXmlBankInstKeyRegion xmlKeyRegion in sortedXmlKeyRegions.Values)
            {
                Add(new Nw4rInstrumentKeyRegion(xmlKeyRegion));
            }
        }

        #endregion
    }

    public class Nw4rInstrumentKeyRegion : Nw4rInstrumentItem
    {
        public const int VelocityMax = 128;

        private Nw4rXmlBankInstKeyRegion _xmlKeyRegion = null;

        public Nw4rInstrumentKeyRegion(Nw4rXmlBankInstKeyRegion xmlData)
        {
            if (null == xmlData) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("xmlData")); }
            _xmlKeyRegion = xmlData;

            CreateComponents();
        }

        #region ** プロパティ

        public Nw4rXmlBankInstKeyRegion XmlData
        {
            get { return _xmlKeyRegion; }
        }

        public override string Label
        {
            get { return _xmlKeyRegion.RangeMin.ToString(); }
        }

        public override string Key
        {
            get { return Label; }
        }

        #endregion

        #region ** メソッド

        private void CreateComponents()
        {
            Debug.Assert(null != _xmlKeyRegion);

            // キー順にソートする
            SortedList<int, Nw4rXmlBankInstVelRegion> sortedXmlVelocityRegions = new SortedList<int, Nw4rXmlBankInstVelRegion>();

            foreach (Nw4rXmlBankInstVelRegion xmlVelocityRegion in _xmlKeyRegion)
            {
                sortedXmlVelocityRegions.Add(xmlVelocityRegion.RangeMin, xmlVelocityRegion);
            }

            // キーリージョンの作成
            foreach (Nw4rXmlBankInstVelRegion xmlVelocityRegion in sortedXmlVelocityRegions.Values)
            {
                Add(new Nw4rInstrumentVelocityRegion(xmlVelocityRegion));
            }
        }

        #endregion
    }

    public class Nw4rInstrumentVelocityRegion : Nw4rInstrumentItem
    {
        private Nw4rXmlBankInstVelRegion _xmlVelRegion = null;
        private Nw4rSoundBinaryFile _binaryFile = null;

        public Nw4rInstrumentVelocityRegion(Nw4rXmlBankInstVelRegion xmlData)
        {
            if (null == xmlData) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("xmlData")); }
            _xmlVelRegion = xmlData;
        }

        #region ** プロパティ

        public Nw4rXmlBankInstVelRegion XmlData
        {
            get { return _xmlVelRegion; }
        }

        public override string Label
        {
            get { return _xmlVelRegion.RangeMin.ToString(); }
        }

        public override string Key
        {
            get { return Label; }
        }

        public Nw4rSoundBinaryFile BinaryFile
        {
            get { return _binaryFile; }
            set { _binaryFile = value; }
        }

        public string FilePath
        {
            get { return Path.Combine(Path.GetDirectoryName(Bank.XmlData.FullPath), _xmlVelRegion.FilePath).Replace('/', '\\'); }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPrepareConvert()
        {
            base.OnPrepareConvert();

            if (null == BinaryFile)
            {
                throw new Nw4rFileFormatInternalException("bank wave binary file not found");
            }

            if (!Bank.BinaryFile.Components.Contains(BinaryFile.Key))
            {
                throw new Nw4rFileFormatInternalException(
                            string.Format("bank wave binary file not registered \"{0}\"", BinaryFile.Key));
            }
        }

        #endregion
    }

    #endregion

    #region ** プレイヤー

    public class Nw4rPlayer : Nw4rSoundSetItem
    {
        public Nw4rPlayer(Nw4rSoundSetPlayer xmlData) : base(xmlData) { }

        #region ** プロパティ

        public new Nw4rSoundSetPlayer XmlData
        {
            get { return base.XmlData as Nw4rSoundSetPlayer; }
        }

        #endregion
    }

    #endregion

    #region ** グループ

    public class Nw4rGroup : Nw4rSoundSetItem
    {
        private Nw4rGroupItemCollection _items = new Nw4rGroupItemCollection();

        private Nw4rSoundSetItemCollection _allDependItems = new Nw4rSoundSetItemCollection();
        private Nw4rBinaryFileCollection _files = new Nw4rBinaryFileCollection();

        public Nw4rGroup(Nw4rSoundSetGroup xmlData) : base(xmlData)
        {
            CreateGroupItems();
        }

        #region ** プロパティ

        public new Nw4rSoundSetGroup XmlData
        {
            get { return base.XmlData as Nw4rSoundSetGroup; }
        }

        public INw4rComponentCollection<Nw4rGroupItem> Items
        {
            get { return _items; }
        }

        public INw4rComponentCollection<Nw4rSoundSetItem> AllDependItems
        {
            get { return _allDependItems; }
        }

        public INw4rComponentCollection<Nw4rSoundBinaryFile> Files
        {
            get { return _files; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnResolveName(Nw4rFileFormatExceptionList results)
        {
            base.OnResolveName(results);

            ResolveNames(results);

            AggregateDependItems(this, new LinkedList<Nw4rGroup>());
        }

        protected override void OnPrepareConvert()
        {
            base.OnPrepareConvert();

            LinkItems();
        }

        #endregion

        #region ** メソッド

        public void AddItem(Nw4rGroupItem item)
        {
            if (null == item) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("item")); }

            if (Items.Contains(item.Key)) { return; }
            _items.Add(item);
        }

        public void AddFile(Nw4rSoundBinaryFile file)
        {
            if (null == file) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("file")); }

            file.Groups.Add(this);
            _files.Add(file);
        }

        private void CreateGroupItems()
        {
            // グループアイテムの生成
            foreach (Nw4rSoundSetGroupItem xmlGroupItem in XmlData)
            {
                AddItem(new Nw4rGroupItem(this, xmlGroupItem));
            }
        }

        private void ResolveNames(Nw4rFileFormatExceptionList results)
        {
            foreach (Nw4rGroupItem item in _items)
            {
                if (Root.DisabledItems.Contains(item.Label)) { continue; }
                item.ResolveName(Root, results);
            }
        }

        /// <summary>
        /// 依存アイテムコンテナを構築します。
        /// </summary>
        /// <param name="group"></param>
        /// <param name="stack"></param>
        private void AggregateDependItems(Nw4rGroup group, LinkedList<Nw4rGroup> stack)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != stack);

            stack.AddFirst(this);

            foreach (Nw4rGroupItem item in _items)
            {

                if (Root.DisabledItems.Contains(item.Label)) { continue; }

                if (item.LinkedItem is Nw4rGroup)
                {

                    Nw4rGroup innerGroup = item.LinkedItem as Nw4rGroup;
                    if (stack.Contains(innerGroup)) { throw new Nw4rMultipleInnerGroupException(innerGroup.Label, group.Label); }

                    innerGroup.AggregateDependItems(group, stack);

                }
                else
                {
                    if (group._allDependItems.Contains(item.LinkedItem.Key)) { continue; }
                    group._allDependItems.Add(item.LinkedItem as Nw4rSoundSetItem);
                }
            }

            stack.RemoveFirst();
        }

        /// <summary>
        /// this.Items に含まれるグループアイテムを指定グループとリンクします。
        /// </summary>
        /// <param name="group"></param>
        /// <param name="stack"></param>
        private void LinkItems()
        {
            foreach (Nw4rSoundSetItem item in _allDependItems)
            {
                LinkItem(item);
            }

            if (XmlData.WaveArchiveShare)
            {

                if (null == BinaryFile)
                {
                    throw new Nw4rFileFormatInternalException("group binary file is null");
                }

                _files.Add(BinaryFile);

            }
        }

        /// <summary>
        /// 指定グループと指定アイテムをリンクします。
        /// </summary>
        /// <param name="group">リンク元グループ</param>
        /// <param name="linkedItem">リンク先アイテム</param>
        private void LinkItem(Nw4rSoundSetItem linkedItem)
        {
            if (null == linkedItem)
            {
                Debug.Assert(false);
                throw new Nw4rFileFormatInternalException();
            }

            // Group->File / GroupItem->File リンク
            // 複合サウンド（シーケンスサウンドセット）の場合、全ての関連ファイルとリンクする
            if (linkedItem is INw4rSoundContainer)
            {

                foreach (Nw4rSoundSetItem containerItem in (linkedItem as INw4rSoundContainer).Sounds)
                {

                    if (!XmlData.OutputFlag)
                    {
                        if (containerItem.BinaryFile.InnerGroups.Contains(Key)) { continue; }
                        containerItem.BinaryFile.InnerGroups.Add(this);
                    }
                    else
                    {
                        if (containerItem.BinaryFile.Groups.Contains(Key)) { continue; }
                        containerItem.BinaryFile.Groups.Add(this);
                    }


                    Nw4rSoundBinaryFile fileForThis = containerItem.GetBinaryFileForGroup(this);

                    if (null != fileForThis)
                    {
                        _files.Add(fileForThis);
                    }
                    else
                    {
                        _files.Add(containerItem.BinaryFile);
                    }

                }

            }
            // 単体サウンドの場合
            else if (null != linkedItem.BinaryFile)
            {

                if (!XmlData.OutputFlag)
                {
                    if (linkedItem.BinaryFile.InnerGroups.Contains(Key)) { return; }
                    linkedItem.BinaryFile.InnerGroups.Add(this);
                }
                else
                {
                    if (linkedItem.BinaryFile.Groups.Contains(Key)) { return; }
                    linkedItem.BinaryFile.Groups.Add(this);
                }


                Nw4rSoundBinaryFile fileForThis = linkedItem.GetBinaryFileForGroup(this);

                if (null != fileForThis)
                {
                    _files.Add(fileForThis);
                }
                else
                {
                    _files.Add(linkedItem.BinaryFile);
                }

            }
            else
            {
                throw new Nw4rFileFormatInternalException();
            }


            // 以下、波形データを共有する場合はグループバイナリファイルに追加する
            if (!XmlData.WaveArchiveShare) { return; }

            if (linkedItem.GetBinaryFileForGroup(this) != null)
            {

                foreach (Nw4rSoundBinaryFile binaryFile in linkedItem.BinaryFile.Components)
                {
                    if (BinaryFile.Components.Contains(binaryFile.Key)) { continue; }
                    BinaryFile.Components.Add(binaryFile);
                }

            }
        }

        #endregion
    }

    public class Nw4rGroupItem : Nw4rComponent
    {
        private Nw4rGroup _parent = null;
        private Nw4rSoundSetItem _linkedItem = null;

        Nw4rSoundSetGroupItem _xmlGroupItem = null;

        public Nw4rGroupItem(Nw4rGroup parent, Nw4rSoundSetGroupItem xmlData)
        {
            if (null == parent) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("parent")); }
            if (null == xmlData) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("xmlData")); }

            _parent = parent;
            _xmlGroupItem = xmlData;
        }

        #region ** プロパティ

        public new Nw4rGroup Parent
        {
            get { return _parent; }
        }

        public Nw4rSoundSetItem LinkedItem
        {
            get { return _linkedItem; }
        }

        public string Label
        {
            get { return _xmlGroupItem.Label; }
        }

        public override string Key
        {
            get { return Parent.Label + "." + Label; }
        }

        #endregion

        #region ** メソッド

        public void ResolveName(Nw4rSoundArchive archive, Nw4rFileFormatExceptionList results)
        {
            if (null != _linkedItem) { return; }

            // コンポーネントの存在確認
            if (!archive.AllItems.Contains(_xmlGroupItem.Label))
            {
                throw new Nw4rLabelNotFoundException(_xmlGroupItem.Label, Parent.Label);
            }

            _linkedItem = archive.AllItems[_xmlGroupItem.Label] as Nw4rSoundSetItem;

            if (null == _linkedItem)
            {
                throw new Nw4rLabelNotFoundException(_xmlGroupItem.Label, Parent.Label);
            }

            if (_linkedItem is Nw4rGroup)
            {

                _linkedItem.ResolveName(results);

                // 参照リストに追加する
                if (!Parent.Root.ReferencedInnerGroups.Contains(_linkedItem.Key))
                {
                    (Parent.Root.ReferencedInnerGroups as Nw4rGroupCollection).Add(_linkedItem as Nw4rGroup);
                }

            }
        }

        #endregion
    }

    #endregion

    #endregion

    #region ** サウンドアーカイブシンボル

    /// <summary>
    /// サウンドアーカイブシンボル
    /// </summary>
    public class Nw4rSymbol : Nw4rComponent
    {
        private Nw4rSoundSetItem _item = null;

        public Nw4rSymbol(Model.Nw4rSoundSetItem item)
        {
            if (null == item) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("item")); }
            _item = item;
        }

        #region ** プロパティ

        public override string Key
        {
            get { return Item.Key; }
        }

        public Nw4rSoundSetItem Item
        {
            get { return _item; }
        }

        #endregion
    }

    #endregion
}
