﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model
{
    #region ** コレクション 基本クラス

    #region ** コレクション インターフェイス

    public interface INw4rCollection : IEnumerable
    {
        #region ** プロパティ

        /// <summary>
        /// コレクションに格納されているアイテム数を取得します。
        /// </summary>
        int Count { get; }

        #endregion
    }

    public interface INw4rCollection<_ItemType> : INw4rCollection, IEnumerable<_ItemType>
    {
        #region ** インデクサ

        /// <summary>
        /// 指定したインデックス位置にあるアイテムを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のアイテム。インデックスがノード数以上の場合、null を返します。</returns>
        _ItemType this[int index] { get; }

        #endregion
    }

    #endregion

    #region ** インデックスコレクション

    /// <summary>
    /// インデックス付きコレクション
    /// </summary>
    /// <typeparam name="_KeyType">キーの型</typeparam>
    /// <typeparam name="_ItemType">アイテムの型</typeparam>
    public abstract class Nw4rIndexedCollection<_KeyType, _ItemType> : INw4rCollection<_ItemType>
    {
        private const string PropertyName_Index = "Index";

        private InnerCollection _collection = null;

        public Nw4rIndexedCollection()
        {
            _collection = new InnerCollection(this);
        }

        #region ** プロパティ

        /// <summary>
        /// コレクションに格納されているアイテム数を取得します。
        /// </summary>
        public int Count
        {
            get { return _collection.Count; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 指定キーをもつアイテムがコレクション内に含まれているかどうかを調べます。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクションに含まれている場合はtrue、含まれていない場合はfalse。</returns>
        public bool Contains(_KeyType key)
        {
            if (null == key) { throw new ArgumentNullException("key"); }
            return _collection.Contains(key);
        }

        /// <summary>
        /// 指定アイテムのインデックスを調べます。
        /// </summary>
        /// <param name="item">アイテム</param>
        /// <returns>指定アイテムのインデックス。</returns>
        public int IndexOf(_ItemType item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return (int)_collection.GetAttachedProperty(new AttachedPropertyID(item, PropertyName_Index));
        }

        /// <summary>
        /// 指定キーを持つアイテムのインデックスを調べます。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>指定アイテムのインデックス。</returns>
        public int IndexOf(_KeyType key)
        {
            if (null == key) { throw new ArgumentNullException("key"); }
            return IndexOf(this[key]);
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>列挙子</returns>
        public IEnumerator<_ItemType> GetEnumerator()
        {
            return _collection.GetEnumerator();
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>列挙子</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return _collection.GetEnumerator();
        }

        /// <summary>
        /// アイテムを追加します。
        /// </summary>
        /// <param name="item">新しいアイテム</param>
        public void Add(_ItemType item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            _collection.Add(item);
        }

        public void Clear()
        {
            _collection.Clear();
        }

        protected abstract _KeyType GetKeyForItem(_ItemType item);

        #endregion

        #region ** インデクサ

        /// <summary>
        /// 指定したインデックス位置にあるアイテムを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のアイテム。インデックスがノード数以上の場合、null を返します。</returns>
        public _ItemType this[int index]
        {
            get { return _collection[index]; }
        }

        /// <summary>
        /// 指定したキーをもつ最初のアイテムを取得します。
        /// </summary>
        /// <param name="key">キー</param>
        /// <returns>コレクション内のアイテム。該当ノードが存在しない場合、null を返します。</returns>
        public _ItemType this[_KeyType key]
        {
            get { return _collection[key]; }
        }

        #endregion

        #region ** 内部コレクションクラス

        protected class InnerCollection : KeyedCollection<_KeyType, _ItemType>
        {
            #region ** フィールド

            private Nw4rIndexedCollection<_KeyType, _ItemType> _owner = null;

            private AttachedPropertyDictionary _attachedProperties = new AttachedPropertyDictionary();

            #endregion

            public InnerCollection(Nw4rIndexedCollection<_KeyType, _ItemType> owner)
            {
                Debug.Assert(null != owner);
                _owner = owner;
            }

            #region ** メソッド

            public object GetAttachedProperty(AttachedPropertyID id)
            {
                return _attachedProperties[id];
            }

            public void SetAttachedProperty(AttachedPropertyID id, object value)
            {
                Debug.Assert(null != value);
                _attachedProperties.Add(id, value);
            }

            protected override _KeyType GetKeyForItem(_ItemType item)
            {
                return _owner.GetKeyForItem(item);
            }

            protected override void InsertItem(int index, _ItemType item)
            {
                SetAttachedProperty(new AttachedPropertyID(item, PropertyName_Index), index);
                base.InsertItem(index, item);
            }

            protected override void ClearItems()
            {
                base.ClearItems();
                _attachedProperties.Clear();
            }

            #endregion
        }

        #endregion
    }

    #endregion

    #region ** ディクショナリ

    /// <summary>
    /// 読み取り専用ディクショナリ
    /// </summary>
    public interface INw4rReadOnlyDictionary<_KeyType, _ItemType>
    {
        /// <summary>
        /// キーコレクションを取得します。
        /// </summary>
        Dictionary<_KeyType, _ItemType>.KeyCollection Keys { get; }

        /// <summary>
        /// アイテムコレクションを取得します。
        /// </summary>
        Dictionary<_KeyType, _ItemType>.ValueCollection Values { get; }

        /// <summary>
        /// 指定したキーを持つアイテムを取得します。
        /// </summary>
        /// <param name="key">取得するアイテムのキー。</param>
        /// <returns>指定したキーを持つアイテム。</returns>
        _ItemType this[_KeyType key] { get; }

        /// <summary>
        /// 指定したキーのアイテムが含まれているかどうかを確認します。
        /// </summary>
        /// <param name="key">検索されるキー。</param>
        /// <returns>指定したキーを持つアイテムを含む場合は true。それ以外の場合は false。</returns>
        bool ContainsKey(_KeyType key);
    }

    /// <summary>
    /// ディクショナリ クラス
    /// </summary>
    public class Nw4rDictionary<_KeyType, _ItemType>
        : Dictionary<_KeyType, _ItemType>, INw4rReadOnlyDictionary<_KeyType, _ItemType>
    { }

    #endregion

    #endregion
}
